/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.compress;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Longs;
import java.io.BufferedOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeSet;
import org.apache.cassandra.cache.RefCountedMemory;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.compress.CompressionParameters;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableWriter;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.Memory;
import org.apache.cassandra.utils.Pair;

public class CompressionMetadata {
    public final long dataLength;
    public final long compressedFileLength;
    public final boolean hasPostCompressionAdlerChecksums;
    private final Memory chunkOffsets;
    private final long chunkOffsetsSize;
    public final String indexFilePath;
    public final CompressionParameters parameters;

    public static CompressionMetadata create(String dataFilePath) {
        Descriptor desc = Descriptor.fromFilename(dataFilePath);
        return new CompressionMetadata(desc.filenameFor(Component.COMPRESSION_INFO), new File(dataFilePath).length(), desc.version.hasPostCompressionAdlerChecksums);
    }

    @VisibleForTesting
    CompressionMetadata(String indexFilePath, long compressedLength, boolean hasPostCompressionAdlerChecksums) {
        DataInputStream stream;
        this.indexFilePath = indexFilePath;
        this.hasPostCompressionAdlerChecksums = hasPostCompressionAdlerChecksums;
        try {
            stream = new DataInputStream(new FileInputStream(indexFilePath));
        }
        catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
        try {
            String compressorName = stream.readUTF();
            int optionCount = stream.readInt();
            HashMap<String, String> options = new HashMap<String, String>();
            for (int i = 0; i < optionCount; ++i) {
                String key = stream.readUTF();
                String value = stream.readUTF();
                options.put(key, value);
            }
            int chunkLength = stream.readInt();
            try {
                this.parameters = new CompressionParameters(compressorName, (Integer)chunkLength, options);
            }
            catch (ConfigurationException e) {
                throw new RuntimeException("Cannot create CompressionParameters for stored parameters", e);
            }
            this.dataLength = stream.readLong();
            this.compressedFileLength = compressedLength;
            this.chunkOffsets = this.readChunkOffsets(stream);
        }
        catch (IOException e) {
            throw new CorruptSSTableException((Exception)e, indexFilePath);
        }
        finally {
            FileUtils.closeQuietly(stream);
        }
        this.chunkOffsetsSize = this.chunkOffsets.size();
    }

    private CompressionMetadata(String filePath, CompressionParameters parameters, RefCountedMemory offsets, long offsetsSize, long dataLength, long compressedLength, boolean hasPostCompressionAdlerChecksums) {
        this.indexFilePath = filePath;
        this.parameters = parameters;
        this.dataLength = dataLength;
        this.compressedFileLength = compressedLength;
        this.hasPostCompressionAdlerChecksums = hasPostCompressionAdlerChecksums;
        this.chunkOffsets = offsets;
        offsets.reference();
        this.chunkOffsetsSize = offsetsSize;
    }

    public ICompressor compressor() {
        return this.parameters.sstableCompressor;
    }

    public int chunkLength() {
        return this.parameters.chunkLength();
    }

    public long offHeapSize() {
        return this.chunkOffsets.size();
    }

    private Memory readChunkOffsets(DataInput input) {
        try {
            int chunkCount = input.readInt();
            if (chunkCount <= 0) {
                throw new IOException("Compressed file with 0 chunks encountered: " + input);
            }
            Memory offsets = Memory.allocate(chunkCount * 8);
            for (int i = 0; i < chunkCount; ++i) {
                try {
                    offsets.setLong(i * 8, input.readLong());
                    continue;
                }
                catch (EOFException e) {
                    String msg = String.format("Corrupted Index File %s: read %d but expected %d chunks.", this.indexFilePath, i, chunkCount);
                    throw new CorruptSSTableException((Exception)new IOException(msg, e), this.indexFilePath);
                }
            }
            return offsets;
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, this.indexFilePath);
        }
    }

    public Chunk chunkFor(long position) {
        int idx = 8 * (int)(position / (long)this.parameters.chunkLength());
        if ((long)idx >= this.chunkOffsetsSize) {
            throw new CorruptSSTableException((Exception)new EOFException(), this.indexFilePath);
        }
        long chunkOffset = this.chunkOffsets.getLong(idx);
        long nextChunkOffset = (long)(idx + 8) == this.chunkOffsetsSize ? this.compressedFileLength : this.chunkOffsets.getLong(idx + 8);
        return new Chunk(chunkOffset, (int)(nextChunkOffset - chunkOffset - 4L));
    }

    public Chunk[] getChunksForSections(Collection<Pair<Long, Long>> sections) {
        TreeSet<Chunk> offsets = new TreeSet<Chunk>(new Comparator<Chunk>(){

            @Override
            public int compare(Chunk o1, Chunk o2) {
                return Longs.compare((long)o1.offset, (long)o2.offset);
            }
        });
        for (Pair<Long, Long> section : sections) {
            int startIndex = (int)((Long)section.left / (long)this.parameters.chunkLength());
            int endIndex = (int)((Long)section.right / (long)this.parameters.chunkLength());
            endIndex = (Long)section.right % (long)this.parameters.chunkLength() == 0L ? endIndex - 1 : endIndex;
            for (int i = startIndex; i <= endIndex; ++i) {
                long offset = i * 8;
                long chunkOffset = this.chunkOffsets.getLong(offset);
                long nextChunkOffset = offset + 8L == this.chunkOffsetsSize ? this.compressedFileLength : this.chunkOffsets.getLong(offset + 8L);
                offsets.add(new Chunk(chunkOffset, (int)(nextChunkOffset - chunkOffset - 4L)));
            }
        }
        return offsets.toArray(new Chunk[offsets.size()]);
    }

    public void close() {
        if (this.chunkOffsets instanceof RefCountedMemory) {
            ((RefCountedMemory)this.chunkOffsets).unreference();
        } else {
            this.chunkOffsets.free();
        }
    }

    static class ChunkSerializer
    implements IVersionedSerializer<Chunk> {
        ChunkSerializer() {
        }

        @Override
        public void serialize(Chunk chunk, DataOutputPlus out, int version) throws IOException {
            out.writeLong(chunk.offset);
            out.writeInt(chunk.length);
        }

        @Override
        public Chunk deserialize(DataInput in, int version) throws IOException {
            return new Chunk(in.readLong(), in.readInt());
        }

        @Override
        public long serializedSize(Chunk chunk, int version) {
            long size = TypeSizes.NATIVE.sizeof(chunk.offset);
            return size += (long)TypeSizes.NATIVE.sizeof(chunk.length);
        }
    }

    public static class Chunk {
        public static final IVersionedSerializer<Chunk> serializer = new ChunkSerializer();
        public final long offset;
        public final int length;

        public Chunk(long offset, int length) {
            assert (length > 0);
            this.offset = offset;
            this.length = length;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Chunk chunk = (Chunk)o;
            return this.length == chunk.length && this.offset == chunk.offset;
        }

        public int hashCode() {
            int result = (int)(this.offset ^ this.offset >>> 32);
            result = 31 * result + this.length;
            return result;
        }

        public String toString() {
            return String.format("Chunk<offset: %d, length: %d>", this.offset, this.length);
        }
    }

    public static class Writer {
        private final CompressionParameters parameters;
        private final String filePath;
        private int maxCount = 100;
        private RefCountedMemory offsets = new RefCountedMemory(this.maxCount * 8);
        private int count = 0;

        private Writer(CompressionParameters parameters, String path) {
            this.parameters = parameters;
            this.filePath = path;
        }

        public static Writer open(CompressionParameters parameters, String path) {
            return new Writer(parameters, path);
        }

        public void addOffset(long offset) {
            if (this.count == this.maxCount) {
                RefCountedMemory newOffsets = this.offsets.copy((this.maxCount *= 2) * 8);
                this.offsets.unreference();
                this.offsets = newOffsets;
            }
            this.offsets.setLong(8 * this.count++, offset);
        }

        private void writeHeader(DataOutput out, long dataLength, int chunks) {
            try {
                out.writeUTF(this.parameters.sstableCompressor.getClass().getSimpleName());
                out.writeInt(this.parameters.otherOptions.size());
                for (Map.Entry<String, String> entry : this.parameters.otherOptions.entrySet()) {
                    out.writeUTF(entry.getKey());
                    out.writeUTF(entry.getValue());
                }
                out.writeInt(this.parameters.chunkLength());
                out.writeLong(dataLength);
                out.writeInt(chunks);
            }
            catch (IOException e) {
                throw new FSWriteError((Throwable)e, this.filePath);
            }
        }

        public CompressionMetadata open(long dataLength, long compressedLength, SSTableWriter.FinishType finishType) {
            RefCountedMemory offsets;
            switch (finishType) {
                case EARLY: {
                    offsets = this.offsets;
                    break;
                }
                case NORMAL: 
                case FINISH_EARLY: {
                    offsets = this.offsets.copy((long)this.count * 8L);
                    this.offsets.unreference();
                    break;
                }
                default: {
                    throw new AssertionError();
                }
            }
            return new CompressionMetadata(this.filePath, this.parameters, offsets, (long)this.count * 8L, dataLength, compressedLength, Descriptor.Version.CURRENT.hasPostCompressionAdlerChecksums);
        }

        public long chunkOffsetBy(int chunkIndex) {
            return this.offsets.getLong((long)chunkIndex * 8L);
        }

        public void resetAndTruncate(int chunkIndex) {
            this.count = chunkIndex;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void close(long dataLength, int chunks) throws IOException {
            DataOutputStream out = null;
            try {
                out = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(this.filePath)));
                assert (chunks == this.count);
                this.writeHeader(out, dataLength, chunks);
                for (int i = 0; i < this.count; ++i) {
                    out.writeLong(this.offsets.getLong(i * 8));
                }
            }
            catch (Throwable throwable) {
                FileUtils.closeQuietly(out);
                throw throwable;
            }
            FileUtils.closeQuietly(out);
        }

        public void abort() {
            if (this.offsets != null) {
                this.offsets.unreference();
                this.offsets = null;
            }
        }
    }
}

