/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DataTracker;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.io.sstable.IndexSummaryBuilder;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexSummaryRedistribution
extends CompactionInfo.Holder {
    private static final Logger logger = LoggerFactory.getLogger(IndexSummaryRedistribution.class);
    private final List<SSTableReader> compacting;
    private final List<SSTableReader> nonCompacting;
    private final long memoryPoolBytes;
    private volatile long remainingSpace;

    public IndexSummaryRedistribution(List<SSTableReader> compacting, List<SSTableReader> nonCompacting, long memoryPoolBytes) {
        this.compacting = compacting;
        this.nonCompacting = nonCompacting;
        this.memoryPoolBytes = memoryPoolBytes;
    }

    public List<SSTableReader> redistributeSummaries() throws IOException {
        long total = 0L;
        for (SSTableReader sstable : Iterables.concat(this.compacting, this.nonCompacting)) {
            total += sstable.getIndexSummaryOffHeapSize();
        }
        ArrayList<SSTableReader> oldFormatSSTables = new ArrayList<SSTableReader>();
        for (SSTableReader sstable : this.nonCompacting) {
            logger.trace("SSTable {} cannot be re-sampled due to old sstable format", (Object)sstable);
            if (sstable.descriptor.version.hasSamplingLevel) continue;
            oldFormatSSTables.add(sstable);
        }
        this.nonCompacting.removeAll(oldFormatSSTables);
        logger.debug("Beginning redistribution of index summaries for {} sstables with memory pool size {} MB; current spaced used is {} MB", new Object[]{this.nonCompacting.size(), this.memoryPoolBytes / 1024L / 1024L, (double)total / 1024.0 / 1024.0});
        HashMap<SSTableReader, Double> readRates = new HashMap<SSTableReader, Double>(this.nonCompacting.size());
        double totalReadsPerSec = 0.0;
        for (SSTableReader sstable : this.nonCompacting) {
            if (this.isStopRequested()) {
                throw new CompactionInterruptedException(this.getCompactionInfo());
            }
            if (sstable.getReadMeter() == null) continue;
            Double readRate = sstable.getReadMeter().fifteenMinuteRate();
            totalReadsPerSec += readRate.doubleValue();
            readRates.put(sstable, readRate);
        }
        logger.trace("Total reads/sec across all sstables in index summary resize process: {}", (Object)totalReadsPerSec);
        ArrayList<SSTableReader> sstablesByHotness = new ArrayList<SSTableReader>(this.nonCompacting);
        Collections.sort(sstablesByHotness, new ReadRateComparator(readRates));
        long remainingBytes = this.memoryPoolBytes;
        for (SSTableReader sstable : Iterables.concat(this.compacting, oldFormatSSTables)) {
            remainingBytes -= sstable.getIndexSummaryOffHeapSize();
        }
        logger.trace("Index summaries for compacting SSTables are using {} MB of space", (Object)((double)(this.memoryPoolBytes - remainingBytes) / 1024.0 / 1024.0));
        List<SSTableReader> newSSTables = this.adjustSamplingLevels(sstablesByHotness, totalReadsPerSec, remainingBytes);
        total = 0L;
        for (SSTableReader sstable : Iterables.concat(this.compacting, oldFormatSSTables, newSSTables)) {
            total += sstable.getIndexSummaryOffHeapSize();
        }
        logger.debug("Completed resizing of index summaries; current approximate memory used: {} MB", (Object)((double)total / 1024.0 / 1024.0));
        return newSSTables;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<SSTableReader> adjustSamplingLevels(List<SSTableReader> sstables, double totalReadsPerSec, long memoryPoolCapacity) throws IOException {
        List<ResampleEntry> toDownsample = new ArrayList<ResampleEntry>(sstables.size() / 4);
        ArrayList<ResampleEntry> toUpsample = new ArrayList<ResampleEntry>(sstables.size() / 4);
        ArrayList<ResampleEntry> forceResample = new ArrayList<ResampleEntry>();
        ArrayList<ResampleEntry> forceUpsample = new ArrayList<ResampleEntry>();
        ArrayList<SSTableReader> newSSTables = new ArrayList<SSTableReader>(sstables.size());
        this.remainingSpace = memoryPoolCapacity;
        for (SSTableReader sstable : sstables) {
            long spaceUsed;
            if (this.isStopRequested()) {
                throw new CompactionInterruptedException(this.getCompactionInfo());
            }
            int minIndexInterval = sstable.metadata.getMinIndexInterval();
            int maxIndexInterval = sstable.metadata.getMaxIndexInterval();
            double readsPerSec = sstable.getReadMeter() == null ? 0.0 : sstable.getReadMeter().fifteenMinuteRate();
            long idealSpace = Math.round((double)this.remainingSpace * (readsPerSec / totalReadsPerSec));
            int currentNumEntries = sstable.getIndexSummarySize();
            double avgEntrySize = (double)sstable.getIndexSummaryOffHeapSize() / (double)currentNumEntries;
            long targetNumEntries = Math.max(1L, Math.round((double)idealSpace / avgEntrySize));
            int currentSamplingLevel = sstable.getIndexSummarySamplingLevel();
            int maxSummarySize = sstable.getMaxIndexSummarySize();
            if (sstable.getMinIndexInterval() != minIndexInterval) {
                int effectiveSamplingLevel = (int)Math.round((double)currentSamplingLevel * ((double)minIndexInterval / (double)sstable.getMinIndexInterval()));
                maxSummarySize = (int)Math.round((double)maxSummarySize * ((double)sstable.getMinIndexInterval() / (double)minIndexInterval));
                logger.trace("min_index_interval changed from {} to {}, so the current sampling level for {} is effectively now {} (was {})", new Object[]{sstable.getMinIndexInterval(), minIndexInterval, sstable, effectiveSamplingLevel, currentSamplingLevel});
                currentSamplingLevel = effectiveSamplingLevel;
            }
            int newSamplingLevel = IndexSummaryBuilder.calculateSamplingLevel(currentSamplingLevel, currentNumEntries, targetNumEntries, minIndexInterval, maxIndexInterval);
            int numEntriesAtNewSamplingLevel = IndexSummaryBuilder.entriesAtSamplingLevel(newSamplingLevel, maxSummarySize);
            double effectiveIndexInterval = sstable.getEffectiveIndexInterval();
            logger.trace("{} has {} reads/sec; ideal space for index summary: {} bytes ({} entries); considering moving from level {} ({} entries, {} bytes) to level {} ({} entries, {} bytes)", new Object[]{sstable.getFilename(), readsPerSec, idealSpace, targetNumEntries, currentSamplingLevel, currentNumEntries, (double)currentNumEntries * avgEntrySize, newSamplingLevel, numEntriesAtNewSamplingLevel, (double)numEntriesAtNewSamplingLevel * avgEntrySize});
            if (effectiveIndexInterval < (double)minIndexInterval) {
                logger.debug("Forcing resample of {} because the current index interval ({}) is below min_index_interval ({})", new Object[]{sstable, effectiveIndexInterval, minIndexInterval});
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                forceResample.add(new ResampleEntry(sstable, spaceUsed, newSamplingLevel));
                this.remainingSpace -= spaceUsed;
            } else if (effectiveIndexInterval > (double)maxIndexInterval) {
                logger.debug("Forcing upsample of {} because the current index interval ({}) is above max_index_interval ({})", new Object[]{sstable, effectiveIndexInterval, maxIndexInterval});
                newSamplingLevel = Math.max(1, 128 * minIndexInterval / maxIndexInterval);
                numEntriesAtNewSamplingLevel = IndexSummaryBuilder.entriesAtSamplingLevel(newSamplingLevel, sstable.getMaxIndexSummarySize());
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                forceUpsample.add(new ResampleEntry(sstable, spaceUsed, newSamplingLevel));
                this.remainingSpace = (long)((double)this.remainingSpace - avgEntrySize * (double)numEntriesAtNewSamplingLevel);
            } else if ((double)targetNumEntries >= (double)currentNumEntries * 1.5 && newSamplingLevel > currentSamplingLevel) {
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                toUpsample.add(new ResampleEntry(sstable, spaceUsed, newSamplingLevel));
                this.remainingSpace = (long)((double)this.remainingSpace - avgEntrySize * (double)numEntriesAtNewSamplingLevel);
            } else if ((double)targetNumEntries < (double)currentNumEntries * 0.75 && newSamplingLevel < currentSamplingLevel) {
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                toDownsample.add(new ResampleEntry(sstable, spaceUsed, newSamplingLevel));
                this.remainingSpace -= spaceUsed;
            } else {
                logger.trace("SSTable {} is within thresholds of ideal sampling", (Object)sstable);
                this.remainingSpace -= sstable.getIndexSummaryOffHeapSize();
                newSSTables.add(sstable);
            }
            totalReadsPerSec -= readsPerSec;
        }
        if (this.remainingSpace > 0L) {
            Pair<List<SSTableReader>, List<ResampleEntry>> result = IndexSummaryRedistribution.distributeRemainingSpace(toDownsample, this.remainingSpace);
            toDownsample = (List)result.right;
            newSSTables.addAll((Collection)result.left);
        }
        toDownsample.addAll(forceResample);
        toDownsample.addAll(toUpsample);
        toDownsample.addAll(forceUpsample);
        HashMultimap replacedByTracker = HashMultimap.create();
        HashMultimap replacementsByTracker = HashMultimap.create();
        try {
            for (ResampleEntry entry : toDownsample) {
                if (this.isStopRequested()) {
                    throw new CompactionInterruptedException(this.getCompactionInfo());
                }
                SSTableReader sstable = entry.sstable;
                logger.debug("Re-sampling index summary for {} from {}/{} to {}/{} of the original number of entries", new Object[]{sstable, sstable.getIndexSummarySamplingLevel(), 128, entry.newSamplingLevel, 128});
                ColumnFamilyStore cfs = Keyspace.open(sstable.getKeyspaceName()).getColumnFamilyStore(sstable.getColumnFamilyName());
                DataTracker tracker = cfs.getDataTracker();
                SSTableReader replacement = sstable.cloneWithNewSummarySamplingLevel(cfs, entry.newSamplingLevel);
                newSSTables.add(replacement);
                replacedByTracker.put((Object)tracker, (Object)sstable);
                replacementsByTracker.put((Object)tracker, (Object)replacement);
            }
        }
        finally {
            for (DataTracker tracker : replacedByTracker.keySet()) {
                tracker.replaceWithNewInstances(replacedByTracker.get((Object)tracker), replacementsByTracker.get((Object)tracker));
            }
        }
        return newSSTables;
    }

    @VisibleForTesting
    static Pair<List<SSTableReader>, List<ResampleEntry>> distributeRemainingSpace(List<ResampleEntry> toDownsample, long remainingSpace) {
        int noDownsampleCutoff;
        long extraSpaceRequired;
        Collections.sort(toDownsample, new Comparator<ResampleEntry>(){

            @Override
            public int compare(ResampleEntry o1, ResampleEntry o2) {
                return Double.compare(o1.sstable.getIndexSummaryOffHeapSize() - o1.newSpaceUsed, o2.sstable.getIndexSummaryOffHeapSize() - o2.newSpaceUsed);
            }
        });
        ArrayList<SSTableReader> willNotDownsample = new ArrayList<SSTableReader>();
        for (noDownsampleCutoff = 0; remainingSpace > 0L && noDownsampleCutoff < toDownsample.size(); remainingSpace -= extraSpaceRequired, ++noDownsampleCutoff) {
            ResampleEntry entry = toDownsample.get(noDownsampleCutoff);
            extraSpaceRequired = entry.sstable.getIndexSummaryOffHeapSize() - entry.newSpaceUsed;
            if (extraSpaceRequired > remainingSpace) break;
            logger.trace("Using leftover space to keep {} at the current sampling level ({})", (Object)entry.sstable, (Object)entry.sstable.getIndexSummarySamplingLevel());
            willNotDownsample.add(entry.sstable);
        }
        return Pair.create(willNotDownsample, toDownsample.subList(noDownsampleCutoff, toDownsample.size()));
    }

    @Override
    public CompactionInfo getCompactionInfo() {
        return new CompactionInfo(OperationType.INDEX_SUMMARY, this.remainingSpace - this.memoryPoolBytes, this.memoryPoolBytes, "bytes");
    }

    private static class ResampleEntry {
        public final SSTableReader sstable;
        public final long newSpaceUsed;
        public final int newSamplingLevel;

        ResampleEntry(SSTableReader sstable, long newSpaceUsed, int newSamplingLevel) {
            this.sstable = sstable;
            this.newSpaceUsed = newSpaceUsed;
            this.newSamplingLevel = newSamplingLevel;
        }
    }

    private static class ReadRateComparator
    implements Comparator<SSTableReader> {
        private final Map<SSTableReader, Double> readRates;

        ReadRateComparator(Map<SSTableReader, Double> readRates) {
            this.readRates = readRates;
        }

        @Override
        public int compare(SSTableReader o1, SSTableReader o2) {
            Double readRate1 = this.readRates.get(o1);
            Double readRate2 = this.readRates.get(o2);
            if (readRate1 == null && readRate2 == null) {
                return 0;
            }
            if (readRate1 == null) {
                return -1;
            }
            if (readRate2 == null) {
                return 1;
            }
            return Double.compare(readRate1, readRate2);
        }
    }
}

