/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.io.util.IAllocator;
import org.apache.cassandra.io.util.NativeAllocator;
import org.apache.cassandra.io.util.SafeMemory;
import org.apache.cassandra.utils.FastByteOperations;
import org.apache.cassandra.utils.concurrent.Ref;
import org.apache.cassandra.utils.memory.MemoryUtil;
import sun.misc.Unsafe;
import sun.nio.ch.DirectBuffer;

public class Memory
implements AutoCloseable {
    private static final Unsafe unsafe = NativeAllocator.unsafe;
    static final IAllocator allocator = DatabaseDescriptor.getoffHeapMemoryAllocator();
    private static final long BYTE_ARRAY_BASE_OFFSET = unsafe.arrayBaseOffset(byte[].class);
    private static final boolean bigEndian = ByteOrder.nativeOrder().equals(ByteOrder.BIG_ENDIAN);
    private static final boolean unaligned;
    protected long peer;
    protected final long size;

    protected Memory(long bytes) {
        if (bytes <= 0L) {
            throw new AssertionError();
        }
        this.size = bytes;
        this.peer = allocator.allocate(this.size);
        if (this.peer == 0L) {
            throw new OutOfMemoryError();
        }
    }

    protected Memory(Memory copyOf) {
        this.size = copyOf.size;
        this.peer = copyOf.peer;
    }

    public static Memory allocate(long bytes) {
        if (bytes < 0L) {
            throw new IllegalArgumentException();
        }
        if (Ref.DEBUG_ENABLED) {
            return new SafeMemory(bytes);
        }
        return new Memory(bytes);
    }

    public void setByte(long offset, byte b) {
        this.checkBounds(offset, offset + 1L);
        unsafe.putByte(this.peer + offset, b);
    }

    public void setMemory(long offset, long bytes, byte b) {
        this.checkBounds(offset, offset + bytes);
        unsafe.setMemory(this.peer + offset, bytes, b);
    }

    public void setLong(long offset, long l) {
        this.checkBounds(offset, offset + 8L);
        if (unaligned) {
            unsafe.putLong(this.peer + offset, l);
        } else {
            this.putLongByByte(this.peer + offset, l);
        }
    }

    private void putLongByByte(long address, long value) {
        if (bigEndian) {
            unsafe.putByte(address, (byte)(value >> 56));
            unsafe.putByte(address + 1L, (byte)(value >> 48));
            unsafe.putByte(address + 2L, (byte)(value >> 40));
            unsafe.putByte(address + 3L, (byte)(value >> 32));
            unsafe.putByte(address + 4L, (byte)(value >> 24));
            unsafe.putByte(address + 5L, (byte)(value >> 16));
            unsafe.putByte(address + 6L, (byte)(value >> 8));
            unsafe.putByte(address + 7L, (byte)value);
        } else {
            unsafe.putByte(address + 7L, (byte)(value >> 56));
            unsafe.putByte(address + 6L, (byte)(value >> 48));
            unsafe.putByte(address + 5L, (byte)(value >> 40));
            unsafe.putByte(address + 4L, (byte)(value >> 32));
            unsafe.putByte(address + 3L, (byte)(value >> 24));
            unsafe.putByte(address + 2L, (byte)(value >> 16));
            unsafe.putByte(address + 1L, (byte)(value >> 8));
            unsafe.putByte(address, (byte)value);
        }
    }

    public void setInt(long offset, int l) {
        this.checkBounds(offset, offset + 4L);
        if (unaligned) {
            unsafe.putInt(this.peer + offset, l);
        } else {
            this.putIntByByte(this.peer + offset, l);
        }
    }

    private void putIntByByte(long address, int value) {
        if (bigEndian) {
            unsafe.putByte(address, (byte)(value >> 24));
            unsafe.putByte(address + 1L, (byte)(value >> 16));
            unsafe.putByte(address + 2L, (byte)(value >> 8));
            unsafe.putByte(address + 3L, (byte)value);
        } else {
            unsafe.putByte(address + 3L, (byte)(value >> 24));
            unsafe.putByte(address + 2L, (byte)(value >> 16));
            unsafe.putByte(address + 1L, (byte)(value >> 8));
            unsafe.putByte(address, (byte)value);
        }
    }

    public void setShort(long offset, short l) {
        this.checkBounds(offset, offset + 4L);
        if (unaligned) {
            unsafe.putShort(this.peer + offset, l);
        } else {
            this.putShortByByte(this.peer + offset, l);
        }
    }

    private void putShortByByte(long address, short value) {
        if (bigEndian) {
            unsafe.putByte(address, (byte)(value >> 8));
            unsafe.putByte(address + 1L, (byte)value);
        } else {
            unsafe.putByte(address + 1L, (byte)(value >> 8));
            unsafe.putByte(address, (byte)value);
        }
    }

    public void setBytes(long memoryOffset, ByteBuffer buffer) {
        if (buffer == null) {
            throw new NullPointerException();
        }
        if (buffer.remaining() == 0) {
            return;
        }
        this.checkBounds(memoryOffset, memoryOffset + (long)buffer.remaining());
        if (buffer.hasArray()) {
            this.setBytes(memoryOffset, buffer.array(), buffer.arrayOffset() + buffer.position(), buffer.remaining());
        } else if (buffer instanceof DirectBuffer) {
            unsafe.copyMemory(((DirectBuffer)((Object)buffer)).address() + (long)buffer.position(), this.peer + memoryOffset, buffer.remaining());
        } else {
            throw new IllegalStateException();
        }
    }

    public void setBytes(long memoryOffset, byte[] buffer, int bufferOffset, int count) {
        if (buffer == null) {
            throw new NullPointerException();
        }
        if (bufferOffset < 0 || count < 0 || bufferOffset + count > buffer.length) {
            throw new IndexOutOfBoundsException();
        }
        if (count == 0) {
            return;
        }
        long end = memoryOffset + (long)count;
        this.checkBounds(memoryOffset, end);
        unsafe.copyMemory(buffer, BYTE_ARRAY_BASE_OFFSET + (long)bufferOffset, null, this.peer + memoryOffset, count);
    }

    public byte getByte(long offset) {
        this.checkBounds(offset, offset + 1L);
        return unsafe.getByte(this.peer + offset);
    }

    public long getLong(long offset) {
        this.checkBounds(offset, offset + 8L);
        if (unaligned) {
            return unsafe.getLong(this.peer + offset);
        }
        return this.getLongByByte(this.peer + offset);
    }

    private long getLongByByte(long address) {
        if (bigEndian) {
            return (long)unsafe.getByte(address) << 56 | ((long)unsafe.getByte(address + 1L) & 0xFFL) << 48 | ((long)unsafe.getByte(address + 2L) & 0xFFL) << 40 | ((long)unsafe.getByte(address + 3L) & 0xFFL) << 32 | ((long)unsafe.getByte(address + 4L) & 0xFFL) << 24 | ((long)unsafe.getByte(address + 5L) & 0xFFL) << 16 | ((long)unsafe.getByte(address + 6L) & 0xFFL) << 8 | (long)unsafe.getByte(address + 7L) & 0xFFL;
        }
        return (long)unsafe.getByte(address + 7L) << 56 | ((long)unsafe.getByte(address + 6L) & 0xFFL) << 48 | ((long)unsafe.getByte(address + 5L) & 0xFFL) << 40 | ((long)unsafe.getByte(address + 4L) & 0xFFL) << 32 | ((long)unsafe.getByte(address + 3L) & 0xFFL) << 24 | ((long)unsafe.getByte(address + 2L) & 0xFFL) << 16 | ((long)unsafe.getByte(address + 1L) & 0xFFL) << 8 | (long)unsafe.getByte(address) & 0xFFL;
    }

    public int getInt(long offset) {
        this.checkBounds(offset, offset + 4L);
        if (unaligned) {
            return unsafe.getInt(this.peer + offset);
        }
        return this.getIntByByte(this.peer + offset);
    }

    private int getIntByByte(long address) {
        if (bigEndian) {
            return unsafe.getByte(address) << 24 | (unsafe.getByte(address + 1L) & 0xFF) << 16 | (unsafe.getByte(address + 2L) & 0xFF) << 8 | unsafe.getByte(address + 3L) & 0xFF;
        }
        return unsafe.getByte(address + 3L) << 24 | (unsafe.getByte(address + 2L) & 0xFF) << 16 | (unsafe.getByte(address + 1L) & 0xFF) << 8 | unsafe.getByte(address) & 0xFF;
    }

    public void getBytes(long memoryOffset, byte[] buffer, int bufferOffset, int count) {
        if (buffer == null) {
            throw new NullPointerException();
        }
        if (bufferOffset < 0 || count < 0 || count > buffer.length - bufferOffset) {
            throw new IndexOutOfBoundsException();
        }
        if (count == 0) {
            return;
        }
        this.checkBounds(memoryOffset, memoryOffset + (long)count);
        FastByteOperations.UnsafeOperations.copy(null, this.peer + memoryOffset, buffer, bufferOffset, count);
    }

    protected void checkBounds(long start, long end) {
        assert (this.peer != 0L) : "Memory was freed";
        assert (start >= 0L && end <= this.size && start <= end) : "Illegal bounds [" + start + ".." + end + "); size: " + this.size;
    }

    public void put(long trgOffset, Memory memory, long srcOffset, long size) {
        this.checkBounds(trgOffset, trgOffset + size);
        memory.checkBounds(srcOffset, srcOffset + size);
        unsafe.copyMemory(memory.peer + srcOffset, this.peer + trgOffset, size);
    }

    public Memory copy(long newSize) {
        Memory copy = Memory.allocate(newSize);
        copy.put(0L, this, 0L, Math.min(this.size(), newSize));
        return copy;
    }

    public void free() {
        if (this.peer != 0L) {
            allocator.free(this.peer);
        } else assert (this.size == 0L);
        this.peer = 0L;
    }

    @Override
    public void close() {
        this.free();
    }

    public long size() {
        assert (this.peer != 0L);
        return this.size;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Memory)) {
            return false;
        }
        Memory b = (Memory)o;
        return this.peer == b.peer && this.size == b.size;
    }

    public ByteBuffer[] asByteBuffers(long offset, long length) {
        if (this.size() == 0L) {
            return new ByteBuffer[0];
        }
        ByteBuffer[] result = new ByteBuffer[(int)(length / Integer.MAX_VALUE) + 1];
        int size = (int)(this.size() / (long)result.length);
        for (int i = 0; i < result.length - 1; ++i) {
            result[i] = MemoryUtil.getByteBuffer(this.peer + offset, size);
            offset += (long)size;
            length -= (long)size;
        }
        result[result.length - 1] = MemoryUtil.getByteBuffer(this.peer + offset, (int)length);
        return result;
    }

    public String toString() {
        return Memory.toString(this.peer, this.size);
    }

    protected static String toString(long peer, long size) {
        return String.format("Memory@[%x..%x)", peer, peer + size);
    }

    static {
        String arch = System.getProperty("os.arch");
        unaligned = arch.equals("i386") || arch.equals("x86") || arch.equals("amd64") || arch.equals("x86_64");
    }
}

