/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.cql;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeoutException;
import org.antlr.runtime.ANTLRStringStream;
import org.antlr.runtime.CharStream;
import org.antlr.runtime.CommonTokenStream;
import org.antlr.runtime.RecognitionException;
import org.antlr.runtime.TokenSource;
import org.antlr.runtime.TokenStream;
import org.apache.cassandra.auth.Permission;
import org.apache.cassandra.cli.CliUtils;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.config.IndexType;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.cql.AbstractModification;
import org.apache.cassandra.cql.AlterTableStatement;
import org.apache.cassandra.cql.BatchStatement;
import org.apache.cassandra.cql.CQLStatement;
import org.apache.cassandra.cql.CqlLexer;
import org.apache.cassandra.cql.CqlParser;
import org.apache.cassandra.cql.CreateColumnFamilyStatement;
import org.apache.cassandra.cql.CreateIndexStatement;
import org.apache.cassandra.cql.CreateKeyspaceStatement;
import org.apache.cassandra.cql.DeleteStatement;
import org.apache.cassandra.cql.DropIndexStatement;
import org.apache.cassandra.cql.Relation;
import org.apache.cassandra.cql.RelationType;
import org.apache.cassandra.cql.SelectStatement;
import org.apache.cassandra.cql.StatementType;
import org.apache.cassandra.cql.Term;
import org.apache.cassandra.cql.UpdateStatement;
import org.apache.cassandra.cql.hooks.ExecutionContext;
import org.apache.cassandra.cql.hooks.PostPreparationHook;
import org.apache.cassandra.cql.hooks.PreExecutionHook;
import org.apache.cassandra.cql.hooks.PreparationContext;
import org.apache.cassandra.db.Cell;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.CounterCell;
import org.apache.cassandra.db.IMutation;
import org.apache.cassandra.db.IndexExpression;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.RangeSliceCommand;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.SliceByNamesReadCommand;
import org.apache.cassandra.db.SliceFromReadCommand;
import org.apache.cassandra.db.composites.CellName;
import org.apache.cassandra.db.composites.CellNameType;
import org.apache.cassandra.db.composites.Composite;
import org.apache.cassandra.db.context.CounterContext;
import org.apache.cassandra.db.filter.IDiskAtomFilter;
import org.apache.cassandra.db.filter.NamesQueryFilter;
import org.apache.cassandra.db.filter.SliceQueryFilter;
import org.apache.cassandra.db.index.SecondaryIndexManager;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.AsciiType;
import org.apache.cassandra.db.marshal.TypeParser;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.RandomPartitioner;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.exceptions.IsBootstrappingException;
import org.apache.cassandra.exceptions.ReadTimeoutException;
import org.apache.cassandra.exceptions.RequestExecutionException;
import org.apache.cassandra.exceptions.RequestValidationException;
import org.apache.cassandra.exceptions.SyntaxException;
import org.apache.cassandra.exceptions.TruncateException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.serializers.MarshalException;
import org.apache.cassandra.service.MigrationManager;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.thrift.Column;
import org.apache.cassandra.thrift.CqlMetadata;
import org.apache.cassandra.thrift.CqlPreparedResult;
import org.apache.cassandra.thrift.CqlResult;
import org.apache.cassandra.thrift.CqlResultType;
import org.apache.cassandra.thrift.CqlRow;
import org.apache.cassandra.thrift.ThriftClientState;
import org.apache.cassandra.thrift.ThriftValidation;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.SemanticVersion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueryProcessor {
    public static final SemanticVersion CQL_VERSION = new SemanticVersion("2.0.0");
    private static final Logger logger = LoggerFactory.getLogger(QueryProcessor.class);
    public static final String DEFAULT_KEY_NAME = "key".toUpperCase();
    private static final List<PreExecutionHook> preExecutionHooks = new CopyOnWriteArrayList<PreExecutionHook>();
    private static final List<PostPreparationHook> postPreparationHooks = new CopyOnWriteArrayList<PostPreparationHook>();

    public static void addPreExecutionHook(PreExecutionHook hook) {
        preExecutionHooks.add(hook);
    }

    public static void removePreExecutionHook(PreExecutionHook hook) {
        preExecutionHooks.remove(hook);
    }

    public static void addPostPreparationHook(PostPreparationHook hook) {
        postPreparationHooks.add(hook);
    }

    public static void removePostPreparationHook(PostPreparationHook hook) {
        postPreparationHooks.remove(hook);
    }

    private static List<Row> getSlice(CFMetaData metadata, SelectStatement select, List<ByteBuffer> variables, long now) throws InvalidRequestException, ReadTimeoutException, UnavailableException, IsBootstrappingException {
        ArrayList<ReadCommand> commands = new ArrayList<ReadCommand>();
        if (!select.isColumnRange()) {
            SortedSet<CellName> columnNames = QueryProcessor.getColumnNames(select, metadata, variables);
            QueryProcessor.validateColumnNames(columnNames);
            for (Term rawKey : select.getKeys()) {
                ByteBuffer key = rawKey.getByteBuffer(metadata.getKeyValidator(), variables);
                QueryProcessor.validateKey(key);
                commands.add(new SliceByNamesReadCommand(metadata.ksName, key, select.getColumnFamily(), now, new NamesQueryFilter(columnNames)));
            }
        } else {
            AbstractType<?> at = metadata.comparator.asAbstractType();
            Composite start = metadata.comparator.fromByteBuffer(select.getColumnStart().getByteBuffer(at, variables));
            Composite finish = metadata.comparator.fromByteBuffer(select.getColumnFinish().getByteBuffer(at, variables));
            for (Term rawKey : select.getKeys()) {
                ByteBuffer key = rawKey.getByteBuffer(metadata.getKeyValidator(), variables);
                QueryProcessor.validateKey(key);
                QueryProcessor.validateSliceFilter(metadata, start, finish, select.isColumnsReversed());
                commands.add(new SliceFromReadCommand(metadata.ksName, key, select.getColumnFamily(), now, new SliceQueryFilter(start, finish, select.isColumnsReversed(), select.getColumnsLimit())));
            }
        }
        return StorageProxy.read(commands, select.getConsistencyLevel());
    }

    private static SortedSet<CellName> getColumnNames(SelectStatement select, CFMetaData metadata, List<ByteBuffer> variables) throws InvalidRequestException {
        String keyString = metadata.getCQL2KeyName();
        List<Term> selectColumnNames = select.getColumnNames();
        TreeSet<Composite> columnNames = new TreeSet<Composite>(metadata.comparator);
        for (Term column : selectColumnNames) {
            if (column.getText().equalsIgnoreCase(keyString)) continue;
            columnNames.add(metadata.comparator.cellFromByteBuffer(column.getByteBuffer(metadata.comparator.asAbstractType(), variables)));
        }
        return columnNames;
    }

    private static List<Row> multiRangeSlice(CFMetaData metadata, SelectStatement select, List<ByteBuffer> variables, long now) throws ReadTimeoutException, UnavailableException, InvalidRequestException {
        RowPosition finishKey;
        IPartitioner p = StorageService.getPartitioner();
        AbstractType<?> keyType = Schema.instance.getCFMetaData(metadata.ksName, select.getColumnFamily()).getKeyValidator();
        ByteBuffer startKeyBytes = select.getKeyStart() != null ? select.getKeyStart().getByteBuffer(keyType, variables) : null;
        ByteBuffer finishKeyBytes = select.getKeyFinish() != null ? select.getKeyFinish().getByteBuffer(keyType, variables) : null;
        RowPosition startKey = RowPosition.ForKey.get(startKeyBytes, p);
        if (startKey.compareTo(finishKey = RowPosition.ForKey.get(finishKeyBytes, p)) > 0 && !finishKey.isMinimum(p)) {
            if (p instanceof RandomPartitioner) {
                throw new InvalidRequestException("Start key sorts after end key. This is not allowed; you probably should not specify end key at all, under RandomPartitioner");
            }
            throw new InvalidRequestException("Start key must sort before (or equal to) finish key in your partitioner!");
        }
        Bounds<RowPosition> bounds = new Bounds<RowPosition>(startKey, finishKey);
        IDiskAtomFilter columnFilter = QueryProcessor.filterFromSelect(select, metadata, variables);
        QueryProcessor.validateFilter(metadata, columnFilter);
        List<Relation> columnRelations = select.getColumnRelations();
        ArrayList<IndexExpression> expressions = new ArrayList<IndexExpression>(columnRelations.size());
        for (Relation columnRelation : columnRelations) {
            ByteBuffer entity = columnRelation.getEntity().getByteBuffer(metadata.comparator.asAbstractType(), variables);
            ByteBuffer value = columnRelation.getValue().getByteBuffer(metadata.getValueValidator(metadata.comparator.cellFromByteBuffer(entity)), variables);
            expressions.add(new IndexExpression(entity, IndexExpression.Operator.valueOf(columnRelation.operator().toString()), value));
        }
        int limit = select.isKeyRange() && select.getKeyStart() != null ? select.getNumRecords() + 1 : select.getNumRecords();
        List<Row> rows = StorageProxy.getRangeSlice(new RangeSliceCommand(metadata.ksName, select.getColumnFamily(), now, columnFilter, bounds, expressions, limit), select.getConsistencyLevel());
        if (select.getKeyStart() != null && !select.includeStartKey() && !rows.isEmpty() && rows.get((int)0).key.getKey().equals(startKeyBytes)) {
            rows.remove(0);
        }
        if (select.getKeyFinish() != null && !select.includeFinishKey() && !rows.isEmpty()) {
            int lastIndex = rows.size() - 1;
            if (rows.get((int)lastIndex).key.getKey().equals(finishKeyBytes)) {
                rows.remove(lastIndex);
            }
        }
        return rows.subList(0, select.getNumRecords() < rows.size() ? select.getNumRecords() : rows.size());
    }

    private static IDiskAtomFilter filterFromSelect(SelectStatement select, CFMetaData metadata, List<ByteBuffer> variables) throws InvalidRequestException {
        if (select.isColumnRange() || select.getColumnNames().size() == 0) {
            AbstractType<?> comparator = metadata.comparator.asAbstractType();
            return new SliceQueryFilter(metadata.comparator.fromByteBuffer(select.getColumnStart().getByteBuffer(comparator, variables)), metadata.comparator.fromByteBuffer(select.getColumnFinish().getByteBuffer(comparator, variables)), select.isColumnsReversed(), select.getColumnsLimit());
        }
        return new NamesQueryFilter(QueryProcessor.getColumnNames(select, metadata, variables));
    }

    private static void validateSelect(String keyspace, SelectStatement select, List<ByteBuffer> variables) throws InvalidRequestException {
        select.getConsistencyLevel().validateForRead(keyspace);
        if (!select.isKeyRange() && select.getKeyFinish() != null) {
            throw new InvalidRequestException("Key range clauses must include a start key (i.e. KEY > term)");
        }
        if (select.isKeyRange() && select.getKeys().size() > 0) {
            throw new InvalidRequestException("You cannot combine key range and by-key clauses in a SELECT");
        }
        if (select.isKeyRange() && select.getKeyFinish() != null && select.getColumnRelations().size() > 0) {
            throw new InvalidRequestException("You cannot combine key range and by-column clauses in a SELECT");
        }
        if (!select.isMultiKey() && select.getKeys().size() > 1) {
            throw new InvalidRequestException("You cannot use more than one KEY = in a SELECT");
        }
        if (select.getColumnRelations().size() > 0) {
            ColumnFamilyStore cfstore = Keyspace.open(keyspace).getColumnFamilyStore(select.getColumnFamily());
            CellNameType comparator = cfstore.metadata.comparator;
            AbstractType<?> at = comparator.asAbstractType();
            SecondaryIndexManager idxManager = cfstore.indexManager;
            for (Relation relation : select.getColumnRelations()) {
                ByteBuffer name = relation.getEntity().getByteBuffer(at, variables);
                if (relation.operator() != RelationType.EQ || !idxManager.indexes(comparator.cellFromByteBuffer(name))) continue;
                return;
            }
            throw new InvalidRequestException("No indexed columns present in by-columns clause with \"equals\" operator");
        }
    }

    public static void validateKey(ByteBuffer key) throws InvalidRequestException {
        if (key == null || key.remaining() == 0) {
            throw new InvalidRequestException("Key may not be empty");
        }
        if (key.remaining() > 65535) {
            throw new InvalidRequestException("Key length of " + key.remaining() + " is longer than maximum of " + 65535);
        }
    }

    public static void validateKeyAlias(CFMetaData cfm, String key) throws InvalidRequestException {
        assert (key.toUpperCase().equals(key));
        String realKeyAlias = cfm.getCQL2KeyName().toUpperCase();
        if (!realKeyAlias.equals(key)) {
            throw new InvalidRequestException(String.format("Expected key '%s' to be present in WHERE clause for '%s'", realKeyAlias, cfm.cfName));
        }
    }

    private static void validateColumnNames(Iterable<CellName> columns) throws InvalidRequestException {
        for (CellName name : columns) {
            if (name.dataSize() > 65535) {
                throw new InvalidRequestException(String.format("column name is too long (%s > %s)", name.dataSize(), 65535));
            }
            if (!name.isEmpty()) continue;
            throw new InvalidRequestException("zero-length column name");
        }
    }

    public static void validateColumnName(CellName column) throws InvalidRequestException {
        QueryProcessor.validateColumnNames(Arrays.asList(column));
    }

    public static void validateColumn(CFMetaData metadata, CellName name, ByteBuffer value) throws InvalidRequestException {
        QueryProcessor.validateColumnName(name);
        AbstractType<?> validator = metadata.getValueValidator(name);
        try {
            if (validator != null) {
                validator.validate(value);
            }
        }
        catch (MarshalException me) {
            throw new InvalidRequestException(String.format("Invalid column value for column (name=%s); %s", ByteBufferUtil.bytesToHex(name.toByteBuffer()), me.getMessage()));
        }
    }

    private static void validateFilter(CFMetaData metadata, IDiskAtomFilter filter) throws InvalidRequestException {
        if (filter instanceof SliceQueryFilter) {
            QueryProcessor.validateSliceFilter(metadata, (SliceQueryFilter)filter);
        } else {
            QueryProcessor.validateColumnNames(((NamesQueryFilter)filter).columns);
        }
    }

    private static void validateSliceFilter(CFMetaData metadata, SliceQueryFilter range) throws InvalidRequestException {
        QueryProcessor.validateSliceFilter(metadata, range.start(), range.finish(), range.reversed);
    }

    private static void validateSliceFilter(CFMetaData metadata, Composite start, Composite finish, boolean reversed) throws InvalidRequestException {
        Comparator<Composite> orderedComparator;
        CellNameType comparator = metadata.comparator;
        Comparator<Composite> comparator2 = orderedComparator = reversed ? comparator.reverseComparator() : comparator;
        if (!start.isEmpty() && !finish.isEmpty() && orderedComparator.compare(start, finish) > 0) {
            throw new InvalidRequestException("range finish must come after start in traversal order");
        }
    }

    public static CqlResult processStatement(CQLStatement statement, ExecutionContext context) throws RequestExecutionException, RequestValidationException {
        String keyspace = null;
        ThriftClientState clientState = context.clientState;
        List<ByteBuffer> variables = context.variables;
        if (statement.type != StatementType.SELECT && StatementType.REQUIRES_KEYSPACE.contains((Object)statement.type)) {
            keyspace = clientState.getKeyspace();
        }
        CqlResult result = new CqlResult();
        if (!preExecutionHooks.isEmpty()) {
            for (PreExecutionHook hook : preExecutionHooks) {
                statement = hook.processStatement(statement, context);
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("CQL statement type: {}", (Object)statement.type.toString());
        }
        switch (statement.type) {
            case SELECT: {
                SelectStatement select = (SelectStatement)statement.statement;
                String oldKeyspace = clientState.getRawKeyspace();
                if (select.isSetKeyspace()) {
                    keyspace = CliUtils.unescapeSQLString(select.getKeyspace());
                    ThriftValidation.validateKeyspace(keyspace);
                } else {
                    if (oldKeyspace == null) {
                        throw new InvalidRequestException("no keyspace has been specified");
                    }
                    keyspace = oldKeyspace;
                }
                clientState.hasColumnFamilyAccess(keyspace, select.getColumnFamily(), Permission.SELECT);
                CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily());
                select.extractKeyAliasFromColumns(metadata);
                if (select.getKeys().size() > 0) {
                    QueryProcessor.validateKeyAlias(metadata, select.getKeyAlias());
                }
                QueryProcessor.validateSelect(keyspace, select, variables);
                long now = System.currentTimeMillis();
                List<Row> rows = !select.isKeyRange() && select.getKeys().size() > 0 ? QueryProcessor.getSlice(metadata, select, variables, now) : QueryProcessor.multiRangeSlice(metadata, select, variables, now);
                result.type = CqlResultType.ROWS;
                if (select.isCountOperation()) {
                    QueryProcessor.validateCountOperation(select);
                    ByteBuffer countBytes = ByteBufferUtil.bytes("count");
                    result.schema = new CqlMetadata(Collections.emptyMap(), Collections.emptyMap(), "AsciiType", "LongType");
                    List<Column> columns = Collections.singletonList(new Column(countBytes).setValue(ByteBufferUtil.bytes((long)rows.size())));
                    result.rows = Collections.singletonList(new CqlRow(countBytes, columns));
                    return result;
                }
                result.schema = new CqlMetadata(new HashMap(), new HashMap(), TypeParser.getShortName(metadata.comparator.asAbstractType()), TypeParser.getShortName(metadata.getDefaultValidator()));
                ArrayList<CqlRow> cqlRows = new ArrayList<CqlRow>(rows.size());
                for (Row row : rows) {
                    ArrayList<Column> thriftColumns = new ArrayList<Column>();
                    if (select.isColumnRange()) {
                        if (select.isFullWildcard()) {
                            ByteBuffer keyName = ByteBufferUtil.bytes(metadata.getCQL2KeyName());
                            thriftColumns.add(new Column(keyName).setValue(row.key.getKey()).setTimestamp(-1L));
                            result.schema.name_types.put(keyName, TypeParser.getShortName(AsciiType.instance));
                            result.schema.value_types.put(keyName, TypeParser.getShortName(metadata.getKeyValidator()));
                        }
                        if (row.cf != null) {
                            for (Cell c : row.cf.getSortedColumns()) {
                                if (!c.isLive(now)) continue;
                                ColumnDefinition cd = metadata.getColumnDefinition(c.name());
                                if (cd != null) {
                                    result.schema.value_types.put(c.name().toByteBuffer(), TypeParser.getShortName(cd.type));
                                }
                                thriftColumns.add(QueryProcessor.thriftify(c));
                            }
                        }
                    } else {
                        String keyString = metadata.getCQL2KeyName();
                        for (Term term : select.getColumnNames()) {
                            Cell c;
                            ByteBuffer nameBytes;
                            if (term.getText().equalsIgnoreCase(keyString)) {
                                ByteBuffer requestedKey = ByteBufferUtil.bytes(term.getText());
                                thriftColumns.add(new Column(requestedKey).setValue(row.key.getKey()).setTimestamp(-1L));
                                result.schema.name_types.put(requestedKey, TypeParser.getShortName(AsciiType.instance));
                                result.schema.value_types.put(requestedKey, TypeParser.getShortName(metadata.getKeyValidator()));
                                continue;
                            }
                            if (row.cf == null) continue;
                            try {
                                nameBytes = term.getByteBuffer(metadata.comparator.asAbstractType(), variables);
                            }
                            catch (InvalidRequestException e) {
                                throw new AssertionError((Object)e);
                            }
                            CellName name = metadata.comparator.cellFromByteBuffer(nameBytes);
                            ColumnDefinition cd = metadata.getColumnDefinition(name);
                            if (cd != null) {
                                result.schema.value_types.put(nameBytes, TypeParser.getShortName(cd.type));
                            }
                            if ((c = row.cf.getColumn(name)) == null || !c.isLive()) {
                                thriftColumns.add(new Column().setName(nameBytes));
                                continue;
                            }
                            thriftColumns.add(QueryProcessor.thriftify(c));
                        }
                    }
                    CqlRow cqlRow = new CqlRow();
                    cqlRow.key = row.key.getKey();
                    cqlRow.columns = thriftColumns;
                    if (select.isColumnsReversed()) {
                        Collections.reverse(cqlRow.columns);
                    }
                    cqlRows.add(cqlRow);
                }
                result.rows = cqlRows;
                return result;
            }
            case INSERT: 
            case UPDATE: {
                UpdateStatement update = (UpdateStatement)statement.statement;
                update.getConsistencyLevel().validateForWrite(keyspace);
                keyspace = update.keyspace == null ? clientState.getKeyspace() : update.keyspace;
                List<IMutation> rowMutations = update.prepareRowMutations(keyspace, clientState, variables);
                for (IMutation mutation : rowMutations) {
                    QueryProcessor.validateKey(mutation.key());
                }
                StorageProxy.mutateWithTriggers(rowMutations, update.getConsistencyLevel(), false);
                result.type = CqlResultType.VOID;
                return result;
            }
            case BATCH: {
                BatchStatement batch = (BatchStatement)statement.statement;
                batch.getConsistencyLevel().validateForWrite(keyspace);
                if (batch.getTimeToLive() != 0) {
                    throw new InvalidRequestException("Global TTL on the BATCH statement is not supported.");
                }
                for (AbstractModification up : batch.getStatements()) {
                    if (up.isSetConsistencyLevel()) {
                        throw new InvalidRequestException("Consistency level must be set on the BATCH, not individual statements");
                    }
                    if (!batch.isSetTimestamp() || !up.isSetTimestamp()) continue;
                    throw new InvalidRequestException("Timestamp must be set either on BATCH or individual statements");
                }
                List<IMutation> mutations = batch.getMutations(keyspace, clientState, variables);
                for (IMutation mutation : mutations) {
                    QueryProcessor.validateKey(mutation.key());
                }
                StorageProxy.mutateWithTriggers(mutations, batch.getConsistencyLevel(), false);
                result.type = CqlResultType.VOID;
                return result;
            }
            case USE: {
                clientState.validateLogin();
                clientState.setKeyspace(CliUtils.unescapeSQLString((String)statement.statement));
                result.type = CqlResultType.VOID;
                return result;
            }
            case TRUNCATE: {
                Pair columnFamily = (Pair)statement.statement;
                keyspace = columnFamily.left == null ? clientState.getKeyspace() : (String)columnFamily.left;
                ThriftValidation.validateColumnFamily(keyspace, (String)columnFamily.right);
                clientState.hasColumnFamilyAccess(keyspace, (String)columnFamily.right, Permission.MODIFY);
                try {
                    StorageProxy.truncateBlocking(keyspace, (String)columnFamily.right);
                }
                catch (TimeoutException e) {
                    throw new TruncateException(e);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DELETE: {
                DeleteStatement delete = (DeleteStatement)statement.statement;
                keyspace = delete.keyspace == null ? clientState.getKeyspace() : delete.keyspace;
                List<IMutation> deletions = delete.prepareRowMutations(keyspace, clientState, variables);
                for (IMutation deletion : deletions) {
                    QueryProcessor.validateKey(deletion.key());
                }
                StorageProxy.mutateWithTriggers(deletions, delete.getConsistencyLevel(), false);
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_KEYSPACE: {
                CreateKeyspaceStatement create = (CreateKeyspaceStatement)statement.statement;
                create.validate();
                ThriftValidation.validateKeyspaceNotSystem(create.getName());
                clientState.hasAllKeyspacesAccess(Permission.CREATE);
                try {
                    KSMetaData ksm = KSMetaData.newKeyspace(create.getName(), create.getStrategyClass(), create.getStrategyOptions(), true);
                    ThriftValidation.validateKeyspaceNotYetExisting(ksm.name);
                    MigrationManager.announceNewKeyspace(ksm);
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause(e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_COLUMNFAMILY: {
                CreateColumnFamilyStatement createCf = (CreateColumnFamilyStatement)statement.statement;
                clientState.hasKeyspaceAccess(keyspace, Permission.CREATE);
                try {
                    MigrationManager.announceNewColumnFamily(createCf.getCFMetaData(keyspace, variables));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause(e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_INDEX: {
                CreateIndexStatement createIdx = (CreateIndexStatement)statement.statement;
                clientState.hasColumnFamilyAccess(keyspace, createIdx.getColumnFamily(), Permission.ALTER);
                CFMetaData oldCfm = Schema.instance.getCFMetaData(keyspace, createIdx.getColumnFamily());
                if (oldCfm == null) {
                    throw new InvalidRequestException("No such column family: " + createIdx.getColumnFamily());
                }
                boolean columnExists = false;
                ByteBuffer columnName = createIdx.getColumnName().getByteBuffer();
                CFMetaData cfm = oldCfm.copy();
                for (ColumnDefinition cd : cfm.regularColumns()) {
                    if (!cd.name.bytes.equals(columnName)) continue;
                    if (cd.getIndexType() != null) {
                        throw new InvalidRequestException("Index already exists");
                    }
                    if (logger.isDebugEnabled()) {
                        logger.debug("Updating column {} definition for index {}", (Object)cfm.comparator.getString(cfm.comparator.fromByteBuffer(columnName)), (Object)createIdx.getIndexName());
                    }
                    cd.setIndexType(IndexType.KEYS, Collections.emptyMap());
                    cd.setIndexName(createIdx.getIndexName());
                    columnExists = true;
                    break;
                }
                if (!columnExists) {
                    throw new InvalidRequestException("No column definition found for column " + oldCfm.comparator.getString(cfm.comparator.fromByteBuffer(columnName)));
                }
                try {
                    cfm.addDefaultIndexNames();
                    MigrationManager.announceColumnFamilyUpdate(cfm, true);
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause(e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DROP_INDEX: {
                DropIndexStatement dropIdx = (DropIndexStatement)statement.statement;
                keyspace = clientState.getKeyspace();
                dropIdx.setKeyspace(keyspace);
                clientState.hasColumnFamilyAccess(keyspace, dropIdx.getColumnFamily(), Permission.ALTER);
                try {
                    CFMetaData updatedCF = dropIdx.generateCFMetadataUpdate();
                    MigrationManager.announceColumnFamilyUpdate(updatedCF, true);
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause(e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DROP_KEYSPACE: {
                String deleteKeyspace = (String)statement.statement;
                ThriftValidation.validateKeyspaceNotSystem(deleteKeyspace);
                clientState.hasKeyspaceAccess(deleteKeyspace, Permission.DROP);
                try {
                    MigrationManager.announceKeyspaceDrop(deleteKeyspace);
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause(e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DROP_COLUMNFAMILY: {
                String deleteColumnFamily = (String)statement.statement;
                clientState.hasColumnFamilyAccess(keyspace, deleteColumnFamily, Permission.DROP);
                try {
                    MigrationManager.announceColumnFamilyDrop(keyspace, deleteColumnFamily);
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause(e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case ALTER_TABLE: {
                AlterTableStatement alterTable = (AlterTableStatement)statement.statement;
                ThriftValidation.validateColumnFamily(keyspace, alterTable.columnFamily);
                clientState.hasColumnFamilyAccess(keyspace, alterTable.columnFamily, Permission.ALTER);
                try {
                    MigrationManager.announceColumnFamilyUpdate(alterTable.getCFMetaData(keyspace), true);
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause(e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
        }
        return null;
    }

    public static CqlResult process(String queryString, ThriftClientState clientState) throws RequestValidationException, RequestExecutionException {
        logger.trace("CQL QUERY: {}", (Object)queryString);
        return QueryProcessor.processStatement(QueryProcessor.getStatement(queryString), new ExecutionContext(clientState, queryString, Collections.emptyList()));
    }

    public static CqlPreparedResult prepare(String queryString, ThriftClientState clientState) throws RequestValidationException {
        logger.trace("CQL QUERY: {}", (Object)queryString);
        CQLStatement statement = QueryProcessor.getStatement(queryString);
        int statementId = QueryProcessor.makeStatementId(queryString);
        logger.trace("Discovered " + statement.boundTerms + " bound variables.");
        clientState.getPrepared().put(statementId, statement);
        logger.trace(String.format("Stored prepared statement #%d with %d bind markers", statementId, statement.boundTerms));
        if (!postPreparationHooks.isEmpty()) {
            PreparationContext context = new PreparationContext(clientState, queryString, statement);
            for (PostPreparationHook hook : postPreparationHooks) {
                hook.processStatement(statement, context);
            }
        }
        return new CqlPreparedResult(statementId, statement.boundTerms);
    }

    public static CqlResult processPrepared(CQLStatement statement, ThriftClientState clientState, List<ByteBuffer> variables) throws RequestValidationException, RequestExecutionException {
        if (!variables.isEmpty() || statement.boundTerms != 0) {
            if (variables.size() != statement.boundTerms) {
                throw new InvalidRequestException(String.format("there were %d markers(?) in CQL but %d bound variables", statement.boundTerms, variables.size()));
            }
            if (logger.isTraceEnabled()) {
                for (int i = 0; i < variables.size(); ++i) {
                    logger.trace("[{}] '{}'", (Object)(i + 1), (Object)variables.get(i));
                }
            }
        }
        return QueryProcessor.processStatement(statement, new ExecutionContext(clientState, null, variables));
    }

    private static final int makeStatementId(String cql) {
        return cql.hashCode();
    }

    private static Column thriftify(Cell c) {
        ByteBuffer value = c instanceof CounterCell ? ByteBufferUtil.bytes(CounterContext.instance().total(c.value())) : c.value();
        return new Column(c.name().toByteBuffer()).setValue(value).setTimestamp(c.timestamp());
    }

    private static CQLStatement getStatement(String queryStr) throws SyntaxException {
        try {
            ANTLRStringStream stream = new ANTLRStringStream(queryStr);
            CqlLexer lexer = new CqlLexer((CharStream)stream);
            CommonTokenStream tokenStream = new CommonTokenStream((TokenSource)lexer);
            CqlParser parser = new CqlParser((TokenStream)tokenStream);
            CQLStatement statement = parser.query();
            lexer.throwLastRecognitionError();
            parser.throwLastRecognitionError();
            return statement;
        }
        catch (RuntimeException re) {
            SyntaxException ire = new SyntaxException("Failed parsing statement: [" + queryStr + "] reason: " + re.getClass().getSimpleName() + " " + re.getMessage());
            throw ire;
        }
        catch (RecognitionException e) {
            SyntaxException ire = new SyntaxException("Invalid or malformed CQL query string: " + e.getMessage());
            throw ire;
        }
    }

    private static void validateCountOperation(SelectStatement select) throws InvalidRequestException {
        if (select.isWildcard()) {
            return;
        }
        if (!select.isColumnRange()) {
            List<Term> columnNames = select.getColumnNames();
            String firstColumn = columnNames.get(0).getText();
            if (columnNames.size() == 1 && (firstColumn.equals("*") || firstColumn.equals("1"))) {
                return;
            }
        }
        throw new InvalidRequestException("Only COUNT(*) and COUNT(1) operations are currently supported.");
    }
}

