/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import com.google.common.util.concurrent.Futures;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.locks.Lock;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.cache.AutoSavingCache;
import org.apache.cassandra.cache.ConcurrentLinkedHashCache;
import org.apache.cassandra.cache.CounterCacheKey;
import org.apache.cassandra.cache.ICache;
import org.apache.cassandra.cache.IRowCacheEntry;
import org.apache.cassandra.cache.KeyCacheKey;
import org.apache.cassandra.cache.RowCacheKey;
import org.apache.cassandra.cache.SerializingCacheProvider;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.Cell;
import org.apache.cassandra.db.ClockAndCount;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.composites.CellName;
import org.apache.cassandra.db.context.CounterContext;
import org.apache.cassandra.db.filter.QueryFilter;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.service.CacheServiceMBean;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CacheService
implements CacheServiceMBean {
    private static final Logger logger = LoggerFactory.getLogger(CacheService.class);
    public static final String MBEAN_NAME = "org.apache.cassandra.db:type=Caches";
    public static final CacheService instance = new CacheService();
    public final AutoSavingCache<KeyCacheKey, RowIndexEntry> keyCache;
    public final AutoSavingCache<RowCacheKey, IRowCacheEntry> rowCache;
    public final AutoSavingCache<CounterCacheKey, ClockAndCount> counterCache;

    private CacheService() {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        this.keyCache = this.initKeyCache();
        this.rowCache = this.initRowCache();
        this.counterCache = this.initCounterCache();
    }

    private AutoSavingCache<KeyCacheKey, RowIndexEntry> initKeyCache() {
        logger.info("Initializing key cache with capacity of {} MBs.", (Object)DatabaseDescriptor.getKeyCacheSizeInMB());
        long keyCacheInMemoryCapacity = DatabaseDescriptor.getKeyCacheSizeInMB() * 1024L * 1024L;
        ConcurrentLinkedHashCache kc = ConcurrentLinkedHashCache.create(keyCacheInMemoryCapacity);
        AutoSavingCache<KeyCacheKey, RowIndexEntry> keyCache = new AutoSavingCache<KeyCacheKey, RowIndexEntry>(kc, CacheType.KEY_CACHE, new KeyCacheSerializer());
        int keyCacheKeysToSave = DatabaseDescriptor.getKeyCacheKeysToSave();
        keyCache.scheduleSaving(DatabaseDescriptor.getKeyCacheSavePeriod(), keyCacheKeysToSave);
        return keyCache;
    }

    private AutoSavingCache<RowCacheKey, IRowCacheEntry> initRowCache() {
        logger.info("Initializing row cache with capacity of {} MBs", (Object)DatabaseDescriptor.getRowCacheSizeInMB());
        long rowCacheInMemoryCapacity = DatabaseDescriptor.getRowCacheSizeInMB() * 1024L * 1024L;
        ICache<RowCacheKey, IRowCacheEntry> rc = new SerializingCacheProvider().create(rowCacheInMemoryCapacity);
        AutoSavingCache<RowCacheKey, IRowCacheEntry> rowCache = new AutoSavingCache<RowCacheKey, IRowCacheEntry>(rc, CacheType.ROW_CACHE, new RowCacheSerializer());
        int rowCacheKeysToSave = DatabaseDescriptor.getRowCacheKeysToSave();
        rowCache.scheduleSaving(DatabaseDescriptor.getRowCacheSavePeriod(), rowCacheKeysToSave);
        return rowCache;
    }

    private AutoSavingCache<CounterCacheKey, ClockAndCount> initCounterCache() {
        logger.info("Initializing counter cache with capacity of {} MBs", (Object)DatabaseDescriptor.getCounterCacheSizeInMB());
        long capacity = DatabaseDescriptor.getCounterCacheSizeInMB() * 1024L * 1024L;
        AutoSavingCache<CounterCacheKey, ClockAndCount> cache = new AutoSavingCache<CounterCacheKey, ClockAndCount>(ConcurrentLinkedHashCache.create(capacity), CacheType.COUNTER_CACHE, new CounterCacheSerializer());
        int keysToSave = DatabaseDescriptor.getCounterCacheKeysToSave();
        logger.info("Scheduling counter cache save to every {} seconds (going to save {} keys).", (Object)DatabaseDescriptor.getCounterCacheSavePeriod(), keysToSave == Integer.MAX_VALUE ? "all" : Integer.valueOf(keysToSave));
        cache.scheduleSaving(DatabaseDescriptor.getCounterCacheSavePeriod(), keysToSave);
        return cache;
    }

    @Override
    public long getKeyCacheHits() {
        return this.keyCache.getMetrics().hits.count();
    }

    @Override
    public long getRowCacheHits() {
        return this.rowCache.getMetrics().hits.count();
    }

    @Override
    public long getKeyCacheRequests() {
        return this.keyCache.getMetrics().requests.count();
    }

    @Override
    public long getRowCacheRequests() {
        return this.rowCache.getMetrics().requests.count();
    }

    @Override
    public double getKeyCacheRecentHitRate() {
        return this.keyCache.getMetrics().getRecentHitRate();
    }

    @Override
    public double getRowCacheRecentHitRate() {
        return this.rowCache.getMetrics().getRecentHitRate();
    }

    @Override
    public int getRowCacheSavePeriodInSeconds() {
        return DatabaseDescriptor.getRowCacheSavePeriod();
    }

    @Override
    public void setRowCacheSavePeriodInSeconds(int seconds) {
        if (seconds < 0) {
            throw new RuntimeException("RowCacheSavePeriodInSeconds must be non-negative.");
        }
        DatabaseDescriptor.setRowCacheSavePeriod(seconds);
        this.rowCache.scheduleSaving(seconds, DatabaseDescriptor.getRowCacheKeysToSave());
    }

    @Override
    public int getKeyCacheSavePeriodInSeconds() {
        return DatabaseDescriptor.getKeyCacheSavePeriod();
    }

    @Override
    public void setKeyCacheSavePeriodInSeconds(int seconds) {
        if (seconds < 0) {
            throw new RuntimeException("KeyCacheSavePeriodInSeconds must be non-negative.");
        }
        DatabaseDescriptor.setKeyCacheSavePeriod(seconds);
        this.keyCache.scheduleSaving(seconds, DatabaseDescriptor.getKeyCacheKeysToSave());
    }

    @Override
    public int getCounterCacheSavePeriodInSeconds() {
        return DatabaseDescriptor.getCounterCacheSavePeriod();
    }

    @Override
    public void setCounterCacheSavePeriodInSeconds(int seconds) {
        if (seconds < 0) {
            throw new RuntimeException("CounterCacheSavePeriodInSeconds must be non-negative.");
        }
        DatabaseDescriptor.setCounterCacheSavePeriod(seconds);
        this.counterCache.scheduleSaving(seconds, DatabaseDescriptor.getCounterCacheKeysToSave());
    }

    @Override
    public int getRowCacheKeysToSave() {
        return DatabaseDescriptor.getRowCacheKeysToSave();
    }

    @Override
    public void setRowCacheKeysToSave(int count) {
        if (count < 0) {
            throw new RuntimeException("RowCacheKeysToSave must be non-negative.");
        }
        DatabaseDescriptor.setRowCacheKeysToSave(count);
        this.rowCache.scheduleSaving(this.getRowCacheSavePeriodInSeconds(), count);
    }

    @Override
    public int getKeyCacheKeysToSave() {
        return DatabaseDescriptor.getKeyCacheKeysToSave();
    }

    @Override
    public void setKeyCacheKeysToSave(int count) {
        if (count < 0) {
            throw new RuntimeException("KeyCacheKeysToSave must be non-negative.");
        }
        DatabaseDescriptor.setKeyCacheKeysToSave(count);
        this.keyCache.scheduleSaving(this.getKeyCacheSavePeriodInSeconds(), count);
    }

    @Override
    public int getCounterCacheKeysToSave() {
        return DatabaseDescriptor.getCounterCacheKeysToSave();
    }

    @Override
    public void setCounterCacheKeysToSave(int count) {
        if (count < 0) {
            throw new RuntimeException("CounterCacheKeysToSave must be non-negative.");
        }
        DatabaseDescriptor.setCounterCacheKeysToSave(count);
        this.counterCache.scheduleSaving(this.getCounterCacheSavePeriodInSeconds(), count);
    }

    @Override
    public void invalidateKeyCache() {
        this.keyCache.clear();
    }

    @Override
    public void invalidateRowCache() {
        this.rowCache.clear();
    }

    public void invalidateRowCacheForCf(UUID cfId) {
        Iterator rowCacheIterator = this.rowCache.getKeySet().iterator();
        while (rowCacheIterator.hasNext()) {
            RowCacheKey rowCacheKey = (RowCacheKey)rowCacheIterator.next();
            if (!rowCacheKey.cfId.equals(cfId)) continue;
            rowCacheIterator.remove();
        }
    }

    @Override
    public void invalidateCounterCache() {
        this.counterCache.clear();
    }

    @Override
    public long getRowCacheCapacityInBytes() {
        return (Long)this.rowCache.getMetrics().capacity.value();
    }

    @Override
    public long getRowCacheCapacityInMB() {
        return this.getRowCacheCapacityInBytes() / 1024L / 1024L;
    }

    @Override
    public void setRowCacheCapacityInMB(long capacity) {
        if (capacity < 0L) {
            throw new RuntimeException("capacity should not be negative.");
        }
        this.rowCache.setCapacity(capacity * 1024L * 1024L);
    }

    @Override
    public long getKeyCacheCapacityInBytes() {
        return (Long)this.keyCache.getMetrics().capacity.value();
    }

    @Override
    public long getKeyCacheCapacityInMB() {
        return this.getKeyCacheCapacityInBytes() / 1024L / 1024L;
    }

    @Override
    public void setKeyCacheCapacityInMB(long capacity) {
        if (capacity < 0L) {
            throw new RuntimeException("capacity should not be negative.");
        }
        this.keyCache.setCapacity(capacity * 1024L * 1024L);
    }

    @Override
    public void setCounterCacheCapacityInMB(long capacity) {
        if (capacity < 0L) {
            throw new RuntimeException("capacity should not be negative.");
        }
        this.counterCache.setCapacity(capacity * 1024L * 1024L);
    }

    @Override
    public long getRowCacheSize() {
        return (Long)this.rowCache.getMetrics().size.value();
    }

    @Override
    public long getRowCacheEntries() {
        return this.rowCache.size();
    }

    @Override
    public long getKeyCacheSize() {
        return (Long)this.keyCache.getMetrics().size.value();
    }

    @Override
    public long getKeyCacheEntries() {
        return this.keyCache.size();
    }

    @Override
    public void saveCaches() throws ExecutionException, InterruptedException {
        ArrayList futures = new ArrayList(3);
        logger.debug("submitting cache saves");
        futures.add(this.keyCache.submitWrite(DatabaseDescriptor.getKeyCacheKeysToSave()));
        futures.add(this.rowCache.submitWrite(DatabaseDescriptor.getRowCacheKeysToSave()));
        futures.add(this.counterCache.submitWrite(DatabaseDescriptor.getCounterCacheKeysToSave()));
        FBUtilities.waitOnFutures(futures);
        logger.debug("cache saves completed");
    }

    public class KeyCacheSerializer
    implements AutoSavingCache.CacheSerializer<KeyCacheKey, RowIndexEntry> {
        @Override
        public void serialize(KeyCacheKey key, DataOutput out) throws IOException {
            RowIndexEntry entry = (RowIndexEntry)CacheService.instance.keyCache.get(key);
            if (entry == null) {
                return;
            }
            ByteBufferUtil.writeWithLength(key.key, out);
            Descriptor desc = key.desc;
            out.writeInt(desc.generation);
            out.writeBoolean(true);
            CFMetaData cfm = Schema.instance.getCFMetaData(key.desc.ksname, key.desc.cfname);
            cfm.comparator.rowIndexEntrySerializer().serialize(entry, out);
        }

        @Override
        public Future<Pair<KeyCacheKey, RowIndexEntry>> deserialize(DataInputStream input, ColumnFamilyStore cfs) throws IOException {
            int keyLength = input.readInt();
            if (keyLength > 65535) {
                throw new IOException(String.format("Corrupted key cache. Key length of %d is longer than maximum of %d", keyLength, 65535));
            }
            ByteBuffer key = ByteBufferUtil.read(input, keyLength);
            int generation = input.readInt();
            SSTableReader reader = this.findDesc(generation, cfs.getSSTables());
            input.readBoolean();
            if (reader == null) {
                RowIndexEntry.Serializer.skipPromotedIndex(input);
                return null;
            }
            RowIndexEntry entry = reader.metadata.comparator.rowIndexEntrySerializer().deserialize(input, reader.descriptor.version);
            return Futures.immediateFuture(Pair.create(new KeyCacheKey(cfs.metadata.cfId, reader.descriptor, key), entry));
        }

        private SSTableReader findDesc(int generation, Collection<SSTableReader> collection) {
            for (SSTableReader sstable : collection) {
                if (sstable.descriptor.generation != generation) continue;
                return sstable;
            }
            return null;
        }
    }

    public class RowCacheSerializer
    implements AutoSavingCache.CacheSerializer<RowCacheKey, IRowCacheEntry> {
        @Override
        public void serialize(RowCacheKey key, DataOutput out) throws IOException {
            ByteBufferUtil.writeWithLength(key.key, out);
        }

        @Override
        public Future<Pair<RowCacheKey, IRowCacheEntry>> deserialize(DataInputStream in, final ColumnFamilyStore cfs) throws IOException {
            final ByteBuffer buffer = ByteBufferUtil.readWithLength(in);
            return StageManager.getStage(Stage.READ).submit(new Callable<Pair<RowCacheKey, IRowCacheEntry>>(){

                @Override
                public Pair<RowCacheKey, IRowCacheEntry> call() throws Exception {
                    DecoratedKey key = cfs.partitioner.decorateKey(buffer);
                    QueryFilter cacheFilter = new QueryFilter(key, cfs.getColumnFamilyName(), cfs.readFilterForCache(), Integer.MIN_VALUE);
                    ColumnFamily data = cfs.getTopLevelColumns(cacheFilter, Integer.MIN_VALUE);
                    return Pair.create(new RowCacheKey(cfs.metadata.cfId, key), data);
                }
            });
        }
    }

    public class CounterCacheSerializer
    implements AutoSavingCache.CacheSerializer<CounterCacheKey, ClockAndCount> {
        @Override
        public void serialize(CounterCacheKey key, DataOutput out) throws IOException {
            ByteBufferUtil.writeWithLength(key.partitionKey, out);
            ByteBufferUtil.writeWithLength(key.cellName, out);
        }

        @Override
        public Future<Pair<CounterCacheKey, ClockAndCount>> deserialize(DataInputStream in, final ColumnFamilyStore cfs) throws IOException {
            final ByteBuffer partitionKey = ByteBufferUtil.readWithLength(in);
            final CellName cellName = cfs.metadata.comparator.cellFromByteBuffer(ByteBufferUtil.readWithLength(in));
            return StageManager.getStage(Stage.READ).submit(new Callable<Pair<CounterCacheKey, ClockAndCount>>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public Pair<CounterCacheKey, ClockAndCount> call() throws Exception {
                    DecoratedKey key = cfs.partitioner.decorateKey(partitionKey);
                    Lock lock = cfs.counterLockFor(partitionKey);
                    lock.lock();
                    try {
                        QueryFilter filter = QueryFilter.getNamesFilter(key, cfs.metadata.cfName, FBUtilities.singleton(cellName, cfs.metadata.comparator), Long.MIN_VALUE);
                        ColumnFamily cf = cfs.getTopLevelColumns(filter, Integer.MIN_VALUE);
                        if (cf == null) {
                            Pair<CounterCacheKey, ClockAndCount> pair = null;
                            return pair;
                        }
                        Cell cell = cf.getColumn(cellName);
                        if (cell == null || cell.isMarkedForDelete(Long.MIN_VALUE)) {
                            Pair<CounterCacheKey, ClockAndCount> pair = null;
                            return pair;
                        }
                        ClockAndCount clockAndCount = CounterContext.instance().getLocalClockAndCount(cell.value());
                        Pair<CounterCacheKey, ClockAndCount> pair = Pair.create(CounterCacheKey.create(cfs.metadata.cfId, partitionKey, cellName), clockAndCount);
                        return pair;
                    }
                    finally {
                        lock.unlock();
                    }
                }
            });
        }
    }

    public static enum CacheType {
        KEY_CACHE("KeyCache"),
        ROW_CACHE("RowCache"),
        COUNTER_CACHE("CounterCache");

        private final String name;

        private CacheType(String typeName) {
            this.name = typeName;
        }

        public String toString() {
            return this.name;
        }
    }
}

