/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.Checksum;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.ByteBufferOutputStream;
import org.apache.cassandra.io.util.ChecksummedOutputStream;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.PureJavaCrc32;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogSegment {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogSegment.class);
    private static final long idBase = System.currentTimeMillis();
    private static final AtomicInteger nextId = new AtomicInteger(1);
    static final int ENTRY_OVERHEAD_SIZE = 20;
    private final HashMap<UUID, Integer> cfLastWrite = new HashMap();
    public final long id = CommitLogSegment.getNextId();
    private final File logFile;
    private final RandomAccessFile logFileAccessor;
    private boolean needsSync = false;
    private final MappedByteBuffer buffer;
    private final Checksum checksum;
    private final DataOutputStream bufferStream;
    private boolean closed;
    public final CommitLogDescriptor descriptor = new CommitLogDescriptor(this.id);

    public static CommitLogSegment freshSegment() {
        return new CommitLogSegment(null);
    }

    public static long getNextId() {
        return idBase + (long)nextId.getAndIncrement();
    }

    CommitLogSegment(String filePath) {
        this.logFile = new File(DatabaseDescriptor.getCommitLogLocation(), this.descriptor.fileName());
        boolean isCreating = true;
        try {
            File oldFile;
            if (filePath != null && (oldFile = new File(filePath)).exists()) {
                logger.debug("Re-using discarded CommitLog segment for {} from {}", (Object)this.id, (Object)filePath);
                if (!oldFile.renameTo(this.logFile)) {
                    throw new IOException("Rename from " + filePath + " to " + this.id + " failed");
                }
                isCreating = false;
            }
            this.logFileAccessor = new RandomAccessFile(this.logFile, "rw");
            if (isCreating) {
                logger.debug("Creating new commit log segment {}", (Object)this.logFile.getPath());
            }
            this.logFileAccessor.setLength(DatabaseDescriptor.getCommitLogSegmentSize());
            this.buffer = this.logFileAccessor.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, DatabaseDescriptor.getCommitLogSegmentSize());
            this.checksum = new PureJavaCrc32();
            this.bufferStream = new DataOutputStream(new ChecksummedOutputStream(new ByteBufferOutputStream(this.buffer), this.checksum));
            this.buffer.putInt(0);
            this.buffer.position(0);
            this.needsSync = true;
            this.sync();
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.logFile);
        }
    }

    public void discard(boolean deleteFile) {
        this.close();
        if (deleteFile) {
            try {
                FileUtils.deleteWithConfirm(this.logFile);
            }
            catch (FSWriteError e) {
                block8: {
                    File file = this.logFile;
                    try {
                        File newFile = new File(file.getPath() + ".discarded");
                        FileUtils.renameWithConfirm(file, newFile);
                        file = newFile;
                    }
                    catch (Throwable t) {
                        // empty catch block
                    }
                    try {
                        RandomAccessFile raf = new RandomAccessFile(file, "rw");
                        ByteBuffer write = ByteBuffer.allocate(8);
                        write.putInt(0);
                        write.position(0);
                        raf.getChannel().write(write);
                        raf.close();
                        logger.error("{} {}, as we failed to delete it.", (Object)(file == this.logFile ? "Zeroed" : "Renamed and zeroed"), (Object)file);
                    }
                    catch (Throwable t) {
                        if (this.logFile == file) {
                            logger.error("Could not rename or zero {}, which we also failed to delete. In the face of other issues this could result in unnecessary log replay.", (Object)t, (Object)file);
                            break block8;
                        }
                        logger.error("Renamed {} to {}, as we failed to delete it, however we failed to zero its header.", new Object[]{t, this.logFile, file});
                    }
                }
                throw e;
            }
        }
    }

    public CommitLogSegment recycle() {
        this.close();
        return new CommitLogSegment(this.getPath());
    }

    public boolean hasCapacityFor(long size) {
        return size <= (long)this.buffer.remaining();
    }

    private void markDirty(RowMutation rowMutation, ReplayPosition repPos) {
        for (ColumnFamily columnFamily : rowMutation.getColumnFamilies()) {
            CFMetaData cfm = Schema.instance.getCFMetaData(columnFamily.id());
            if (cfm == null) {
                logger.error("Attempted to write commit log entry for unrecognized column family: " + columnFamily.id());
                continue;
            }
            this.markCFDirty(cfm.cfId, repPos.position);
        }
    }

    public ReplayPosition write(RowMutation mutation) throws IOException {
        assert (!this.closed);
        ReplayPosition repPos = this.getContext();
        this.markDirty(mutation, repPos);
        this.checksum.reset();
        int length = (int)RowMutation.serializer.serializedSize(mutation, 7);
        this.bufferStream.writeInt(length);
        this.buffer.putLong(this.checksum.getValue());
        RowMutation.serializer.serialize(mutation, (DataOutput)this.bufferStream, 7);
        this.buffer.putLong(this.checksum.getValue());
        if (this.buffer.remaining() >= 4) {
            this.buffer.putInt(0);
            this.buffer.position(this.buffer.position() - 4);
        }
        this.needsSync = true;
        return repPos;
    }

    public synchronized void sync() {
        if (this.needsSync) {
            try {
                this.buffer.force();
            }
            catch (Exception e) {
                throw new FSWriteError((Throwable)e, this.getPath());
            }
            this.needsSync = false;
        }
    }

    public ReplayPosition getContext() {
        return new ReplayPosition(this.id, this.buffer.position());
    }

    public String getPath() {
        return this.logFile.getPath();
    }

    public String getName() {
        return this.logFile.getName();
    }

    public synchronized void close() {
        if (this.closed) {
            return;
        }
        this.needsSync = false;
        try {
            FileUtils.clean(this.buffer);
            this.logFileAccessor.close();
            this.closed = true;
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.getPath());
        }
    }

    private void markCFDirty(UUID cfId, Integer position) {
        this.cfLastWrite.put(cfId, position);
    }

    public void markClean(UUID cfId, ReplayPosition context) {
        Integer lastWritten = this.cfLastWrite.get(cfId);
        if (!(lastWritten == null || this.contains(context) && lastWritten >= context.position)) {
            this.cfLastWrite.remove(cfId);
        }
    }

    public Collection<UUID> getDirtyCFIDs() {
        return this.cfLastWrite.keySet();
    }

    public boolean isUnused() {
        return this.cfLastWrite.isEmpty();
    }

    public boolean contains(ReplayPosition context) {
        return context.segment == this.id;
    }

    public String dirtyString() {
        StringBuilder sb = new StringBuilder();
        for (UUID cfId : this.cfLastWrite.keySet()) {
            CFMetaData m = Schema.instance.getCFMetaData(cfId);
            sb.append(m == null ? "<deleted>" : m.cfName).append(" (").append(cfId).append("), ");
        }
        return sb.toString();
    }

    public String toString() {
        return "CommitLogSegment(" + this.getPath() + ')';
    }

    public int position() {
        return this.buffer.position();
    }

    public static class CommitLogSegmentFileComparator
    implements Comparator<File> {
        @Override
        public int compare(File f, File f2) {
            CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(f.getName());
            CommitLogDescriptor desc2 = CommitLogDescriptor.fromFileName(f2.getName());
            return (int)(desc.id - desc2.id);
        }
    }
}

