/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.streaming;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SocketChannel;
import java.nio.channels.WritableByteChannel;
import java.util.Collection;
import java.util.Comparator;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.net.OutboundTcpConnectionPool;
import org.apache.cassandra.streaming.StreamSession;
import org.apache.cassandra.streaming.messages.StreamInitMessage;
import org.apache.cassandra.streaming.messages.StreamMessage;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionHandler {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionHandler.class);
    private static final int MAX_CONNECT_ATTEMPTS = 3;
    private final StreamSession session;
    private IncomingMessageHandler incoming;
    private OutgoingMessageHandler outgoing;

    ConnectionHandler(StreamSession session) {
        this.session = session;
    }

    public void initiate() throws IOException {
        logger.debug("[Stream #{}] Sending stream init for incoming stream", (Object)this.session.planId());
        Socket incomingSocket = ConnectionHandler.connect(this.session.peer);
        this.incoming = new IncomingMessageHandler(this.session, incomingSocket, 1);
        this.incoming.sendInitMessage(true);
        this.incoming.start();
        logger.debug("[Stream #{}] Sending stream init for outgoing stream", (Object)this.session.planId());
        Socket outgoingSocket = ConnectionHandler.connect(this.session.peer);
        this.outgoing = new OutgoingMessageHandler(this.session, outgoingSocket, 1);
        this.outgoing.sendInitMessage(false);
        this.outgoing.start();
    }

    public void initiateOnReceivingSide(Socket socket, boolean isForOutgoing, int version) throws IOException {
        if (isForOutgoing) {
            this.outgoing = new OutgoingMessageHandler(this.session, socket, version);
            this.outgoing.start();
        } else {
            this.incoming = new IncomingMessageHandler(this.session, socket, version);
            this.incoming.start();
        }
    }

    private static Socket connect(InetAddress peer) throws IOException {
        int attempts = 0;
        while (true) {
            try {
                Socket socket = OutboundTcpConnectionPool.newSocket(peer);
                socket.setSoTimeout(DatabaseDescriptor.getStreamingSocketTimeout());
                return socket;
            }
            catch (IOException e) {
                if (++attempts >= 3) {
                    throw e;
                }
                long waitms = DatabaseDescriptor.getRpcTimeout() * (long)Math.pow(2.0, attempts);
                logger.warn("Failed attempt " + attempts + " to connect to " + peer + ". Retrying in " + waitms + " ms. (" + e + ")");
                try {
                    Thread.sleep(waitms);
                }
                catch (InterruptedException wtf) {
                    throw new IOException("interrupted", wtf);
                }
            }
        }
    }

    public ListenableFuture<?> close() {
        logger.debug("[Stream #{}] Closing stream connection handler on {}", (Object)this.session.planId(), (Object)this.session.peer);
        ListenableFuture<?> inClosed = this.incoming == null ? Futures.immediateFuture(null) : this.incoming.close();
        ListenableFuture<?> outClosed = this.outgoing == null ? Futures.immediateFuture(null) : this.outgoing.close();
        return Futures.allAsList((ListenableFuture[])new ListenableFuture[]{inClosed, outClosed});
    }

    public void sendMessages(Collection<? extends StreamMessage> messages) {
        for (StreamMessage streamMessage : messages) {
            this.sendMessage(streamMessage);
        }
    }

    public void sendMessage(StreamMessage message) {
        if (this.outgoing.isClosed()) {
            throw new RuntimeException("Outgoing stream handler has been closed");
        }
        this.outgoing.enqueue(message);
    }

    public boolean isOutgoingConnected() {
        return this.outgoing != null && !this.outgoing.isClosed();
    }

    static class OutgoingMessageHandler
    extends MessageHandler {
        private final PriorityBlockingQueue<StreamMessage> messageQueue = new PriorityBlockingQueue<StreamMessage>(64, new Comparator<StreamMessage>(){

            @Override
            public int compare(StreamMessage o1, StreamMessage o2) {
                return o2.getPriority() - o1.getPriority();
            }
        });
        private final WritableByteChannel out = this.getWriteChannel();

        OutgoingMessageHandler(StreamSession session, Socket socket, int protocolVersion) throws IOException {
            super(session, socket, protocolVersion);
        }

        @Override
        protected String name() {
            return "STREAM-OUT";
        }

        public void enqueue(StreamMessage message) {
            this.messageQueue.put(message);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            StreamMessage next;
            while (!this.isClosed()) {
                try {
                    next = this.messageQueue.poll(1L, TimeUnit.SECONDS);
                    if (next == null) continue;
                    logger.debug("[Stream #{}] Sending {}", (Object)this.session.planId(), (Object)next);
                    this.sendMessage(next);
                    if (next.type != StreamMessage.Type.SESSION_FAILED) continue;
                    this.close();
                }
                catch (InterruptedException e) {
                    throw new AssertionError((Object)e);
                }
            }
            try {
                while ((next = this.messageQueue.poll()) != null) {
                    this.sendMessage(next);
                }
            }
            finally {
                this.signalCloseDone();
            }
        }

        private void sendMessage(StreamMessage message) {
            try {
                StreamMessage.serialize(message, this.out, this.protocolVersion, this.session);
            }
            catch (SocketException e) {
                this.session.onError(e);
                this.close();
            }
            catch (IOException e) {
                this.session.onError(e);
            }
        }
    }

    static class IncomingMessageHandler
    extends MessageHandler {
        private final ReadableByteChannel in = this.getReadChannel();

        IncomingMessageHandler(StreamSession session, Socket socket, int protocolVersion) throws IOException {
            super(session, socket, protocolVersion);
        }

        @Override
        protected String name() {
            return "STREAM-IN";
        }

        @Override
        public void run() {
            while (!this.isClosed()) {
                try {
                    StreamMessage message = StreamMessage.deserialize(this.in, this.protocolVersion, this.session);
                    if (message == null) continue;
                    logger.debug("[Stream #{}] Received {}", (Object)this.session.planId(), (Object)message);
                    this.session.messageReceived(message);
                }
                catch (SocketException e) {
                    this.close();
                }
                catch (Throwable e) {
                    this.session.onError(e);
                }
            }
            this.signalCloseDone();
        }
    }

    static abstract class MessageHandler
    implements Runnable {
        protected final StreamSession session;
        protected final Socket socket;
        protected final int protocolVersion;
        private final AtomicReference<SettableFuture<?>> closeFuture = new AtomicReference();

        protected MessageHandler(StreamSession session, Socket socket, int protocolVersion) {
            this.session = session;
            this.socket = socket;
            this.protocolVersion = protocolVersion;
        }

        protected abstract String name();

        protected WritableByteChannel getWriteChannel() throws IOException {
            SocketChannel out = this.socket.getChannel();
            return out == null ? Channels.newChannel(this.socket.getOutputStream()) : out;
        }

        protected ReadableByteChannel getReadChannel() throws IOException {
            SocketChannel in = this.socket.getChannel();
            return in == null ? Channels.newChannel(this.socket.getInputStream()) : in;
        }

        public void sendInitMessage(boolean isForOutgoing) throws IOException {
            StreamInitMessage message = new StreamInitMessage(FBUtilities.getBroadcastAddress(), this.session.planId(), this.session.description(), isForOutgoing);
            this.getWriteChannel().write(message.createMessage(false, this.protocolVersion));
        }

        public void start() {
            new Thread((Runnable)this, this.name() + "-" + this.session.peer).start();
        }

        public ListenableFuture<?> close() {
            SettableFuture<?> future = SettableFuture.create();
            return this.closeFuture.compareAndSet(null, future) ? future : this.closeFuture.get();
        }

        public boolean isClosed() {
            return this.closeFuture.get() != null;
        }

        protected void signalCloseDone() {
            this.closeFuture.get().set(null);
            try {
                this.socket.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }
}

