/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Longs;
import com.google.common.util.concurrent.RateLimiter;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.cache.InstrumentingCache;
import org.apache.cassandra.cache.KeyCacheKey;
import org.apache.cassandra.concurrent.DebuggableThreadPoolExecutor;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.DataRange;
import org.apache.cassandra.db.DataTracker;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.columniterator.OnDiskAtomIterator;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.db.compaction.ICompactionScanner;
import org.apache.cassandra.db.index.SecondaryIndex;
import org.apache.cassandra.dht.AbstractBounds;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.LocalPartitioner;
import org.apache.cassandra.dht.LocalToken;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.compress.CompressedRandomAccessReader;
import org.apache.cassandra.io.compress.CompressedThrottledReader;
import org.apache.cassandra.io.compress.CompressionMetadata;
import org.apache.cassandra.io.sstable.BloomFilterTracker;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.IndexSummary;
import org.apache.cassandra.io.sstable.IndexSummaryBuilder;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableDeletingTask;
import org.apache.cassandra.io.sstable.SSTableMetadata;
import org.apache.cassandra.io.sstable.SSTableScanner;
import org.apache.cassandra.io.util.BufferedSegmentedFile;
import org.apache.cassandra.io.util.CompressedSegmentedFile;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.ICompressedFile;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.io.util.SegmentedFile;
import org.apache.cassandra.io.util.ThrottledReader;
import org.apache.cassandra.metrics.RestorableMeter;
import org.apache.cassandra.service.CacheService;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.CLibrary;
import org.apache.cassandra.utils.EstimatedHistogram;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.FilterFactory;
import org.apache.cassandra.utils.IFilter;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SSTableReader
extends SSTable
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(SSTableReader.class);
    private static final ScheduledThreadPoolExecutor syncExecutor = new ScheduledThreadPoolExecutor(1);
    private static final RateLimiter meterSyncThrottle;
    public final long maxDataAge;
    private SegmentedFile ifile;
    private SegmentedFile dfile;
    private IndexSummary indexSummary;
    private IFilter bf;
    private InstrumentingCache<KeyCacheKey, RowIndexEntry> keyCache;
    private final BloomFilterTracker bloomFilterTracker = new BloomFilterTracker();
    private final AtomicInteger references = new AtomicInteger(1);
    private final AtomicBoolean isCompacted = new AtomicBoolean(false);
    private final AtomicBoolean isSuspect = new AtomicBoolean(false);
    private final SSTableDeletingTask deletingTask;
    private volatile SSTableMetadata sstableMetadata;
    private final AtomicLong keyCacheHit = new AtomicLong(0L);
    private final AtomicLong keyCacheRequest = new AtomicLong(0L);
    @VisibleForTesting
    public RestorableMeter readMeter;
    private final ScheduledFuture readMeterSyncFuture;

    public static long getApproximateKeyCount(Iterable<SSTableReader> sstables, CFMetaData metadata) {
        long count = 0L;
        for (SSTableReader sstable : sstables) {
            int indexKeyCount = sstable.getKeySampleSize();
            count += (long)((indexKeyCount + 1) * sstable.indexSummary.getIndexInterval());
            if (!logger.isDebugEnabled()) continue;
            logger.debug("index size for bloom filter calc for file  : " + sstable.getFilename() + "   : " + count);
        }
        return count;
    }

    public static SSTableReader open(Descriptor descriptor) throws IOException {
        CFMetaData metadata;
        if (descriptor.cfname.contains(".")) {
            int i = descriptor.cfname.indexOf(".");
            String parentName = descriptor.cfname.substring(0, i);
            CFMetaData parent = Schema.instance.getCFMetaData(descriptor.ksname, parentName);
            ColumnDefinition def = parent.getColumnDefinitionForIndex(descriptor.cfname.substring(i + 1));
            metadata = CFMetaData.newIndexMetadata(parent, def, SecondaryIndex.getIndexComparator(parent, def));
        } else {
            metadata = Schema.instance.getCFMetaData(descriptor.ksname, descriptor.cfname);
        }
        return SSTableReader.open(descriptor, metadata);
    }

    public static SSTableReader open(Descriptor desc, CFMetaData metadata) throws IOException {
        IPartitioner<LocalToken> p = desc.cfname.contains(".") ? new LocalPartitioner(metadata.getKeyValidator()) : StorageService.getPartitioner();
        return SSTableReader.open(desc, SSTableReader.componentsFor(desc), metadata, p);
    }

    public static SSTableReader openNoValidation(Descriptor descriptor, Set<Component> components, CFMetaData metadata) throws IOException {
        return SSTableReader.open(descriptor, components, metadata, StorageService.getPartitioner(), false, false);
    }

    public static SSTableReader openForBatch(Descriptor descriptor, Set<Component> components, CFMetaData metadata, IPartitioner partitioner) throws IOException {
        SegmentedFile.Builder dbuilder;
        SSTableMetadata sstableMetadata = SSTableReader.openMetadata(descriptor, components, partitioner, true);
        SSTableReader sstable = new SSTableReader(descriptor, components, metadata, partitioner, System.currentTimeMillis(), sstableMetadata, false);
        BufferedSegmentedFile.Builder ibuilder = new BufferedSegmentedFile.Builder();
        SegmentedFile.Builder builder = dbuilder = sstable.compression ? new CompressedSegmentedFile.Builder() : new BufferedSegmentedFile.Builder();
        if (!SSTableReader.loadSummary(sstable, ibuilder, dbuilder, sstable.metadata)) {
            sstable.buildSummary(false, ibuilder, dbuilder, false);
        }
        sstable.ifile = ((SegmentedFile.Builder)ibuilder).complete(sstable.descriptor.filenameFor(Component.PRIMARY_INDEX));
        sstable.dfile = dbuilder.complete(sstable.descriptor.filenameFor(Component.DATA));
        sstable.bf = FilterFactory.AlwaysPresent;
        return sstable;
    }

    public static SSTableReader open(Descriptor descriptor, Set<Component> components, CFMetaData metadata, IPartitioner partitioner) throws IOException {
        boolean trackHotness = !"system".equals(descriptor.ksname) && !Config.isClientMode();
        return SSTableReader.open(descriptor, components, metadata, partitioner, true, trackHotness);
    }

    public static SSTableReader open(Descriptor descriptor, Set<Component> components, CFMetaData metadata, IPartitioner partitioner, boolean validate, boolean trackHotness) throws IOException {
        long start = System.nanoTime();
        SSTableMetadata sstableMetadata = SSTableReader.openMetadata(descriptor, components, partitioner, validate);
        SSTableReader sstable = new SSTableReader(descriptor, components, metadata, partitioner, System.currentTimeMillis(), sstableMetadata, trackHotness);
        sstable.load();
        if (validate) {
            sstable.validate();
        }
        logger.debug("INDEX LOAD TIME for {}: {} ms.", (Object)descriptor, (Object)TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - start));
        if (sstable.getKeyCache() != null) {
            logger.debug("key cache contains {}/{} keys", (Object)sstable.getKeyCache().size(), (Object)sstable.getKeyCache().getCapacity());
        }
        return sstable;
    }

    private static SSTableMetadata openMetadata(Descriptor descriptor, Set<Component> components, IPartitioner partitioner, boolean primaryIndexRequired) throws IOException {
        assert (partitioner != null);
        assert (components.contains(Component.DATA)) : "Data component is missing for sstable" + descriptor;
        assert (!primaryIndexRequired || components.contains(Component.PRIMARY_INDEX)) : "Primary index component is missing for sstable " + descriptor;
        logger.info("Opening {} ({} bytes)", (Object)descriptor, (Object)new File(descriptor.filenameFor(COMPONENT_DATA)).length());
        SSTableMetadata sstableMetadata = (SSTableMetadata)SSTableMetadata.serializer.deserialize((Descriptor)descriptor).left;
        String partitionerName = partitioner.getClass().getCanonicalName();
        if (sstableMetadata.partitioner != null && !partitionerName.equals(sstableMetadata.partitioner)) {
            logger.error(String.format("Cannot open %s; partitioner %s does not match system partitioner %s.  Note that the default partitioner starting with Cassandra 1.2 is Murmur3Partitioner, so you will need to edit that to match your old partitioner if upgrading.", descriptor, sstableMetadata.partitioner, partitionerName));
            System.exit(1);
        }
        return sstableMetadata;
    }

    public static void logOpenException(Descriptor descriptor, IOException e) {
        if (e instanceof FileNotFoundException) {
            logger.error("Missing sstable component in " + descriptor + "; skipped because of " + e.getMessage());
        } else {
            logger.error("Corrupt sstable " + descriptor + "; skipped", (Throwable)e);
        }
    }

    public static Collection<SSTableReader> openAll(Set<Map.Entry<Descriptor, Set<Component>>> entries, final CFMetaData metadata, final IPartitioner partitioner) {
        final LinkedBlockingQueue<SSTableReader> sstables = new LinkedBlockingQueue<SSTableReader>();
        DebuggableThreadPoolExecutor executor = DebuggableThreadPoolExecutor.createWithFixedPoolSize("SSTableBatchOpen", FBUtilities.getAvailableProcessors());
        for (final Map.Entry<Descriptor, Set<Component>> entry : entries) {
            Runnable runnable = new Runnable(){

                @Override
                public void run() {
                    SSTableReader sstable;
                    try {
                        sstable = SSTableReader.open((Descriptor)entry.getKey(), (Set)entry.getValue(), metadata, partitioner);
                    }
                    catch (IOException ex) {
                        logger.error("Corrupt sstable " + entry + "; skipped", (Throwable)ex);
                        return;
                    }
                    sstables.add(sstable);
                }
            };
            executor.submit(runnable);
        }
        executor.shutdown();
        try {
            executor.awaitTermination(7L, TimeUnit.DAYS);
        }
        catch (InterruptedException e) {
            throw new AssertionError((Object)e);
        }
        return sstables;
    }

    static SSTableReader internalOpen(Descriptor desc, Set<Component> components, CFMetaData metadata, IPartitioner partitioner, SegmentedFile ifile, SegmentedFile dfile, IndexSummary isummary, IFilter bf, long maxDataAge, SSTableMetadata sstableMetadata) {
        assert (desc != null && partitioner != null && ifile != null && dfile != null && isummary != null && bf != null && sstableMetadata != null);
        boolean trackHotness = !"system".equals(desc.ksname) && !Config.isClientMode();
        return new SSTableReader(desc, components, metadata, partitioner, ifile, dfile, isummary, bf, maxDataAge, sstableMetadata, trackHotness);
    }

    private SSTableReader(final Descriptor desc, Set<Component> components, CFMetaData metadata, IPartitioner partitioner, long maxDataAge, SSTableMetadata sstableMetadata, boolean trackHotness) {
        super(desc, components, metadata, partitioner);
        this.sstableMetadata = sstableMetadata;
        this.maxDataAge = maxDataAge;
        this.deletingTask = new SSTableDeletingTask(this);
        if (trackHotness) {
            this.readMeter = SystemKeyspace.getSSTableReadMeter(desc.ksname, desc.cfname, desc.generation);
            this.readMeterSyncFuture = syncExecutor.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    if (!SSTableReader.this.isCompacted.get()) {
                        meterSyncThrottle.acquire();
                        SystemKeyspace.persistSSTableReadMeter(desc.ksname, desc.cfname, desc.generation, SSTableReader.this.readMeter);
                    }
                }
            }, 1L, 5L, TimeUnit.MINUTES);
        } else {
            this.readMeter = null;
            this.readMeterSyncFuture = null;
        }
    }

    private SSTableReader(Descriptor desc, Set<Component> components, CFMetaData metadata, IPartitioner partitioner, SegmentedFile ifile, SegmentedFile dfile, IndexSummary indexSummary, IFilter bloomFilter, long maxDataAge, SSTableMetadata sstableMetadata, boolean trackHotness) {
        this(desc, components, metadata, partitioner, maxDataAge, sstableMetadata, trackHotness);
        this.ifile = ifile;
        this.dfile = dfile;
        this.indexSummary = indexSummary;
        this.bf = bloomFilter;
    }

    @Override
    public void close() throws IOException {
        if (this.readMeterSyncFuture != null) {
            this.readMeterSyncFuture.cancel(true);
        }
        if (null != this.ifile) {
            this.ifile.cleanup();
        }
        this.dfile.cleanup();
        if (null != this.bf) {
            this.bf.close();
        }
        if (null != this.indexSummary) {
            this.indexSummary.close();
        }
    }

    public void setTrackedBy(DataTracker tracker) {
        this.deletingTask.setTracker(tracker);
        this.keyCache = CacheService.instance.keyCache;
    }

    private void load() throws IOException {
        if (this.metadata.getBloomFilterFpChance() == 1.0) {
            this.load(false, true);
            this.bf = FilterFactory.AlwaysPresent;
        } else if (!this.components.contains(Component.PRIMARY_INDEX)) {
            this.load(false, false);
        } else if (!this.components.contains(Component.FILTER)) {
            this.load(true, true);
        } else if (this.descriptor.version.hasBloomFilterFPChance && this.sstableMetadata.bloomFilterFPChance != this.metadata.getBloomFilterFpChance()) {
            this.load(true, true);
        } else {
            this.load(false, true);
            this.loadBloomFilter();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void loadBloomFilter() throws IOException {
        DataInputStream stream = null;
        try {
            stream = new DataInputStream(new BufferedInputStream(new FileInputStream(this.descriptor.filenameFor(Component.FILTER))));
            this.bf = FilterFactory.deserialize(stream, true);
        }
        catch (Throwable throwable) {
            FileUtils.closeQuietly(stream);
            throw throwable;
        }
        FileUtils.closeQuietly(stream);
    }

    private void load(boolean recreateBloomFilter, boolean saveSummaryIfCreated) throws IOException {
        SegmentedFile.Builder ibuilder = SegmentedFile.getBuilder(DatabaseDescriptor.getIndexAccessMode());
        SegmentedFile.Builder dbuilder = this.compression ? SegmentedFile.getCompressedBuilder() : SegmentedFile.getBuilder(DatabaseDescriptor.getDiskAccessMode());
        boolean summaryLoaded = SSTableReader.loadSummary(this, ibuilder, dbuilder, this.metadata);
        if (recreateBloomFilter || !summaryLoaded) {
            this.buildSummary(recreateBloomFilter, ibuilder, dbuilder, summaryLoaded);
        }
        if (this.components.contains(Component.PRIMARY_INDEX)) {
            this.ifile = ibuilder.complete(this.descriptor.filenameFor(Component.PRIMARY_INDEX));
        }
        this.dfile = dbuilder.complete(this.descriptor.filenameFor(Component.DATA));
        if (saveSummaryIfCreated && (recreateBloomFilter || !summaryLoaded)) {
            SSTableReader.saveSummary(this, ibuilder, dbuilder);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void buildSummary(boolean recreateBloomFilter, SegmentedFile.Builder ibuilder, SegmentedFile.Builder dbuilder, boolean summaryLoaded) throws IOException {
        if (!this.components.contains(Component.PRIMARY_INDEX)) {
            return;
        }
        RandomAccessReader primaryIndex = RandomAccessReader.open(new File(this.descriptor.filenameFor(Component.PRIMARY_INDEX)));
        try {
            long indexPosition;
            long estimatedKeys;
            long indexSize = primaryIndex.length();
            long histogramCount = this.sstableMetadata.estimatedRowSize.count();
            long l = estimatedKeys = histogramCount > 0L && !this.sstableMetadata.estimatedRowSize.isOverflowed() ? histogramCount : this.estimateRowsFromIndex(primaryIndex);
            if (recreateBloomFilter) {
                this.bf = FilterFactory.getFilter(estimatedKeys, this.metadata.getBloomFilterFpChance(), true);
            }
            IndexSummaryBuilder summaryBuilder = null;
            if (!summaryLoaded) {
                summaryBuilder = new IndexSummaryBuilder(estimatedKeys, this.metadata.getIndexInterval());
            }
            while ((indexPosition = primaryIndex.getFilePointer()) != indexSize) {
                ByteBuffer key = ByteBufferUtil.readWithShortLength(primaryIndex);
                RowIndexEntry indexEntry = RowIndexEntry.serializer.deserialize(primaryIndex, this.descriptor.version);
                DecoratedKey decoratedKey = this.partitioner.decorateKey(key);
                if (this.first == null) {
                    this.first = decoratedKey;
                }
                this.last = decoratedKey;
                if (recreateBloomFilter) {
                    this.bf.add(decoratedKey.key);
                }
                if (summaryLoaded) continue;
                summaryBuilder.maybeAddEntry(decoratedKey, indexPosition);
                ibuilder.addPotentialBoundary(indexPosition);
                dbuilder.addPotentialBoundary(indexEntry.position);
            }
            if (!summaryLoaded) {
                this.indexSummary = summaryBuilder.build(this.partitioner);
            }
        }
        finally {
            FileUtils.closeQuietly(primaryIndex);
        }
        this.first = SSTableReader.getMinimalKey(this.first);
        this.last = SSTableReader.getMinimalKey(this.last);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean loadSummary(SSTableReader reader, SegmentedFile.Builder ibuilder, SegmentedFile.Builder dbuilder, CFMetaData metadata) {
        DataInputStream iStream;
        File summariesFile;
        block6: {
            summariesFile = new File(reader.descriptor.filenameFor(Component.SUMMARY));
            if (!reader.descriptor.version.offHeapSummaries || !summariesFile.exists()) {
                return false;
            }
            iStream = null;
            iStream = new DataInputStream(new FileInputStream(summariesFile));
            reader.indexSummary = IndexSummary.serializer.deserialize(iStream, reader.partitioner);
            if (reader.indexSummary.getIndexInterval() == metadata.getIndexInterval()) break block6;
            iStream.close();
            logger.debug("Cannot read the saved summary for {} because Index Interval changed from {} to {}.", new Object[]{reader.toString(), reader.indexSummary.getIndexInterval(), metadata.getIndexInterval()});
            FileUtils.deleteWithConfirm(summariesFile);
            boolean bl = false;
            FileUtils.closeQuietly(iStream);
            return bl;
        }
        try {
            reader.first = reader.partitioner.decorateKey(ByteBufferUtil.readWithLength(iStream));
            reader.last = reader.partitioner.decorateKey(ByteBufferUtil.readWithLength(iStream));
            ibuilder.deserializeBounds(iStream);
            dbuilder.deserializeBounds(iStream);
        }
        catch (IOException e) {
            boolean bl;
            try {
                logger.debug("Cannot deserialize SSTable Summary: ", (Throwable)e);
                FileUtils.closeQuietly(iStream);
                FileUtils.deleteWithConfirm(summariesFile);
                bl = false;
            }
            catch (Throwable throwable) {
                FileUtils.closeQuietly(iStream);
                throw throwable;
            }
            FileUtils.closeQuietly(iStream);
            return bl;
        }
        FileUtils.closeQuietly(iStream);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void saveSummary(SSTableReader reader, SegmentedFile.Builder ibuilder, SegmentedFile.Builder dbuilder) {
        File summariesFile = new File(reader.descriptor.filenameFor(Component.SUMMARY));
        if (summariesFile.exists()) {
            summariesFile.delete();
        }
        DataOutputStream oStream = null;
        try {
            oStream = new DataOutputStream(new FileOutputStream(summariesFile));
            IndexSummary.serializer.serialize(reader.indexSummary, oStream);
            ByteBufferUtil.writeWithLength(reader.first.key, (DataOutput)oStream);
            ByteBufferUtil.writeWithLength(reader.last.key, (DataOutput)oStream);
            ibuilder.serializeBounds(oStream);
            dbuilder.serializeBounds(oStream);
        }
        catch (IOException e) {
            block5: {
                try {
                    logger.debug("Cannot save SSTable Summary: ", (Throwable)e);
                    if (!summariesFile.exists()) break block5;
                    summariesFile.delete();
                }
                catch (Throwable throwable) {
                    FileUtils.closeQuietly(oStream);
                    throw throwable;
                }
            }
            FileUtils.closeQuietly(oStream);
        }
        FileUtils.closeQuietly(oStream);
    }

    public void releaseSummary() throws IOException {
        this.indexSummary.close();
        this.indexSummary = null;
    }

    private void validate() {
        if (this.first.compareTo(this.last) > 0) {
            this.releaseReference();
            throw new IllegalStateException(String.format("SSTable first key %s > last key %s", this.first, this.last));
        }
    }

    public long getIndexScanPosition(RowPosition key) {
        int index = this.indexSummary.binarySearch(key);
        if (index < 0) {
            int greaterThan = (index + 1) * -1;
            if (greaterThan == 0) {
                return -1L;
            }
            return this.indexSummary.getPosition(greaterThan - 1);
        }
        return this.indexSummary.getPosition(index);
    }

    public CompressionMetadata getCompressionMetadata() {
        if (!this.compression) {
            throw new IllegalStateException(this + " is not compressed");
        }
        CompressionMetadata cmd = ((ICompressedFile)((Object)this.dfile)).getMetadata();
        String cfName = this.metadata.isSecondaryIndex() ? this.metadata.getParentColumnFamilyName() : this.metadata.cfName;
        cmd.parameters.setLiveMetadata(Schema.instance.getCFMetaData(this.metadata.ksName, cfName));
        return cmd;
    }

    public long getCompressionMetadataOffHeapSize() {
        if (!this.compression) {
            return 0L;
        }
        return this.getCompressionMetadata().offHeapSize();
    }

    public void forceFilterFailures() {
        this.bf = FilterFactory.AlwaysPresent;
    }

    public IFilter getBloomFilter() {
        return this.bf;
    }

    public long getBloomFilterSerializedSize() {
        return this.bf.serializedSize();
    }

    public long getBloomFilterOffHeapSize() {
        return this.bf.offHeapSize();
    }

    public long getIndexSummaryOffHeapSize() {
        return this.indexSummary.offHeapSize();
    }

    public long estimatedKeys() {
        return (long)this.indexSummary.size() * (long)this.indexSummary.getIndexInterval();
    }

    public long estimatedKeysForRanges(Collection<Range<Token>> ranges) {
        long sampleKeyCount = 0L;
        List<Pair<Integer, Integer>> sampleIndexes = SSTableReader.getSampleIndexesForRanges(this.indexSummary, ranges);
        for (Pair<Integer, Integer> sampleIndexRange : sampleIndexes) {
            sampleKeyCount += (long)((Integer)sampleIndexRange.right - (Integer)sampleIndexRange.left + 1);
        }
        return Math.max(1L, sampleKeyCount * (long)this.indexSummary.getIndexInterval());
    }

    public int getKeySampleSize() {
        return this.indexSummary.size();
    }

    public byte[] getKeySample(int position) {
        return this.indexSummary.getKey(position);
    }

    private static List<Pair<Integer, Integer>> getSampleIndexesForRanges(IndexSummary summary, Collection<Range<Token>> ranges) {
        ArrayList<Pair<Integer, Integer>> positions = new ArrayList<Pair<Integer, Integer>>();
        for (Range range : Range.normalize(ranges)) {
            int right;
            Token.KeyBound leftPosition = ((Token)range.left).maxKeyBound();
            Token.KeyBound rightPosition = ((Token)range.right).maxKeyBound();
            int left = summary.binarySearch(leftPosition);
            left = left < 0 ? (left + 1) * -1 : ++left;
            if (left == summary.size()) continue;
            int n = right = Range.isWrapAround(range.left, range.right) ? summary.size() - 1 : summary.binarySearch(rightPosition);
            if (right < 0) {
                if ((right = (right + 1) * -1) == 0) continue;
                --right;
            }
            if (left > right) continue;
            positions.add(Pair.create(left, right));
        }
        return positions;
    }

    public Iterable<DecoratedKey> getKeySamples(Range<Token> range) {
        final List<Pair<Integer, Integer>> indexRanges = SSTableReader.getSampleIndexesForRanges(this.indexSummary, Collections.singletonList(range));
        if (indexRanges.isEmpty()) {
            return Collections.emptyList();
        }
        return new Iterable<DecoratedKey>(){

            @Override
            public Iterator<DecoratedKey> iterator() {
                return new Iterator<DecoratedKey>(){
                    private Iterator<Pair<Integer, Integer>> rangeIter;
                    private Pair<Integer, Integer> current;
                    private int idx;
                    {
                        this.rangeIter = indexRanges.iterator();
                    }

                    @Override
                    public boolean hasNext() {
                        if (this.current == null || this.idx > (Integer)this.current.right) {
                            if (this.rangeIter.hasNext()) {
                                this.current = this.rangeIter.next();
                                this.idx = (Integer)this.current.left;
                                return true;
                            }
                            return false;
                        }
                        return true;
                    }

                    @Override
                    public DecoratedKey next() {
                        byte[] bytes = SSTableReader.this.indexSummary.getKey(this.idx++);
                        return SSTableReader.this.partitioner.decorateKey(ByteBuffer.wrap(bytes));
                    }

                    @Override
                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        };
    }

    public List<Pair<Long, Long>> getPositionsForRanges(Collection<Range<Token>> ranges) {
        ArrayList<Pair<Long, Long>> positions = new ArrayList<Pair<Long, Long>>();
        for (Range range : Range.normalize(ranges)) {
            long right;
            AbstractBounds<RowPosition> keyRange = range.toRowBounds();
            RowIndexEntry idxLeft = this.getPosition((RowPosition)keyRange.left, Operator.GT);
            long left = idxLeft == null ? -1L : idxLeft.position;
            if (left == -1L) continue;
            RowIndexEntry idxRight = this.getPosition((RowPosition)keyRange.right, Operator.GT);
            long l = right = idxRight == null ? -1L : idxRight.position;
            if (right == -1L || Range.isWrapAround(range.left, range.right)) {
                right = this.uncompressedLength();
            }
            if (left == right) continue;
            positions.add(Pair.create(left, right));
        }
        return positions;
    }

    public void cacheKey(DecoratedKey key, RowIndexEntry info) {
        CFMetaData.Caching caching = this.metadata.getCaching();
        if (caching == CFMetaData.Caching.NONE || caching == CFMetaData.Caching.ROWS_ONLY || this.keyCache == null || this.keyCache.getCapacity() == 0L) {
            return;
        }
        KeyCacheKey cacheKey = new KeyCacheKey(this.descriptor, key.key);
        logger.trace("Adding cache entry for {} -> {}", (Object)cacheKey, (Object)info);
        this.keyCache.put(cacheKey, info);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void preheat(Map<DecoratedKey, RowIndexEntry> cachedKeys) throws IOException {
        RandomAccessFile f = new RandomAccessFile(this.getFilename(), "r");
        try {
            int fd = CLibrary.getfd(f.getFD());
            for (Map.Entry<DecoratedKey, RowIndexEntry> entry : cachedKeys.entrySet()) {
                this.cacheKey(entry.getKey(), entry.getValue());
                if (!DatabaseDescriptor.shouldPreheatPageCache() || fd <= 0) continue;
                CLibrary.preheatPage(fd, entry.getValue().position);
            }
        }
        finally {
            FileUtils.closeQuietly(f);
        }
    }

    public RowIndexEntry getCachedPosition(DecoratedKey key, boolean updateStats) {
        return this.getCachedPosition(new KeyCacheKey(this.descriptor, key.key), updateStats);
    }

    private RowIndexEntry getCachedPosition(KeyCacheKey unifiedKey, boolean updateStats) {
        if (this.keyCache != null && this.keyCache.getCapacity() > 0L) {
            if (updateStats) {
                RowIndexEntry cachedEntry = this.keyCache.get(unifiedKey);
                this.keyCacheRequest.incrementAndGet();
                if (cachedEntry != null) {
                    this.keyCacheHit.incrementAndGet();
                    this.bloomFilterTracker.addTruePositive();
                }
                return cachedEntry;
            }
            return this.keyCache.getInternal(unifiedKey);
        }
        return null;
    }

    public RowIndexEntry getPosition(RowPosition key, Operator op) {
        return this.getPosition(key, op, true);
    }

    public RowIndexEntry getPosition(RowPosition key, Operator op, boolean updateCacheAndStats) {
        long sampledPosition;
        if (op == Operator.EQ) {
            assert (key instanceof DecoratedKey);
            if (!this.bf.isPresent(((DecoratedKey)key).key)) {
                Tracing.trace("Bloom filter allows skipping sstable {}", this.descriptor.generation);
                return null;
            }
        }
        if ((op == Operator.EQ || op == Operator.GE) && key instanceof DecoratedKey) {
            DecoratedKey decoratedKey = (DecoratedKey)key;
            KeyCacheKey cacheKey = new KeyCacheKey(this.descriptor, decoratedKey.key);
            RowIndexEntry cachedPosition = this.getCachedPosition(cacheKey, updateCacheAndStats);
            if (cachedPosition != null) {
                Tracing.trace("Key cache hit for sstable {}", this.descriptor.generation);
                return cachedPosition;
            }
        }
        if ((sampledPosition = this.getIndexScanPosition(key)) == -1L) {
            if (op == Operator.EQ && updateCacheAndStats) {
                this.bloomFilterTracker.addFalsePositive();
            }
            if (op.apply(1) >= 0) {
                sampledPosition = 0L;
            } else {
                Tracing.trace("Partition summary allows skipping sstable {}", this.descriptor.generation);
                return null;
            }
        }
        if (this.ifile == null) {
            return null;
        }
        int i = 0;
        Iterator<FileDataInput> segments = this.ifile.iterator(sampledPosition);
        while (segments.hasNext() && i <= this.indexSummary.getIndexInterval()) {
            FileDataInput in = segments.next();
            try {
                while (!in.isEOF() && i <= this.indexSummary.getIndexInterval()) {
                    boolean opSatisfied;
                    boolean exactMatch;
                    ++i;
                    ByteBuffer indexKey = ByteBufferUtil.readWithShortLength(in);
                    if (op == Operator.EQ) {
                        opSatisfied = exactMatch = indexKey.equals(((DecoratedKey)key).key);
                    } else {
                        DecoratedKey indexDecoratedKey = this.partitioner.decorateKey(indexKey);
                        int comparison = indexDecoratedKey.compareTo(key);
                        int v = op.apply(comparison);
                        opSatisfied = v == 0;
                        boolean bl = exactMatch = comparison == 0;
                        if (v < 0) {
                            Tracing.trace("Partition index lookup allows skipping sstable {}", this.descriptor.generation);
                            RowIndexEntry rowIndexEntry = null;
                            return rowIndexEntry;
                        }
                    }
                    if (opSatisfied) {
                        RowIndexEntry indexEntry = RowIndexEntry.serializer.deserialize(in, this.descriptor.version);
                        if (exactMatch && updateCacheAndStats) {
                            assert (key instanceof DecoratedKey);
                            DecoratedKey decoratedKey = (DecoratedKey)key;
                            if (logger.isTraceEnabled()) {
                                FileDataInput fdi = this.dfile.getSegment(indexEntry.position);
                                DecoratedKey keyInDisk = this.partitioner.decorateKey(ByteBufferUtil.readWithShortLength(fdi));
                                if (!keyInDisk.equals(key)) {
                                    throw new AssertionError((Object)String.format("%s != %s in %s", keyInDisk, key, fdi.getPath()));
                                }
                                fdi.close();
                            }
                            this.cacheKey(decoratedKey, indexEntry);
                        }
                        if (op == Operator.EQ && updateCacheAndStats) {
                            this.bloomFilterTracker.addTruePositive();
                        }
                        Tracing.trace("Partition index with {} entries found for sstable {}", indexEntry.columnsIndex().size(), this.descriptor.generation);
                        RowIndexEntry rowIndexEntry = indexEntry;
                        return rowIndexEntry;
                    }
                    RowIndexEntry.serializer.skip(in);
                }
            }
            catch (IOException e) {
                this.markSuspect();
                throw new CorruptSSTableException((Exception)e, in.getPath());
            }
            finally {
                FileUtils.closeQuietly(in);
            }
        }
        if (op == Operator.EQ && updateCacheAndStats) {
            this.bloomFilterTracker.addFalsePositive();
        }
        Tracing.trace("Partition index lookup complete (bloom filter false positive) for sstable {}", this.descriptor.generation);
        return null;
    }

    public DecoratedKey firstKeyBeyond(RowPosition token) {
        long sampledPosition = this.getIndexScanPosition(token);
        if (sampledPosition == -1L) {
            sampledPosition = 0L;
        }
        if (this.ifile == null) {
            return null;
        }
        Iterator<FileDataInput> segments = this.ifile.iterator(sampledPosition);
        while (segments.hasNext()) {
            FileDataInput in = segments.next();
            try {
                while (!in.isEOF()) {
                    ByteBuffer indexKey = ByteBufferUtil.readWithShortLength(in);
                    DecoratedKey indexDecoratedKey = this.partitioner.decorateKey(indexKey);
                    if (indexDecoratedKey.compareTo(token) > 0) {
                        DecoratedKey decoratedKey = indexDecoratedKey;
                        return decoratedKey;
                    }
                    RowIndexEntry.serializer.skip(in);
                }
            }
            catch (IOException e) {
                this.markSuspect();
                throw new CorruptSSTableException((Exception)e, in.getPath());
            }
            finally {
                FileUtils.closeQuietly(in);
            }
        }
        return null;
    }

    public long uncompressedLength() {
        return this.dfile.length;
    }

    public long onDiskLength() {
        return this.dfile.onDiskLength;
    }

    public boolean acquireReference() {
        int n;
        do {
            if ((n = this.references.get()) > 0) continue;
            return false;
        } while (!this.references.compareAndSet(n, n + 1));
        return true;
    }

    public void releaseReference() {
        if (this.references.decrementAndGet() == 0 && this.isCompacted.get()) {
            this.dropPageCache();
            FileUtils.closeQuietly(this);
            this.deletingTask.schedule();
        }
        assert (this.references.get() >= 0) : "Reference counter " + this.references.get() + " for " + this.dfile.path;
    }

    public boolean markObsolete() {
        if (logger.isDebugEnabled()) {
            logger.debug("Marking " + this.getFilename() + " compacted");
        }
        return !this.isCompacted.getAndSet(true);
    }

    public boolean isMarkedCompacted() {
        return this.isCompacted.get();
    }

    public void markSuspect() {
        if (logger.isDebugEnabled()) {
            logger.debug("Marking " + this.getFilename() + " as a suspect for blacklisting.");
        }
        this.isSuspect.getAndSet(true);
    }

    public boolean isMarkedSuspect() {
        return this.isSuspect.get();
    }

    public SSTableScanner getScanner(DataRange dataRange) {
        return new SSTableScanner(this, dataRange, null);
    }

    public SSTableScanner getScanner() {
        return this.getScanner((RateLimiter)null);
    }

    public SSTableScanner getScanner(RateLimiter limiter) {
        return new SSTableScanner(this, DataRange.allData(this.partitioner), limiter);
    }

    public ICompactionScanner getScanner(Range<Token> range, RateLimiter limiter) {
        if (range == null) {
            return this.getScanner(limiter);
        }
        return this.getScanner(Collections.singletonList(range), limiter);
    }

    public ICompactionScanner getScanner(Collection<Range<Token>> ranges, RateLimiter limiter) {
        List<Pair<Long, Long>> positions = this.getPositionsForRanges(Range.normalize(ranges));
        if (positions.isEmpty()) {
            return new EmptyCompactionScanner(this.getFilename());
        }
        return new SSTableScanner(this, ranges, limiter);
    }

    public FileDataInput getFileDataInput(long position) {
        return this.dfile.getSegment(position);
    }

    public boolean newSince(long age) {
        return this.maxDataAge > age;
    }

    public void createLinks(String snapshotDirectoryPath) {
        for (Component component : this.components) {
            File sourceFile = new File(this.descriptor.filenameFor(component));
            File targetLink = new File(snapshotDirectoryPath, sourceFile.getName());
            FileUtils.createHardLink(sourceFile, targetLink);
        }
    }

    public long getBloomFilterFalsePositiveCount() {
        return this.bloomFilterTracker.getFalsePositiveCount();
    }

    public long getRecentBloomFilterFalsePositiveCount() {
        return this.bloomFilterTracker.getRecentFalsePositiveCount();
    }

    public long getBloomFilterTruePositiveCount() {
        return this.bloomFilterTracker.getTruePositiveCount();
    }

    public long getRecentBloomFilterTruePositiveCount() {
        return this.bloomFilterTracker.getRecentTruePositiveCount();
    }

    public InstrumentingCache<KeyCacheKey, RowIndexEntry> getKeyCache() {
        return this.keyCache;
    }

    public EstimatedHistogram getEstimatedRowSize() {
        return this.sstableMetadata.estimatedRowSize;
    }

    public EstimatedHistogram getEstimatedColumnCount() {
        return this.sstableMetadata.estimatedColumnCount;
    }

    public double getEstimatedDroppableTombstoneRatio(int gcBefore) {
        return this.sstableMetadata.getEstimatedDroppableTombstoneRatio(gcBefore);
    }

    public double getDroppableTombstonesBefore(int gcBefore) {
        return this.sstableMetadata.getDroppableTombstonesBefore(gcBefore);
    }

    public double getCompressionRatio() {
        return this.sstableMetadata.compressionRatio;
    }

    public ReplayPosition getReplayPosition() {
        return this.sstableMetadata.replayPosition;
    }

    public long getMinTimestamp() {
        return this.sstableMetadata.minTimestamp;
    }

    public long getMaxTimestamp() {
        return this.sstableMetadata.maxTimestamp;
    }

    public Set<Integer> getAncestors() {
        try {
            return (Set)SSTableMetadata.serializer.deserialize((Descriptor)this.descriptor).right;
        }
        catch (IOException e) {
            SSTableReader.logOpenException(this.descriptor, e);
            return Collections.emptySet();
        }
    }

    public int getSSTableLevel() {
        return this.sstableMetadata.sstableLevel;
    }

    public void reloadSSTableMetadata() throws IOException {
        this.sstableMetadata = (SSTableMetadata)SSTableMetadata.serializer.deserialize((Descriptor)this.descriptor).left;
    }

    public SSTableMetadata getSSTableMetadata() {
        return this.sstableMetadata;
    }

    public RandomAccessReader openDataReader(RateLimiter limiter) {
        assert (limiter != null);
        return this.compression ? CompressedThrottledReader.open(this.getFilename(), this.getCompressionMetadata(), limiter) : ThrottledReader.open(new File(this.getFilename()), limiter);
    }

    public RandomAccessReader openDataReader() {
        return this.compression ? CompressedRandomAccessReader.open(this.getFilename(), this.getCompressionMetadata()) : RandomAccessReader.open(new File(this.getFilename()));
    }

    public RandomAccessReader openIndexReader() {
        return RandomAccessReader.open(new File(this.getIndexFilename()));
    }

    public long getCreationTimeFor(Component component) {
        return new File(this.descriptor.filenameFor(component)).lastModified();
    }

    public long getKeyCacheHit() {
        return this.keyCacheHit.get();
    }

    public long getKeyCacheRequest() {
        return this.keyCacheRequest.get();
    }

    public static boolean acquireReferences(Iterable<SSTableReader> sstables) {
        SSTableReader failed = null;
        for (SSTableReader sstable : sstables) {
            if (sstable.acquireReference()) continue;
            failed = sstable;
            break;
        }
        if (failed == null) {
            return true;
        }
        for (SSTableReader sstable : sstables) {
            if (sstable == failed) break;
            sstable.releaseReference();
        }
        return false;
    }

    public static void releaseReferences(Iterable<SSTableReader> sstables) {
        for (SSTableReader sstable : sstables) {
            sstable.releaseReference();
        }
    }

    private void dropPageCache() {
        this.dropPageCache(this.dfile.path);
        if (null != this.ifile) {
            this.dropPageCache(this.ifile.path);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     */
    private void dropPageCache(String filePath) {
        RandomAccessFile file;
        block5: {
            file = null;
            try {
                file = new RandomAccessFile(filePath, "r");
                int fd = CLibrary.getfd(file.getFD());
                if (fd <= 0) break block5;
                if (logger.isDebugEnabled()) {
                    logger.debug(String.format("Dropping page cache of file %s.", filePath));
                }
                CLibrary.trySkipCache(fd, 0L, 0);
            }
            catch (IOException iOException) {
                FileUtils.closeQuietly(file);
                catch (Throwable throwable) {
                    FileUtils.closeQuietly(file);
                    throw throwable;
                }
            }
        }
        FileUtils.closeQuietly(file);
    }

    public void incrementReadCount() {
        if (this.readMeter != null) {
            this.readMeter.mark();
        }
    }

    static {
        syncExecutor.setRemoveOnCancelPolicy(true);
        meterSyncThrottle = RateLimiter.create((double)100.0);
    }

    public static class SizeComparator
    implements Comparator<SSTableReader> {
        @Override
        public int compare(SSTableReader o1, SSTableReader o2) {
            return Longs.compare((long)o1.onDiskLength(), (long)o2.onDiskLength());
        }
    }

    protected class EmptyCompactionScanner
    implements ICompactionScanner {
        private final String filename;

        public EmptyCompactionScanner(String filename) {
            this.filename = filename;
        }

        @Override
        public long getLengthInBytes() {
            return 0L;
        }

        @Override
        public long getCurrentPosition() {
            return 0L;
        }

        @Override
        public String getBackingFiles() {
            return this.filename;
        }

        @Override
        public boolean hasNext() {
            return false;
        }

        @Override
        public OnDiskAtomIterator next() {
            return null;
        }

        @Override
        public void close() throws IOException {
        }

        @Override
        public void remove() {
        }
    }

    public static abstract class Operator {
        public static final Operator EQ = new Equals();
        public static final Operator GE = new GreaterThanOrEqualTo();
        public static final Operator GT = new GreaterThan();

        public abstract int apply(int var1);

        static final class GreaterThan
        extends Operator {
            GreaterThan() {
            }

            @Override
            public int apply(int comparison) {
                return comparison > 0 ? 0 : 1;
            }
        }

        static final class GreaterThanOrEqualTo
        extends Operator {
            GreaterThanOrEqualTo() {
            }

            @Override
            public int apply(int comparison) {
                return comparison >= 0 ? 0 : -comparison;
            }
        }

        static final class Equals
        extends Operator {
            Equals() {
            }

            @Override
            public int apply(int comparison) {
                return -comparison;
            }
        }
    }
}

