/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.streaming;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import java.io.IOException;
import java.net.Socket;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SocketChannel;
import java.nio.channels.WritableByteChannel;
import java.util.Collection;
import java.util.Comparator;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.cassandra.streaming.StreamSession;
import org.apache.cassandra.streaming.messages.StreamInitMessage;
import org.apache.cassandra.streaming.messages.StreamMessage;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionHandler {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionHandler.class);
    private final StreamSession session;
    private IncomingMessageHandler incoming;
    private OutgoingMessageHandler outgoing;

    ConnectionHandler(StreamSession session) {
        this.session = session;
        this.incoming = new IncomingMessageHandler(session);
        this.outgoing = new OutgoingMessageHandler(session);
    }

    public void initiate() throws IOException {
        logger.debug("[Stream #{}] Sending stream init for incoming stream", (Object)this.session.planId());
        Socket incomingSocket = this.session.createConnection();
        this.incoming.start(incomingSocket, 1);
        this.incoming.sendInitMessage(incomingSocket, true);
        logger.debug("[Stream #{}] Sending stream init for outgoing stream", (Object)this.session.planId());
        Socket outgoingSocket = this.session.createConnection();
        this.outgoing.start(outgoingSocket, 1);
        this.outgoing.sendInitMessage(outgoingSocket, false);
    }

    public void initiateOnReceivingSide(Socket socket, boolean isForOutgoing, int version) throws IOException {
        if (isForOutgoing) {
            this.outgoing.start(socket, version);
        } else {
            this.incoming.start(socket, version);
        }
    }

    public ListenableFuture<?> close() {
        logger.debug("[Stream #{}] Closing stream connection handler on {}", (Object)this.session.planId(), (Object)this.session.peer);
        ListenableFuture<?> inClosed = this.incoming == null ? Futures.immediateFuture(null) : this.incoming.close();
        ListenableFuture<?> outClosed = this.outgoing == null ? Futures.immediateFuture(null) : this.outgoing.close();
        return Futures.allAsList((ListenableFuture[])new ListenableFuture[]{inClosed, outClosed});
    }

    public void sendMessages(Collection<? extends StreamMessage> messages) {
        for (StreamMessage streamMessage : messages) {
            this.sendMessage(streamMessage);
        }
    }

    public void sendMessage(StreamMessage message) {
        if (this.outgoing.isClosed()) {
            throw new RuntimeException("Outgoing stream handler has been closed");
        }
        this.outgoing.enqueue(message);
    }

    public boolean isOutgoingConnected() {
        return this.outgoing != null && !this.outgoing.isClosed();
    }

    static class OutgoingMessageHandler
    extends MessageHandler {
        private final PriorityBlockingQueue<StreamMessage> messageQueue = new PriorityBlockingQueue<StreamMessage>(64, new Comparator<StreamMessage>(){

            @Override
            public int compare(StreamMessage o1, StreamMessage o2) {
                return o2.getPriority() - o1.getPriority();
            }
        });

        OutgoingMessageHandler(StreamSession session) {
            super(session);
        }

        @Override
        protected String name() {
            return "STREAM-OUT";
        }

        public void enqueue(StreamMessage message) {
            this.messageQueue.put(message);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                StreamMessage next;
                WritableByteChannel out = OutgoingMessageHandler.getWriteChannel(this.socket);
                while (!this.isClosed()) {
                    next = this.messageQueue.poll(1L, TimeUnit.SECONDS);
                    if (next == null) continue;
                    logger.debug("[Stream #{}] Sending {}", (Object)this.session.planId(), (Object)next);
                    this.sendMessage(out, next);
                    if (next.type != StreamMessage.Type.SESSION_FAILED) continue;
                    this.close();
                }
                while ((next = this.messageQueue.poll()) != null) {
                    this.sendMessage(out, next);
                }
            }
            catch (InterruptedException e) {
                throw new AssertionError((Object)e);
            }
            catch (Throwable e) {
                this.session.onError(e);
            }
            finally {
                this.signalCloseDone();
            }
        }

        private void sendMessage(WritableByteChannel out, StreamMessage message) {
            try {
                StreamMessage.serialize(message, out, this.protocolVersion, this.session);
            }
            catch (SocketException e) {
                this.session.onError(e);
                this.close();
            }
            catch (IOException e) {
                this.session.onError(e);
            }
        }
    }

    static class IncomingMessageHandler
    extends MessageHandler {
        IncomingMessageHandler(StreamSession session) {
            super(session);
        }

        @Override
        protected String name() {
            return "STREAM-IN";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                ReadableByteChannel in = IncomingMessageHandler.getReadChannel(this.socket);
                while (!this.isClosed()) {
                    StreamMessage message = StreamMessage.deserialize(in, this.protocolVersion, this.session);
                    if (message == null) continue;
                    logger.debug("[Stream #{}] Received {}", (Object)this.session.planId(), (Object)message);
                    this.session.messageReceived(message);
                }
            }
            catch (SocketException e) {
                this.close();
            }
            catch (Throwable e) {
                this.session.onError(e);
            }
            finally {
                this.signalCloseDone();
            }
        }
    }

    static abstract class MessageHandler
    implements Runnable {
        protected final StreamSession session;
        protected int protocolVersion;
        protected Socket socket;
        private final AtomicReference<SettableFuture<?>> closeFuture = new AtomicReference();

        protected MessageHandler(StreamSession session) {
            this.session = session;
        }

        protected abstract String name();

        protected static WritableByteChannel getWriteChannel(Socket socket) throws IOException {
            SocketChannel out = socket.getChannel();
            return out == null ? Channels.newChannel(socket.getOutputStream()) : out;
        }

        protected static ReadableByteChannel getReadChannel(Socket socket) throws IOException {
            SocketChannel in = socket.getChannel();
            return in == null ? Channels.newChannel(socket.getInputStream()) : in;
        }

        public void sendInitMessage(Socket socket, boolean isForOutgoing) throws IOException {
            StreamInitMessage message = new StreamInitMessage(FBUtilities.getBroadcastAddress(), this.session.planId(), this.session.description(), isForOutgoing);
            ByteBuffer messageBuf = message.createMessage(false, this.protocolVersion);
            while (messageBuf.hasRemaining()) {
                MessageHandler.getWriteChannel(socket).write(messageBuf);
            }
        }

        public void start(Socket socket, int protocolVersion) {
            this.socket = socket;
            this.protocolVersion = protocolVersion;
            new Thread((Runnable)this, this.name() + "-" + this.session.peer).start();
        }

        public ListenableFuture<?> close() {
            SettableFuture<?> future = SettableFuture.create();
            return this.closeFuture.compareAndSet(null, future) ? future : this.closeFuture.get();
        }

        public boolean isClosed() {
            return this.closeFuture.get() != null;
        }

        protected void signalCloseDone() {
            this.closeFuture.get().set(null);
            try {
                this.socket.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }
}

