/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.DefsTable;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.SystemTable;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.db.filter.QueryFilter;
import org.apache.cassandra.db.filter.QueryPath;
import org.apache.cassandra.exceptions.AlreadyExistsException;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.gms.ApplicationState;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.gms.IEndpointStateChangeSubscriber;
import org.apache.cassandra.gms.VersionedValue;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.IMigrationListener;
import org.apache.cassandra.service.MigrationTask;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.UUIDGen;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationManager
implements IEndpointStateChangeSubscriber {
    private static final Logger logger = LoggerFactory.getLogger(MigrationManager.class);
    private static final ByteBuffer LAST_MIGRATION_KEY = ByteBufferUtil.bytes("Last Migration");
    public static final MigrationManager instance = new MigrationManager();
    private final List<IMigrationListener> listeners = new CopyOnWriteArrayList<IMigrationListener>();

    private MigrationManager() {
    }

    public void register(IMigrationListener listener) {
        this.listeners.add(listener);
    }

    public void unregister(IMigrationListener listener) {
        this.listeners.remove(listener);
    }

    @Override
    public void onJoin(InetAddress endpoint, EndpointState epState) {
    }

    @Override
    public void onChange(InetAddress endpoint, ApplicationState state, VersionedValue value) {
        if (state != ApplicationState.SCHEMA || endpoint.equals(FBUtilities.getBroadcastAddress())) {
            return;
        }
        MigrationManager.rectifySchema(UUID.fromString(value.value), endpoint);
    }

    @Override
    public void onAlive(InetAddress endpoint, EndpointState state) {
        VersionedValue value = state.getApplicationState(ApplicationState.SCHEMA);
        if (value != null) {
            MigrationManager.rectifySchema(UUID.fromString(value.value), endpoint);
        }
    }

    @Override
    public void onDead(InetAddress endpoint, EndpointState state) {
    }

    @Override
    public void onRestart(InetAddress endpoint, EndpointState state) {
    }

    @Override
    public void onRemove(InetAddress endpoint) {
    }

    private static void rectifySchema(UUID theirVersion, InetAddress endpoint) {
        if (MessagingService.instance().getVersion(endpoint) < 5) {
            return;
        }
        if (Schema.instance.getVersion().equals(theirVersion)) {
            return;
        }
        StageManager.getStage(Stage.MIGRATION).submit(new MigrationTask(endpoint));
    }

    public static boolean isReadyForBootstrap() {
        return StageManager.getStage(Stage.MIGRATION).getActiveCount() == 0;
    }

    public void notifyCreateKeyspace(KSMetaData ksm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onCreateKeyspace(ksm.name);
        }
    }

    public void notifyCreateColumnFamily(CFMetaData cfm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onCreateColumnFamly(cfm.ksName, cfm.cfName);
        }
    }

    public void notifyUpdateKeyspace(KSMetaData ksm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onUpdateKeyspace(ksm.name);
        }
    }

    public void notifyUpdateColumnFamily(CFMetaData cfm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onUpdateColumnFamly(cfm.ksName, cfm.cfName);
        }
    }

    public void notifyDropKeyspace(KSMetaData ksm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onDropKeyspace(ksm.name);
        }
    }

    public void notifyDropColumnFamily(CFMetaData cfm) {
        for (IMigrationListener listener : this.listeners) {
            listener.onDropColumnFamly(cfm.ksName, cfm.cfName);
        }
    }

    public static void announceNewKeyspace(KSMetaData ksm) throws ConfigurationException {
        ksm.validate();
        if (Schema.instance.getTableDefinition(ksm.name) != null) {
            throw new AlreadyExistsException(ksm.name);
        }
        logger.info(String.format("Create new Keyspace: %s", ksm));
        MigrationManager.announce(ksm.toSchema(FBUtilities.timestampMicros()));
    }

    public static void announceNewColumnFamily(CFMetaData cfm) throws ConfigurationException {
        cfm.validate();
        KSMetaData ksm = Schema.instance.getTableDefinition(cfm.ksName);
        if (ksm == null) {
            throw new ConfigurationException(String.format("Cannot add column family '%s' to non existing keyspace '%s'.", cfm.cfName, cfm.ksName));
        }
        if (ksm.cfMetaData().containsKey(cfm.cfName)) {
            throw new AlreadyExistsException(cfm.cfName, cfm.ksName);
        }
        logger.info(String.format("Create new ColumnFamily: %s", cfm));
        MigrationManager.announce(cfm.toSchema(FBUtilities.timestampMicros()));
    }

    public static void announceKeyspaceUpdate(KSMetaData ksm) throws ConfigurationException {
        ksm.validate();
        KSMetaData oldKsm = Schema.instance.getKSMetaData(ksm.name);
        if (oldKsm == null) {
            throw new ConfigurationException(String.format("Cannot update non existing keyspace '%s'.", ksm.name));
        }
        logger.info(String.format("Update Keyspace '%s' From %s To %s", ksm.name, oldKsm, ksm));
        MigrationManager.announce(oldKsm.toSchemaUpdate(ksm, FBUtilities.timestampMicros()));
    }

    public static void announceColumnFamilyUpdate(CFMetaData cfm) throws ConfigurationException {
        cfm.validate();
        CFMetaData oldCfm = Schema.instance.getCFMetaData(cfm.ksName, cfm.cfName);
        if (oldCfm == null) {
            throw new ConfigurationException(String.format("Cannot update non existing column family '%s' in keyspace '%s'.", cfm.cfName, cfm.ksName));
        }
        logger.info(String.format("Update ColumnFamily '%s/%s' From %s To %s", cfm.ksName, cfm.cfName, oldCfm, cfm));
        MigrationManager.announce(oldCfm.toSchemaUpdate(cfm, FBUtilities.timestampMicros()));
    }

    public static void announceKeyspaceDrop(String ksName) throws ConfigurationException {
        KSMetaData oldKsm = Schema.instance.getKSMetaData(ksName);
        if (oldKsm == null) {
            throw new ConfigurationException(String.format("Cannot drop non existing keyspace '%s'.", ksName));
        }
        logger.info(String.format("Drop Keyspace '%s'", oldKsm.name));
        MigrationManager.announce(oldKsm.dropFromSchema(FBUtilities.timestampMicros()));
    }

    public static void announceColumnFamilyDrop(String ksName, String cfName) throws ConfigurationException {
        CFMetaData oldCfm = Schema.instance.getCFMetaData(ksName, cfName);
        if (oldCfm == null) {
            throw new ConfigurationException(String.format("Cannot drop non existing column family '%s' in keyspace '%s'.", cfName, ksName));
        }
        logger.info(String.format("Drop ColumnFamily '%s/%s'", oldCfm.ksName, oldCfm.cfName));
        MigrationManager.announce(oldCfm.dropFromSchema(FBUtilities.timestampMicros()));
    }

    private static void announce(RowMutation schema) {
        FBUtilities.waitOnFuture(MigrationManager.announce(Collections.singletonList(schema)));
    }

    private static void pushSchemaMutation(InetAddress endpoint, Collection<RowMutation> schema) {
        MessageOut<Collection<RowMutation>> msg = new MessageOut<Collection<RowMutation>>(MessagingService.Verb.DEFINITIONS_UPDATE, schema, MigrationsSerializer.instance);
        MessagingService.instance().sendOneWay(msg, endpoint);
    }

    private static Future<?> announce(final Collection<RowMutation> schema) {
        Future<Object> f = StageManager.getStage(Stage.MIGRATION).submit(new Callable<Object>(){

            @Override
            public Object call() throws Exception {
                DefsTable.mergeSchema(schema);
                return null;
            }
        });
        for (InetAddress endpoint : Gossiper.instance.getLiveMembers()) {
            if (endpoint.equals(FBUtilities.getBroadcastAddress()) || MessagingService.instance().getVersion(endpoint) < 4) continue;
            MigrationManager.pushSchemaMutation(endpoint, schema);
        }
        return f;
    }

    public static void passiveAnnounce(UUID version) {
        assert (Gossiper.instance.isEnabled());
        Gossiper.instance.addLocalApplicationState(ApplicationState.SCHEMA, StorageService.instance.valueFactory.schema(version));
        logger.debug("Gossiping my schema version " + version);
    }

    public static void resetLocalSchema() throws IOException {
        logger.info("Starting local schema reset...");
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("Truncating schema tables...");
            }
            FBUtilities.waitOnFutures(new ArrayList<Future<?>>(3){
                {
                    SystemTable.schemaCFS("schema_keyspaces").truncate();
                    SystemTable.schemaCFS("schema_columnfamilies").truncate();
                    SystemTable.schemaCFS("schema_columns").truncate();
                }
            });
            if (logger.isDebugEnabled()) {
                logger.debug("Clearing local schema keyspace definitions...");
            }
            Schema.instance.clear();
            Set<InetAddress> liveEndpoints = Gossiper.instance.getLiveMembers();
            liveEndpoints.remove(FBUtilities.getBroadcastAddress());
            for (InetAddress node : liveEndpoints) {
                if (MessagingService.instance().getVersion(node) < 5) continue;
                if (logger.isDebugEnabled()) {
                    logger.debug("Requesting schema from " + node);
                }
                FBUtilities.waitOnFuture(StageManager.getStage(Stage.MIGRATION).submit(new MigrationTask(node)));
                break;
            }
            logger.info("Local schema reset is complete.");
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    @Deprecated
    public static UUID getLastMigrationId() {
        QueryFilter filter;
        DecoratedKey dkey = StorageService.getPartitioner().decorateKey(LAST_MIGRATION_KEY);
        Table defs = Table.open("system");
        ColumnFamilyStore cfStore = defs.getColumnFamilyStore("Schema");
        ColumnFamily cf = cfStore.getColumnFamily(filter = QueryFilter.getNamesFilter(dkey, new QueryPath("Schema"), LAST_MIGRATION_KEY));
        if (cf == null || cf.getColumnNames().size() == 0) {
            return null;
        }
        return UUIDGen.getUUID(cf.getColumn(LAST_MIGRATION_KEY).value());
    }

    public static class MigrationsSerializer
    implements IVersionedSerializer<Collection<RowMutation>> {
        public static MigrationsSerializer instance = new MigrationsSerializer();

        @Override
        public void serialize(Collection<RowMutation> schema, DataOutput out, int version) throws IOException {
            out.writeInt(schema.size());
            for (RowMutation rm : schema) {
                RowMutation.serializer.serialize(rm, out, version);
            }
        }

        @Override
        public Collection<RowMutation> deserialize(DataInput in, int version) throws IOException {
            int count = in.readInt();
            ArrayList<RowMutation> schema = new ArrayList<RowMutation>(count);
            for (int i = 0; i < count; ++i) {
                schema.add(RowMutation.serializer.deserialize(in, version));
            }
            return schema;
        }

        @Override
        public long serializedSize(Collection<RowMutation> schema, int version) {
            int size = TypeSizes.NATIVE.sizeof(schema.size());
            for (RowMutation rm : schema) {
                size = (int)((long)size + RowMutation.serializer.serializedSize(rm, version));
            }
            return size;
        }
    }
}

