/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.base.Throwables;
import java.io.Closeable;
import java.io.DataInput;
import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DataTracker;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.RowIndexEntry;
import org.apache.cassandra.db.compaction.AbstractCompactedRow;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.sstable.SSTableWriter;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.OutputHandler;

public class Scrubber
implements Closeable {
    public final ColumnFamilyStore cfs;
    public final SSTableReader sstable;
    public final File destination;
    private final CompactionController controller;
    private final boolean isCommutative;
    private final int expectedBloomFilterSize;
    private final RandomAccessReader dataFile;
    private final RandomAccessReader indexFile;
    private final ScrubInfo scrubInfo;
    private long rowsRead;
    private SSTableWriter writer;
    private SSTableReader newSstable;
    private SSTableReader newInOrderSstable;
    private int goodRows;
    private int badRows;
    private int emptyRows;
    private final OutputHandler outputHandler;
    private static final Comparator<AbstractCompactedRow> acrComparator = new Comparator<AbstractCompactedRow>(){

        @Override
        public int compare(AbstractCompactedRow r1, AbstractCompactedRow r2) {
            return r1.key.compareTo(r2.key);
        }
    };
    private final Set<AbstractCompactedRow> outOfOrderRows = new TreeSet<AbstractCompactedRow>(acrComparator);

    public Scrubber(ColumnFamilyStore cfs, SSTableReader sstable) throws IOException {
        this(cfs, sstable, new OutputHandler.LogOutput(), false);
    }

    public Scrubber(ColumnFamilyStore cfs, SSTableReader sstable, OutputHandler outputHandler, boolean isOffline) throws IOException {
        this.cfs = cfs;
        this.sstable = sstable;
        this.outputHandler = outputHandler;
        this.destination = cfs.directories.getDirectoryForNewSSTables(sstable.onDiskLength());
        if (this.destination == null) {
            throw new IOException("disk full");
        }
        List<SSTableReader> toScrub = Collections.singletonList(sstable);
        this.controller = isOffline ? new ScrubController(cfs) : new CompactionController(cfs, Collections.singletonList(sstable), CompactionManager.getDefaultGcBefore(cfs), true);
        this.isCommutative = cfs.metadata.getDefaultValidator().isCommutative();
        this.expectedBloomFilterSize = Math.max(DatabaseDescriptor.getIndexInterval(), (int)SSTableReader.getApproximateKeyCount(toScrub));
        this.dataFile = sstable.openDataReader(true);
        this.indexFile = RandomAccessReader.open(new File(sstable.descriptor.filenameFor(Component.PRIMARY_INDEX)), true);
        this.scrubInfo = new ScrubInfo(this.dataFile, sstable);
    }

    public void scrub() throws IOException {
        this.outputHandler.output("Scrubbing " + this.sstable);
        try {
            ByteBuffer nextIndexKey = ByteBufferUtil.readWithShortLength(this.indexFile);
            long firstRowPositionFromIndex = RowIndexEntry.serializer.deserialize((DataInput)this.indexFile, (Descriptor.Version)this.sstable.descriptor.version).position;
            assert (firstRowPositionFromIndex == 0L) : firstRowPositionFromIndex;
            this.writer = CompactionManager.maybeCreateWriter(this.cfs, this.destination, this.expectedBloomFilterSize, null, Collections.singletonList(this.sstable));
            AbstractCompactedRow prevRow = null;
            while (!this.dataFile.isEOF()) {
                long nextRowPositionFromIndex;
                if (this.scrubInfo.isStopRequested()) {
                    throw new CompactionInterruptedException(this.scrubInfo.getCompactionInfo());
                }
                long rowStart = this.dataFile.getFilePointer();
                this.outputHandler.debug("Reading row at " + rowStart);
                DecoratedKey key = null;
                long dataSize = -1L;
                try {
                    key = SSTableReader.decodeKey(this.sstable.partitioner, this.sstable.descriptor, ByteBufferUtil.readWithShortLength(this.dataFile));
                    dataSize = this.sstable.descriptor.version.hasIntRowSize ? (long)this.dataFile.readInt() : this.dataFile.readLong();
                    this.outputHandler.debug(String.format("row %s is %s bytes", ByteBufferUtil.bytesToHex(key.key), dataSize));
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                }
                ByteBuffer currentIndexKey = nextIndexKey;
                try {
                    nextIndexKey = this.indexFile.isEOF() ? null : ByteBufferUtil.readWithShortLength(this.indexFile);
                    nextRowPositionFromIndex = this.indexFile.isEOF() ? this.dataFile.length() : RowIndexEntry.serializer.deserialize((DataInput)this.indexFile, (Descriptor.Version)this.sstable.descriptor.version).position;
                }
                catch (Throwable th) {
                    this.outputHandler.warn("Error reading index file", th);
                    nextIndexKey = null;
                    nextRowPositionFromIndex = this.dataFile.length();
                }
                long dataStart = this.dataFile.getFilePointer();
                long dataStartFromIndex = currentIndexKey == null ? -1L : rowStart + 2L + (long)currentIndexKey.remaining() + (long)(this.sstable.descriptor.version.hasIntRowSize ? 4 : 8);
                long dataSizeFromIndex = nextRowPositionFromIndex - dataStartFromIndex;
                assert (currentIndexKey != null || this.indexFile.isEOF());
                if (currentIndexKey != null) {
                    this.outputHandler.debug(String.format("Index doublecheck: row %s is %s bytes", ByteBufferUtil.bytesToHex(currentIndexKey), dataSizeFromIndex));
                }
                this.writer.mark();
                try {
                    if (key == null) {
                        throw new IOError(new IOException("Unable to read row key from data file"));
                    }
                    if (dataSize > this.dataFile.length()) {
                        throw new IOError(new IOException("Impossible row size " + dataSize));
                    }
                    SSTableIdentityIterator row = new SSTableIdentityIterator(this.sstable, this.dataFile, key, dataStart, dataSize, true);
                    AbstractCompactedRow compactedRow = this.controller.getCompactedRow(row);
                    if (compactedRow.isEmpty()) {
                        ++this.emptyRows;
                    } else {
                        if (prevRow != null && acrComparator.compare(prevRow, compactedRow) >= 0) {
                            this.outOfOrderRows.add(compactedRow);
                            this.outputHandler.warn(String.format("Out of order row detected (%s found after %s)", compactedRow.key, prevRow.key));
                            continue;
                        }
                        this.writer.append(compactedRow);
                        prevRow = compactedRow;
                        ++this.goodRows;
                    }
                    if (!key.key.equals(currentIndexKey) || dataStart != dataStartFromIndex) {
                        this.outputHandler.warn("Index file contained a different key or row size; using key from data file");
                    }
                }
                catch (Throwable th) {
                    this.throwIfFatal(th);
                    this.outputHandler.warn("Non-fatal error reading row (stacktrace follows)", th);
                    this.writer.resetAndTruncate();
                    if (!(currentIndexKey == null || key != null && key.key.equals(currentIndexKey) && dataStart == dataStartFromIndex && dataSize == dataSizeFromIndex)) {
                        this.outputHandler.output(String.format("Retrying from row index; data is %s bytes starting at %s", dataSizeFromIndex, dataStartFromIndex));
                        key = SSTableReader.decodeKey(this.sstable.partitioner, this.sstable.descriptor, currentIndexKey);
                        try {
                            SSTableIdentityIterator row = new SSTableIdentityIterator(this.sstable, this.dataFile, key, dataStartFromIndex, dataSizeFromIndex, true);
                            AbstractCompactedRow compactedRow = this.controller.getCompactedRow(row);
                            if (compactedRow.isEmpty()) {
                                ++this.emptyRows;
                            }
                            if (prevRow != null && acrComparator.compare(prevRow, compactedRow) >= 0) {
                                this.outOfOrderRows.add(compactedRow);
                                this.outputHandler.warn(String.format("Out of order row detected (%s found after %s)", compactedRow.key, prevRow.key));
                                continue;
                            }
                            this.writer.append(compactedRow);
                            prevRow = compactedRow;
                            ++this.goodRows;
                        }
                        catch (Throwable th2) {
                            this.throwIfFatal(th2);
                            if (this.isCommutative) {
                                throw new IOError(th2);
                            }
                            this.outputHandler.warn("Retry failed too. Skipping to next row (retry's stacktrace follows)", th2);
                            this.writer.resetAndTruncate();
                            this.dataFile.seek(nextRowPositionFromIndex);
                            ++this.badRows;
                        }
                    }
                    if (this.isCommutative) {
                        throw new IOError(th);
                    }
                    this.outputHandler.warn("Row at " + dataStart + " is unreadable; skipping to next");
                    if (currentIndexKey != null) {
                        this.dataFile.seek(nextRowPositionFromIndex);
                    }
                    ++this.badRows;
                }
                if (this.rowsRead++ % 1000L != 0L) continue;
                this.controller.mayThrottle(this.dataFile.getFilePointer());
            }
            if (this.writer.getFilePointer() > 0L) {
                this.newSstable = this.writer.closeAndOpenReader(this.sstable.maxDataAge);
            }
        }
        catch (Throwable t) {
            if (this.writer != null) {
                this.writer.abort();
            }
            throw Throwables.propagate((Throwable)t);
        }
        if (!this.outOfOrderRows.isEmpty()) {
            SSTableWriter inOrderWriter = CompactionManager.maybeCreateWriter(this.cfs, this.destination, this.expectedBloomFilterSize, null, Collections.singletonList(this.sstable));
            for (AbstractCompactedRow row : this.outOfOrderRows) {
                inOrderWriter.append(row);
            }
            this.newInOrderSstable = inOrderWriter.closeAndOpenReader(this.sstable.maxDataAge);
            this.outputHandler.warn(String.format("%d out of order rows found while scrubbing %s; Those have been written (in order) to a new sstable (%s)", this.outOfOrderRows.size(), this.sstable, this.newInOrderSstable));
        }
        if (this.newSstable == null) {
            if (this.badRows > 0) {
                this.outputHandler.warn("No valid rows found while scrubbing " + this.sstable + "; it is marked for deletion now. If you want to attempt manual recovery, you can find a copy in the pre-scrub snapshot");
            } else {
                this.outputHandler.output("Scrub of " + this.sstable + " complete; looks like all " + this.emptyRows + " rows were tombstoned");
            }
        } else {
            this.outputHandler.output("Scrub of " + this.sstable + " complete: " + this.goodRows + " rows in new sstable and " + this.emptyRows + " empty (tombstoned) rows dropped");
            if (this.badRows > 0) {
                this.outputHandler.warn("Unable to recover " + this.badRows + " rows that were skipped.  You can attempt manual recovery from the pre-scrub snapshot.  You can also run nodetool repair to transfer the data from a healthy replica, if any");
            }
        }
    }

    public SSTableReader getNewSSTable() {
        return this.newSstable;
    }

    public SSTableReader getNewInOrderSSTable() {
        return this.newInOrderSstable;
    }

    private void throwIfFatal(Throwable th) {
        if (th instanceof Error && !(th instanceof AssertionError) && !(th instanceof IOError)) {
            throw (Error)th;
        }
    }

    @Override
    public void close() {
        FileUtils.closeQuietly(this.dataFile);
        FileUtils.closeQuietly(this.indexFile);
    }

    public CompactionInfo.Holder getScrubInfo() {
        return this.scrubInfo;
    }

    private static class ScrubController
    extends CompactionController {
        public ScrubController(ColumnFamilyStore cfs) {
            super(cfs, Integer.MAX_VALUE, DataTracker.buildIntervalTree(Collections.<SSTableReader>emptyList()));
        }

        @Override
        public boolean shouldPurge(DecoratedKey key) {
            return false;
        }
    }

    private static class ScrubInfo
    extends CompactionInfo.Holder {
        private final RandomAccessReader dataFile;
        private final SSTableReader sstable;

        public ScrubInfo(RandomAccessReader dataFile, SSTableReader sstable) {
            this.dataFile = dataFile;
            this.sstable = sstable;
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            try {
                return new CompactionInfo(this.sstable.metadata, OperationType.SCRUB, this.dataFile.getFilePointer(), this.dataFile.length());
            }
            catch (Exception e) {
                throw new RuntimeException();
            }
        }
    }
}

