/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.collect.Sets;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.db.compaction.AbstractCompactedRow;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionType;
import org.apache.cassandra.db.compaction.PrecompactedRow;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.IndexHelper;
import org.apache.cassandra.io.sstable.IndexSummary;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.BufferedRandomAccessFile;
import org.apache.cassandra.io.util.FileMark;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.SegmentedFile;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.streaming.OperationType;
import org.apache.cassandra.utils.BloomFilter;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.EstimatedHistogram;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SSTableWriter
extends SSTable {
    private static Logger logger = LoggerFactory.getLogger(SSTableWriter.class);
    private IndexWriter iwriter;
    private SegmentedFile.Builder dbuilder;
    private final BufferedRandomAccessFile dataFile;
    private DecoratedKey lastWrittenKey;
    private FileMark dataMark;

    public SSTableWriter(String filename, long keyCount) throws IOException {
        this(filename, keyCount, DatabaseDescriptor.getCFMetaData(Descriptor.fromFilename(filename)), StorageService.getPartitioner(), ReplayPosition.NONE);
    }

    public SSTableWriter(String filename, long keyCount, CFMetaData metadata, IPartitioner partitioner, ReplayPosition replayPosition) throws IOException {
        super(Descriptor.fromFilename(filename), new HashSet<Component>(Arrays.asList(Component.DATA, Component.FILTER, Component.PRIMARY_INDEX, Component.STATS)), metadata, replayPosition, partitioner, SSTable.defaultRowHistogram(), SSTable.defaultColumnHistogram());
        this.iwriter = new IndexWriter(this.descriptor, partitioner, keyCount);
        this.dbuilder = SegmentedFile.getBuilder(DatabaseDescriptor.getDiskAccessMode());
        this.dataFile = new BufferedRandomAccessFile(new File(this.getFilename()), "rw", 65535, true);
    }

    public void mark() {
        this.dataMark = this.dataFile.mark();
        this.iwriter.mark();
    }

    public void reset() {
        try {
            this.dataFile.reset(this.dataMark);
            this.iwriter.reset();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    private long beforeAppend(DecoratedKey decoratedKey) throws IOException {
        if (decoratedKey == null) {
            throw new IOException("Keys must not be null.");
        }
        if (this.lastWrittenKey != null && this.lastWrittenKey.compareTo(decoratedKey) > 0) {
            logger.info("Last written key : " + this.lastWrittenKey);
            logger.info("Current key : " + decoratedKey);
            logger.info("Writing into file " + this.getFilename());
            throw new IOException("Keys must be written in ascending order.");
        }
        return this.lastWrittenKey == null ? 0L : this.dataFile.getFilePointer();
    }

    private void afterAppend(DecoratedKey decoratedKey, long dataPosition) throws IOException {
        this.lastWrittenKey = decoratedKey;
        if (logger.isTraceEnabled()) {
            logger.trace("wrote " + decoratedKey + " at " + dataPosition);
        }
        this.iwriter.afterAppend(decoratedKey, dataPosition);
        this.dbuilder.addPotentialBoundary(dataPosition);
    }

    public long append(AbstractCompactedRow row) throws IOException {
        long currentPosition = this.beforeAppend(row.key);
        ByteBufferUtil.writeWithShortLength(row.key.key, this.dataFile);
        row.write(this.dataFile);
        this.estimatedRowSize.add(this.dataFile.getFilePointer() - currentPosition);
        this.estimatedColumnCount.add(row.columnCount());
        this.afterAppend(row.key, currentPosition);
        return currentPosition;
    }

    public void append(DecoratedKey decoratedKey, ColumnFamily cf) throws IOException {
        long startPosition = this.beforeAppend(decoratedKey);
        ByteBufferUtil.writeWithShortLength(decoratedKey.key, this.dataFile);
        long sizePosition = this.dataFile.getFilePointer();
        this.dataFile.writeLong(-1L);
        int columnCount = ColumnFamily.serializer().serializeWithIndexes(cf, this.dataFile);
        long endPosition = this.dataFile.getFilePointer();
        this.dataFile.seek(sizePosition);
        long dataSize = endPosition - (sizePosition + 8L);
        assert (dataSize > 0L);
        this.dataFile.writeLong(dataSize);
        this.dataFile.seek(endPosition);
        this.afterAppend(decoratedKey, startPosition);
        this.estimatedRowSize.add(endPosition - startPosition);
        this.estimatedColumnCount.add(columnCount);
    }

    public void append(DecoratedKey decoratedKey, ByteBuffer value) throws IOException {
        long currentPosition = this.beforeAppend(decoratedKey);
        ByteBufferUtil.writeWithShortLength(decoratedKey.key, this.dataFile);
        assert (value.remaining() > 0);
        this.dataFile.writeLong(value.remaining());
        ByteBufferUtil.write(value, this.dataFile);
        this.afterAppend(decoratedKey, currentPosition);
    }

    public SSTableReader closeAndOpenReader() throws IOException {
        return this.closeAndOpenReader(System.currentTimeMillis());
    }

    public SSTableReader closeAndOpenReader(long maxDataAge) throws IOException {
        this.iwriter.close();
        long position = this.dataFile.getFilePointer();
        this.dataFile.close();
        FileUtils.truncate(this.dataFile.getPath(), position);
        SSTableWriter.writeMetadata(this.descriptor, this.estimatedRowSize, this.estimatedColumnCount, this.replayPosition);
        Descriptor newdesc = SSTableWriter.rename(this.descriptor, this.components);
        SegmentedFile ifile = this.iwriter.builder.complete(newdesc.filenameFor(SSTable.COMPONENT_INDEX));
        SegmentedFile dfile = this.dbuilder.complete(newdesc.filenameFor(SSTable.COMPONENT_DATA));
        SSTableReader sstable = SSTableReader.internalOpen(newdesc, this.components, this.metadata, this.replayPosition, this.partitioner, ifile, dfile, this.iwriter.summary, this.iwriter.bf, maxDataAge, this.estimatedRowSize, this.estimatedColumnCount);
        this.iwriter = null;
        this.dbuilder = null;
        return sstable;
    }

    private static void writeMetadata(Descriptor desc, EstimatedHistogram rowSizes, EstimatedHistogram columnCounts, ReplayPosition rp) throws IOException {
        BufferedRandomAccessFile out = new BufferedRandomAccessFile(new File(desc.filenameFor(SSTable.COMPONENT_STATS)), "rw", 65535, true);
        EstimatedHistogram.serializer.serialize(rowSizes, (DataOutput)out);
        EstimatedHistogram.serializer.serialize(columnCounts, (DataOutput)out);
        ReplayPosition.serializer.serialize(rp, (DataOutput)out);
        out.close();
    }

    static Descriptor rename(Descriptor tmpdesc, Set<Component> components) {
        Descriptor newdesc = tmpdesc.asTemporary(false);
        try {
            for (Component component : Sets.difference(components, Collections.singleton(Component.DATA))) {
                FBUtilities.renameWithConfirm(tmpdesc.filenameFor(component), newdesc.filenameFor(component));
            }
            FBUtilities.renameWithConfirm(tmpdesc.filenameFor(Component.DATA), newdesc.filenameFor(Component.DATA));
        }
        catch (IOException e) {
            throw new IOError(e);
        }
        return newdesc;
    }

    public long getFilePointer() {
        return this.dataFile.getFilePointer();
    }

    public static Builder createBuilder(Descriptor desc, OperationType type) {
        if (!desc.isLatestVersion) {
            throw new RuntimeException(String.format("Cannot recover SSTable %s due to version mismatch. (current version is %s).", desc.toString(), "g"));
        }
        return new Builder(desc, type);
    }

    static class IndexWriter {
        private final BufferedRandomAccessFile indexFile;
        public final Descriptor desc;
        public final IPartitioner partitioner;
        public final SegmentedFile.Builder builder;
        public final IndexSummary summary;
        public final BloomFilter bf;
        private FileMark mark;

        IndexWriter(Descriptor desc, IPartitioner part, long keyCount) throws IOException {
            this.desc = desc;
            this.partitioner = part;
            this.indexFile = new BufferedRandomAccessFile(new File(desc.filenameFor(SSTable.COMPONENT_INDEX)), "rw", 0x800000, true);
            this.builder = SegmentedFile.getBuilder(DatabaseDescriptor.getIndexAccessMode());
            this.summary = new IndexSummary(keyCount);
            this.bf = BloomFilter.getFilter(keyCount, 15);
        }

        public void afterAppend(DecoratedKey key, long dataPosition) throws IOException {
            this.bf.add(key.key);
            long indexPosition = this.indexFile.getFilePointer();
            ByteBufferUtil.writeWithShortLength(key.key, this.indexFile);
            this.indexFile.writeLong(dataPosition);
            if (logger.isTraceEnabled()) {
                logger.trace("wrote index of " + key + " at " + indexPosition);
            }
            this.summary.maybeAddEntry(key, indexPosition);
            this.builder.addPotentialBoundary(indexPosition);
        }

        public void close() throws IOException {
            FileOutputStream fos = new FileOutputStream(this.desc.filenameFor(SSTable.COMPONENT_FILTER));
            DataOutputStream stream = new DataOutputStream(fos);
            BloomFilter.serializer().serialize(this.bf, stream);
            stream.flush();
            fos.getFD().sync();
            stream.close();
            long position = this.indexFile.getFilePointer();
            this.indexFile.close();
            FileUtils.truncate(this.indexFile.getPath(), position);
            this.summary.complete();
        }

        public void mark() {
            this.mark = this.indexFile.mark();
        }

        public void reset() throws IOException {
            this.indexFile.reset(this.mark);
        }
    }

    static class CommutativeRowIndexer
    extends RowIndexer {
        protected BufferedRandomAccessFile writerDfile;

        CommutativeRowIndexer(Descriptor desc, ColumnFamilyStore cfs, OperationType type) throws IOException {
            super(desc, new BufferedRandomAccessFile(new File(desc.filenameFor(SSTable.COMPONENT_DATA)), "r", 0x800000, true), cfs, type);
            this.writerDfile = new BufferedRandomAccessFile(new File(desc.filenameFor(SSTable.COMPONENT_DATA)), "rw", 0x800000, true);
        }

        @Override
        protected long doIndexing() throws IOException {
            EstimatedHistogram rowSizes = SSTable.defaultRowHistogram();
            EstimatedHistogram columnCounts = SSTable.defaultColumnHistogram();
            long rows = 0L;
            CompactionController controller = new CompactionController(this.cfs, Collections.<SSTableReader>emptyList(), Integer.MAX_VALUE, true);
            while (!this.dfile.isEOF()) {
                DecoratedKey key = SSTableReader.decodeKey(StorageService.getPartitioner(), this.desc, ByteBufferUtil.readWithShortLength(this.dfile));
                long dataSize = SSTableReader.readRowSize(this.dfile, this.desc);
                SSTableIdentityIterator iter = new SSTableIdentityIterator(this.cfs.metadata, this.dfile, key, this.dfile.getFilePointer(), dataSize, true);
                AbstractCompactedRow row = controller.getCompactedRow(iter);
                this.updateCache(key, dataSize, row);
                rowSizes.add(dataSize);
                columnCounts.add(row.columnCount());
                this.iwriter.afterAppend(key, this.writerDfile.getFilePointer());
                ByteBufferUtil.writeWithShortLength(key.key, this.writerDfile);
                row.write(this.writerDfile);
                ++rows;
            }
            SSTableWriter.writeMetadata(this.desc, rowSizes, columnCounts, ReplayPosition.NONE);
            if (this.writerDfile.getFilePointer() != this.dfile.getFilePointer()) {
                this.writerDfile.setLength(this.writerDfile.getFilePointer());
            }
            this.writerDfile.sync();
            return rows;
        }

        @Override
        void close() throws IOException {
            super.close();
            this.writerDfile.close();
        }
    }

    static class RowIndexer {
        protected final Descriptor desc;
        public final BufferedRandomAccessFile dfile;
        private final OperationType type;
        protected IndexWriter iwriter;
        protected ColumnFamilyStore cfs;

        RowIndexer(Descriptor desc, ColumnFamilyStore cfs, OperationType type) throws IOException {
            this(desc, new BufferedRandomAccessFile(new File(desc.filenameFor(SSTable.COMPONENT_DATA)), "r", 0x800000, true), cfs, type);
        }

        protected RowIndexer(Descriptor desc, BufferedRandomAccessFile dfile, ColumnFamilyStore cfs, OperationType type) throws IOException {
            this.desc = desc;
            this.dfile = dfile;
            this.type = type;
            this.cfs = cfs;
        }

        long prepareIndexing() throws IOException {
            try {
                long estimatedRows = SSTable.estimateRowsFromData(this.desc, this.dfile);
                this.iwriter = new IndexWriter(this.desc, StorageService.getPartitioner(), estimatedRows);
                return estimatedRows;
            }
            catch (IOException e) {
                this.dfile.close();
                throw e;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        long index() throws IOException {
            try {
                long l = this.doIndexing();
                return l;
            }
            finally {
                try {
                    this.close();
                }
                catch (IOException e) {
                    throw new IOError(e);
                }
            }
        }

        void close() throws IOException {
            this.dfile.close();
            this.iwriter.close();
        }

        protected void updateCache(DecoratedKey key, long dataSize, AbstractCompactedRow row) throws IOException {
            ColumnFamily cached = this.cfs.getRawCachedRow(key);
            if (cached != null) {
                switch (this.type) {
                    case AES: {
                        ColumnFamily cf;
                        if (dataSize > (long)DatabaseDescriptor.getInMemoryCompactionLimit()) {
                            logger.warn("Found a cached row over the in memory compaction limit during post-streaming rebuilt; it is highly recommended to avoid huge row on column family with row cache enabled.");
                            this.cfs.invalidateCachedRow(key);
                            break;
                        }
                        if (row == null) {
                            long position = this.dfile.getFilePointer();
                            cf = ColumnFamily.create(this.cfs.metadata);
                            ColumnFamily.serializer().deserializeColumns(this.dfile, cf, true, true);
                            this.dfile.seek(position);
                        } else {
                            assert (row instanceof PrecompactedRow);
                            cf = ((PrecompactedRow)row).getFullColumnFamily();
                        }
                        this.cfs.updateRowCache(key, cf);
                        break;
                    }
                    default: {
                        this.cfs.invalidateCachedRow(key);
                    }
                }
            }
        }

        protected long doIndexing() throws IOException {
            EstimatedHistogram rowSizes = SSTable.defaultRowHistogram();
            EstimatedHistogram columnCounts = SSTable.defaultColumnHistogram();
            long rows = 0L;
            long rowPosition = 0L;
            while (rowPosition < this.dfile.length()) {
                DecoratedKey key = SSTableReader.decodeKey(StorageService.getPartitioner(), this.desc, ByteBufferUtil.readWithShortLength(this.dfile));
                this.iwriter.afterAppend(key, rowPosition);
                long dataSize = SSTableReader.readRowSize(this.dfile, this.desc);
                rowPosition = this.dfile.getFilePointer() + dataSize;
                IndexHelper.skipBloomFilter(this.dfile);
                IndexHelper.skipIndex(this.dfile);
                ColumnFamily.serializer().deserializeFromSSTableNoColumns(ColumnFamily.create(this.cfs.metadata), this.dfile);
                this.updateCache(key, dataSize, null);
                rowSizes.add(dataSize);
                columnCounts.add(this.dfile.readInt());
                this.dfile.seek(rowPosition);
                ++rows;
            }
            SSTableWriter.writeMetadata(this.desc, rowSizes, columnCounts, ReplayPosition.NONE);
            return rows;
        }
    }

    public static class Builder
    implements CompactionInfo.Holder {
        private final Descriptor desc;
        private final OperationType type;
        private final ColumnFamilyStore cfs;
        private RowIndexer indexer;

        public Builder(Descriptor desc, OperationType type) {
            this.desc = desc;
            this.type = type;
            this.cfs = Table.open(desc.ksname).getColumnFamilyStore(desc.cfname);
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            this.maybeOpenIndexer();
            try {
                return new CompactionInfo(this.desc.ksname, this.desc.cfname, CompactionType.SSTABLE_BUILD, this.indexer.dfile.getFilePointer(), this.indexer.dfile.length());
            }
            catch (IOException e) {
                throw new IOError(e);
            }
        }

        private void maybeOpenIndexer() {
            if (this.indexer != null) {
                return;
            }
            try {
                this.indexer = this.cfs.metadata.getDefaultValidator().isCommutative() ? new CommutativeRowIndexer(this.desc, this.cfs, this.type) : new RowIndexer(this.desc, this.cfs, this.type);
            }
            catch (IOException e) {
                throw new IOError(e);
            }
        }

        public SSTableReader build() throws IOException {
            if (this.cfs.isInvalid()) {
                return null;
            }
            this.maybeOpenIndexer();
            File ifile = new File(this.desc.filenameFor(SSTable.COMPONENT_INDEX));
            File ffile = new File(this.desc.filenameFor(SSTable.COMPONENT_FILTER));
            assert (!ifile.exists());
            assert (!ffile.exists());
            long estimatedRows = this.indexer.prepareIndexing();
            long rows = this.indexer.index();
            logger.debug("estimated row count was {} of real count", (Object)((double)estimatedRows / (double)rows));
            return SSTableReader.open(SSTableWriter.rename(this.desc, SSTable.componentsFor(this.desc, false)));
        }
    }
}

