/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.cql;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;
import org.antlr.runtime.ANTLRStringStream;
import org.antlr.runtime.CharStream;
import org.antlr.runtime.CommonTokenStream;
import org.antlr.runtime.RecognitionException;
import org.antlr.runtime.TokenSource;
import org.antlr.runtime.TokenStream;
import org.apache.cassandra.auth.Permission;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ConfigurationException;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.cql.AbstractModification;
import org.apache.cassandra.cql.AlterTableStatement;
import org.apache.cassandra.cql.BatchStatement;
import org.apache.cassandra.cql.CQLStatement;
import org.apache.cassandra.cql.CqlLexer;
import org.apache.cassandra.cql.CqlParser;
import org.apache.cassandra.cql.CreateColumnFamilyStatement;
import org.apache.cassandra.cql.CreateIndexStatement;
import org.apache.cassandra.cql.CreateKeyspaceStatement;
import org.apache.cassandra.cql.DeleteStatement;
import org.apache.cassandra.cql.DropIndexStatement;
import org.apache.cassandra.cql.Relation;
import org.apache.cassandra.cql.RelationType;
import org.apache.cassandra.cql.SelectStatement;
import org.apache.cassandra.cql.StatementType;
import org.apache.cassandra.cql.Term;
import org.apache.cassandra.cql.UpdateStatement;
import org.apache.cassandra.db.CounterColumn;
import org.apache.cassandra.db.IColumn;
import org.apache.cassandra.db.IMutation;
import org.apache.cassandra.db.RangeSliceCommand;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.SliceByNamesReadCommand;
import org.apache.cassandra.db.SliceFromReadCommand;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.db.context.CounterContext;
import org.apache.cassandra.db.filter.QueryPath;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.MarshalException;
import org.apache.cassandra.db.migration.AddColumnFamily;
import org.apache.cassandra.db.migration.AddKeyspace;
import org.apache.cassandra.db.migration.DropColumnFamily;
import org.apache.cassandra.db.migration.DropKeyspace;
import org.apache.cassandra.db.migration.Migration;
import org.apache.cassandra.db.migration.UpdateColumnFamily;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.RandomPartitioner;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.service.ClientState;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.thrift.CfDef;
import org.apache.cassandra.thrift.Column;
import org.apache.cassandra.thrift.ColumnDef;
import org.apache.cassandra.thrift.ConsistencyLevel;
import org.apache.cassandra.thrift.CqlResult;
import org.apache.cassandra.thrift.CqlResultType;
import org.apache.cassandra.thrift.CqlRow;
import org.apache.cassandra.thrift.IndexClause;
import org.apache.cassandra.thrift.IndexExpression;
import org.apache.cassandra.thrift.IndexOperator;
import org.apache.cassandra.thrift.IndexType;
import org.apache.cassandra.thrift.InvalidRequestException;
import org.apache.cassandra.thrift.KsDef;
import org.apache.cassandra.thrift.SchemaDisagreementException;
import org.apache.cassandra.thrift.SlicePredicate;
import org.apache.cassandra.thrift.SliceRange;
import org.apache.cassandra.thrift.ThriftValidation;
import org.apache.cassandra.thrift.TimedOutException;
import org.apache.cassandra.thrift.UnavailableException;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueryProcessor {
    private static final Logger logger = LoggerFactory.getLogger(QueryProcessor.class);
    private static final long timeLimitForSchemaAgreement = 10000L;
    public static final String DEFAULT_KEY_NAME = QueryProcessor.bufferToString(CFMetaData.DEFAULT_KEY_NAME);

    private static List<Row> getSlice(String keyspace, SelectStatement select) throws InvalidRequestException, TimedOutException, UnavailableException {
        QueryPath queryPath = new QueryPath(select.getColumnFamily());
        CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily());
        ArrayList<ReadCommand> commands = new ArrayList<ReadCommand>();
        if (!select.isColumnRange()) {
            List<ByteBuffer> columnNames = QueryProcessor.getColumnNames(select, metadata);
            QueryProcessor.validateColumnNames(columnNames);
            for (Term rawKey : select.getKeys()) {
                ByteBuffer key = rawKey.getByteBuffer(metadata.getKeyValidator());
                QueryProcessor.validateKey(key);
                commands.add(new SliceByNamesReadCommand(keyspace, key, queryPath, columnNames));
            }
        } else {
            AbstractType comparator = select.getComparator(keyspace);
            ByteBuffer start = select.getColumnStart().getByteBuffer(comparator);
            ByteBuffer finish = select.getColumnFinish().getByteBuffer(comparator);
            for (Term rawKey : select.getKeys()) {
                ByteBuffer key = rawKey.getByteBuffer(metadata.getKeyValidator());
                QueryProcessor.validateKey(key);
                QueryProcessor.validateSliceRange(metadata, start, finish, select.isColumnsReversed());
                commands.add(new SliceFromReadCommand(keyspace, key, queryPath, start, finish, select.isColumnsReversed(), select.getColumnsLimit()));
            }
        }
        try {
            return StorageProxy.read(commands, select.getConsistencyLevel());
        }
        catch (TimeoutException e) {
            throw new TimedOutException();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static List<ByteBuffer> getColumnNames(SelectStatement select, CFMetaData metadata) throws InvalidRequestException {
        String keyString = QueryProcessor.getKeyString(metadata);
        ArrayList<ByteBuffer> columnNames = new ArrayList<ByteBuffer>();
        for (Term column : select.getColumnNames()) {
            if (column.getText().equalsIgnoreCase(keyString)) continue;
            columnNames.add(column.getByteBuffer(metadata.comparator));
        }
        return columnNames;
    }

    private static List<Row> multiRangeSlice(String keyspace, SelectStatement select) throws TimedOutException, UnavailableException, InvalidRequestException {
        List<Row> rows;
        Object finishToken;
        IPartitioner p = StorageService.getPartitioner();
        AbstractType keyType = DatabaseDescriptor.getCFMetaData(keyspace, select.getColumnFamily()).getKeyValidator();
        ByteBuffer startKey = select.getKeyStart() != null ? select.getKeyStart().getByteBuffer(keyType) : new Term().getByteBuffer();
        ByteBuffer finishKey = select.getKeyFinish() != null ? select.getKeyFinish().getByteBuffer(keyType) : new Term().getByteBuffer();
        Object startToken = p.getToken(startKey);
        if (((Token)startToken).compareTo(finishToken = p.getToken(finishKey)) > 0 && !((Token)finishToken).equals(p.getMinimumToken())) {
            if (p instanceof RandomPartitioner) {
                throw new InvalidRequestException("Start key's md5 sorts after end key's md5. This is not allowed; you probably should not specify end key at all, under RandomPartitioner");
            }
            throw new InvalidRequestException("Start key must sort before (or equal to) finish key in your partitioner!");
        }
        Bounds bounds = new Bounds((Token)startToken, (Token)finishToken);
        CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily());
        SlicePredicate thriftSlicePredicate = QueryProcessor.slicePredicateFromSelect(select, metadata);
        QueryProcessor.validateSlicePredicate(metadata, thriftSlicePredicate);
        int limit = select.isKeyRange() && select.getKeyStart() != null ? select.getNumRecords() + 1 : select.getNumRecords();
        try {
            rows = StorageProxy.getRangeSlice(new RangeSliceCommand(keyspace, select.getColumnFamily(), null, thriftSlicePredicate, bounds, limit), select.getConsistencyLevel());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        catch (UnavailableException e) {
            throw new UnavailableException();
        }
        catch (TimeoutException e) {
            throw new TimedOutException();
        }
        if (select.getKeyStart() != null && !select.includeStartKey() && !rows.isEmpty() && rows.get((int)0).key.key.equals(startKey)) {
            rows.remove(0);
        }
        if (select.getKeyFinish() != null && !select.includeFinishKey() && !rows.isEmpty()) {
            int lastIndex = rows.size() - 1;
            if (rows.get((int)lastIndex).key.key.equals(finishKey)) {
                rows.remove(lastIndex);
            }
        }
        return rows.subList(0, select.getNumRecords() < rows.size() ? select.getNumRecords() : rows.size());
    }

    private static List<Row> getIndexedSlices(String keyspace, SelectStatement select) throws TimedOutException, UnavailableException, InvalidRequestException {
        List<Row> rows;
        CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily());
        SlicePredicate thriftSlicePredicate = QueryProcessor.slicePredicateFromSelect(select, metadata);
        QueryProcessor.validateSlicePredicate(metadata, thriftSlicePredicate);
        ArrayList<IndexExpression> expressions = new ArrayList<IndexExpression>();
        for (Relation columnRelation : select.getColumnRelations()) {
            ByteBuffer entity = columnRelation.getEntity().getByteBuffer(metadata.comparator);
            ByteBuffer value = columnRelation.getValue().getByteBuffer(select.getValueValidator(keyspace, entity));
            expressions.add(new IndexExpression(entity, IndexOperator.valueOf((String)columnRelation.operator().toString()), value));
        }
        AbstractType keyType = DatabaseDescriptor.getCFMetaData(keyspace, select.getColumnFamily()).getKeyValidator();
        ByteBuffer startKey = !select.isKeyRange() ? new Term().getByteBuffer() : select.getKeyStart().getByteBuffer(keyType);
        IndexClause thriftIndexClause = new IndexClause(expressions, startKey, select.getNumRecords());
        try {
            rows = StorageProxy.scan(keyspace, select.getColumnFamily(), thriftIndexClause, thriftSlicePredicate, select.getConsistencyLevel());
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        catch (TimeoutException e) {
            throw new TimedOutException();
        }
        return rows;
    }

    private static void batchUpdate(ClientState clientState, List<UpdateStatement> updateStatements, ConsistencyLevel consistency) throws InvalidRequestException, UnavailableException, TimedOutException {
        String keyspace = clientState.getKeyspace();
        ArrayList<IMutation> rowMutations = new ArrayList<IMutation>();
        ArrayList<String> cfamsSeen = new ArrayList<String>();
        for (UpdateStatement update : updateStatements) {
            if (!cfamsSeen.contains(update.getColumnFamily())) {
                clientState.hasColumnFamilyAccess(update.getColumnFamily(), Permission.WRITE);
                cfamsSeen.add(update.getColumnFamily());
            }
            rowMutations.addAll(update.prepareRowMutations(keyspace, clientState));
        }
        try {
            StorageProxy.mutate(rowMutations, consistency);
        }
        catch (UnavailableException e) {
            throw new UnavailableException();
        }
        catch (TimeoutException e) {
            throw new TimedOutException();
        }
    }

    private static SlicePredicate slicePredicateFromSelect(SelectStatement select, CFMetaData metadata) throws InvalidRequestException {
        SlicePredicate thriftSlicePredicate = new SlicePredicate();
        if (select.isColumnRange() || select.getColumnNames().size() == 0) {
            SliceRange sliceRange = new SliceRange();
            sliceRange.start = select.getColumnStart().getByteBuffer(metadata.comparator);
            sliceRange.finish = select.getColumnFinish().getByteBuffer(metadata.comparator);
            sliceRange.reversed = select.isColumnsReversed();
            sliceRange.count = select.getColumnsLimit();
            thriftSlicePredicate.slice_range = sliceRange;
        } else {
            thriftSlicePredicate.column_names = QueryProcessor.getColumnNames(select, metadata);
        }
        return thriftSlicePredicate;
    }

    private static void validateSelect(String keyspace, SelectStatement select) throws InvalidRequestException {
        if (select.isCountOperation() && (select.isKeyRange() || select.getKeys().size() < 1)) {
            throw new InvalidRequestException("Counts can only be performed for a single record (Hint: KEY=term)");
        }
        if (!select.isKeyRange() && select.getKeyFinish() != null) {
            throw new InvalidRequestException("Key range clauses must include a start key (i.e. KEY > term)");
        }
        if (select.isKeyRange() && select.getKeys().size() > 0) {
            throw new InvalidRequestException("You cannot combine key range and by-key clauses in a SELECT");
        }
        if (select.isKeyRange() && select.getKeyFinish() != null && select.getColumnRelations().size() > 0) {
            throw new InvalidRequestException("You cannot combine key range and by-column clauses in a SELECT");
        }
        if (!select.isMultiKey() && select.getKeys().size() > 1) {
            throw new InvalidRequestException("You cannot use more than one KEY = in a SELECT");
        }
        AbstractType comparator = select.getComparator(keyspace);
        if (select.getColumnRelations().size() > 0) {
            SortedSet<ByteBuffer> indexed = Table.open(keyspace).getColumnFamilyStore(select.getColumnFamily()).getIndexedColumns();
            for (Relation relation : select.getColumnRelations()) {
                if (!relation.operator().equals((Object)RelationType.EQ) || !indexed.contains(relation.getEntity().getByteBuffer(comparator))) continue;
                return;
            }
            throw new InvalidRequestException("No indexed columns present in by-columns clause with \"equals\" operator");
        }
    }

    private static void applyMigrationOnStage(final Migration m) throws SchemaDisagreementException, InvalidRequestException {
        Future<Object> f = StageManager.getStage(Stage.MIGRATION).submit(new Callable<Object>(){

            @Override
            public Object call() throws Exception {
                m.apply();
                m.announce();
                return null;
            }
        });
        try {
            f.get();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            if (e.getCause() != null) {
                InvalidRequestException ex = new InvalidRequestException(e.getCause().getMessage());
                ex.initCause(e.getCause());
                throw ex;
            }
            InvalidRequestException ex = new InvalidRequestException(e.getMessage());
            ex.initCause((Throwable)e);
            throw ex;
        }
        QueryProcessor.validateSchemaIsSettled();
    }

    public static void validateKey(ByteBuffer key) throws InvalidRequestException {
        if (key == null || key.remaining() == 0) {
            throw new InvalidRequestException("Key may not be empty");
        }
        if (key.remaining() > 65535) {
            throw new InvalidRequestException("Key length of " + key.remaining() + " is longer than maximum of " + 65535);
        }
    }

    public static void validateKeyAlias(CFMetaData cfm, String key) throws InvalidRequestException {
        assert (key.toUpperCase().equals(key));
        String realKeyAlias = QueryProcessor.bufferToString(cfm.getKeyName()).toUpperCase();
        if (!realKeyAlias.equals(key)) {
            throw new InvalidRequestException(String.format("Expected key '%s' to be present in WHERE clause for '%s'", key, cfm.cfName));
        }
    }

    private static void validateColumnNames(Iterable<ByteBuffer> columns) throws InvalidRequestException {
        for (ByteBuffer name : columns) {
            if (name.remaining() > 65535) {
                throw new InvalidRequestException(String.format("column name is too long (%s > %s)", name.remaining(), 65535));
            }
            if (name.remaining() != 0) continue;
            throw new InvalidRequestException("zero-length column name");
        }
    }

    public static void validateColumnName(ByteBuffer column) throws InvalidRequestException {
        QueryProcessor.validateColumnNames(Arrays.asList(column));
    }

    public static void validateColumn(CFMetaData metadata, ByteBuffer name, ByteBuffer value) throws InvalidRequestException {
        QueryProcessor.validateColumnName(name);
        AbstractType validator = metadata.getValueValidator(name);
        try {
            if (validator != null) {
                validator.validate(value);
            }
        }
        catch (MarshalException me) {
            throw new InvalidRequestException(String.format("Invalid column value for column (name=%s); %s", ByteBufferUtil.bytesToHex(name), me.getMessage()));
        }
    }

    private static void validateSlicePredicate(CFMetaData metadata, SlicePredicate predicate) throws InvalidRequestException {
        if (predicate.slice_range != null) {
            QueryProcessor.validateSliceRange(metadata, predicate.slice_range);
        } else {
            QueryProcessor.validateColumnNames(predicate.column_names);
        }
    }

    private static void validateSliceRange(CFMetaData metadata, SliceRange range) throws InvalidRequestException {
        QueryProcessor.validateSliceRange(metadata, range.start, range.finish, range.reversed);
    }

    private static void validateSliceRange(CFMetaData metadata, ByteBuffer start, ByteBuffer finish, boolean reversed) throws InvalidRequestException {
        Comparator<ByteBuffer> orderedComparator;
        AbstractType comparator = metadata.getComparatorFor(null);
        Comparator<ByteBuffer> comparator2 = orderedComparator = reversed ? comparator.reverseComparator : comparator;
        if (start.remaining() > 0 && finish.remaining() > 0 && orderedComparator.compare(start, finish) > 0) {
            throw new InvalidRequestException("range finish must come after start in traversal order");
        }
    }

    private static void validateSchemaAgreement() throws SchemaDisagreementException {
        if (QueryProcessor.describeSchemaVersions().size() > 1) {
            throw new SchemaDisagreementException();
        }
    }

    private static Map<String, List<String>> describeSchemaVersions() {
        return Maps.filterKeys(StorageProxy.describeSchemaVersions(), (Predicate)Predicates.not((Predicate)Predicates.equalTo((Object)"UNREACHABLE")));
    }

    public static CqlResult process(String queryString, ClientState clientState) throws RecognitionException, UnavailableException, InvalidRequestException, TimedOutException, SchemaDisagreementException {
        logger.trace("CQL QUERY: {}", (Object)queryString);
        CQLStatement statement = QueryProcessor.getStatement(queryString);
        String keyspace = null;
        if (StatementType.requiresKeyspace.contains((Object)statement.type)) {
            keyspace = clientState.getKeyspace();
        }
        CqlResult result = new CqlResult();
        logger.debug("CQL statement type: {}", (Object)statement.type.toString());
        switch (statement.type) {
            case SELECT: {
                List<Row> rows;
                SelectStatement select = (SelectStatement)statement.statement;
                clientState.hasColumnFamilyAccess(select.getColumnFamily(), Permission.READ);
                CFMetaData metadata = ThriftValidation.validateColumnFamily(keyspace, select.getColumnFamily());
                select.extractKeyAliasFromColumns(metadata);
                if (select.getKeys().size() > 0) {
                    QueryProcessor.validateKeyAlias(metadata, select.getKeyAlias());
                }
                QueryProcessor.validateSelect(keyspace, select);
                if (!select.isKeyRange() && select.getKeys().size() > 0) {
                    rows = QueryProcessor.getSlice(keyspace, select);
                    if (select.isCountOperation()) {
                        result.type = CqlResultType.INT;
                        if (rows.size() > 0) {
                            result.setNum(rows.get((int)0).cf != null ? rows.get((int)0).cf.getSortedColumns().size() : 0);
                        } else {
                            result.setNum(0);
                        }
                        return result;
                    }
                } else {
                    rows = select.getKeyFinish() != null || select.getColumnRelations().size() == 0 ? QueryProcessor.multiRangeSlice(keyspace, select) : QueryProcessor.getIndexedSlices(keyspace, select);
                }
                ArrayList<CqlRow> cqlRows = new ArrayList<CqlRow>();
                result.type = CqlResultType.ROWS;
                for (Row row : rows) {
                    if (row.cf == null) continue;
                    List<Column> thriftColumns = QueryProcessor.extractThriftColumns(select, metadata, row);
                    CqlRow cqlRow = new CqlRow();
                    cqlRow.key = row.key.key;
                    cqlRow.columns = thriftColumns;
                    if (select.isColumnsReversed()) {
                        Collections.reverse(cqlRow.columns);
                    }
                    cqlRows.add(cqlRow);
                }
                result.rows = cqlRows;
                return result;
            }
            case INSERT: 
            case UPDATE: {
                UpdateStatement update = (UpdateStatement)statement.statement;
                QueryProcessor.batchUpdate(clientState, Collections.singletonList(update), update.getConsistencyLevel());
                result.type = CqlResultType.VOID;
                return result;
            }
            case BATCH: {
                BatchStatement batch = (BatchStatement)statement.statement;
                if (batch.getTimeToLive() != 0) {
                    throw new InvalidRequestException("Global TTL on the BATCH statement is not supported.");
                }
                for (AbstractModification up : batch.getStatements()) {
                    if (up.isSetConsistencyLevel()) {
                        throw new InvalidRequestException("Consistency level must be set on the BATCH, not individual statements");
                    }
                    if (!batch.isSetTimestamp() || !up.isSetTimestamp()) continue;
                    throw new InvalidRequestException("Timestamp must be set either on BATCH or individual statements");
                }
                try {
                    StorageProxy.mutate(batch.getMutations(keyspace, clientState), batch.getConsistencyLevel());
                }
                catch (UnavailableException e) {
                    throw new UnavailableException();
                }
                catch (TimeoutException e) {
                    throw new TimedOutException();
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case USE: {
                clientState.setKeyspace((String)statement.statement);
                result.type = CqlResultType.VOID;
                return result;
            }
            case TRUNCATE: {
                String columnFamily = (String)statement.statement;
                ThriftValidation.validateColumnFamily(keyspace, columnFamily);
                clientState.hasColumnFamilyAccess(columnFamily, Permission.WRITE);
                try {
                    StorageProxy.truncateBlocking(keyspace, columnFamily);
                }
                catch (TimeoutException e) {
                    throw (UnavailableException)new UnavailableException().initCause((Throwable)e);
                }
                catch (IOException e) {
                    throw (UnavailableException)new UnavailableException().initCause((Throwable)e);
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DELETE: {
                DeleteStatement delete = (DeleteStatement)statement.statement;
                try {
                    StorageProxy.mutate(delete.prepareRowMutations(keyspace, clientState), delete.getConsistencyLevel());
                }
                catch (TimeoutException e) {
                    throw new TimedOutException();
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_KEYSPACE: {
                CreateKeyspaceStatement create = (CreateKeyspaceStatement)statement.statement;
                create.validate();
                clientState.hasKeyspaceListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    KsDef ksd = new KsDef(create.getName(), create.getStrategyClass(), Collections.emptyList()).setStrategy_options(create.getStrategyOptions());
                    ThriftValidation.validateKsDef(ksd);
                    QueryProcessor.applyMigrationOnStage(new AddKeyspace(KSMetaData.fromThrift(ksd, new CFMetaData[0])));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_COLUMNFAMILY: {
                CreateColumnFamilyStatement createCf = (CreateColumnFamilyStatement)statement.statement;
                clientState.hasColumnFamilyListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    QueryProcessor.applyMigrationOnStage(new AddColumnFamily(createCf.getCFMetaData(keyspace)));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case CREATE_INDEX: {
                InvalidRequestException ex;
                CreateIndexStatement createIdx = (CreateIndexStatement)statement.statement;
                clientState.hasColumnFamilyListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                CFMetaData oldCfm = DatabaseDescriptor.getCFMetaData(CFMetaData.getId(keyspace, createIdx.getColumnFamily()));
                if (oldCfm == null) {
                    throw new InvalidRequestException("No such column family: " + createIdx.getColumnFamily());
                }
                boolean columnExists = false;
                ByteBuffer columnName = createIdx.getColumnName().getByteBuffer();
                CfDef cf_def = CFMetaData.convertToThrift(oldCfm);
                for (ColumnDef cd : cf_def.column_metadata) {
                    if (!cd.name.equals(columnName)) continue;
                    if (cd.index_type != null) {
                        throw new InvalidRequestException("Index already exists");
                    }
                    logger.debug("Updating column {} definition for index {}", (Object)oldCfm.comparator.getString(columnName), (Object)createIdx.getIndexName());
                    cd.setIndex_type(IndexType.KEYS);
                    cd.setIndex_name(createIdx.getIndexName());
                    columnExists = true;
                    break;
                }
                if (!columnExists) {
                    throw new InvalidRequestException("No column definition found for column " + oldCfm.comparator.getString(columnName));
                }
                CFMetaData.addDefaultIndexNames(cf_def);
                ThriftValidation.validateCfDef(cf_def, oldCfm);
                try {
                    QueryProcessor.applyMigrationOnStage(new UpdateColumnFamily(CFMetaData.convertToAvro(cf_def)));
                }
                catch (ConfigurationException e) {
                    ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DROP_INDEX: {
                DropIndexStatement dropIdx = (DropIndexStatement)statement.statement;
                clientState.hasColumnFamilyListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    QueryProcessor.applyMigrationOnStage(dropIdx.generateMutation(clientState.getKeyspace()));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.toString());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DROP_KEYSPACE: {
                String deleteKeyspace = (String)statement.statement;
                clientState.hasKeyspaceListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    QueryProcessor.applyMigrationOnStage(new DropKeyspace(deleteKeyspace));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case DROP_COLUMNFAMILY: {
                String deleteColumnFamily = (String)statement.statement;
                clientState.hasColumnFamilyListAccess(Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    QueryProcessor.applyMigrationOnStage(new DropColumnFamily(keyspace, deleteColumnFamily));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
            case ALTER_TABLE: {
                AlterTableStatement alterTable = (AlterTableStatement)statement.statement;
                System.out.println(alterTable);
                ThriftValidation.validateColumnFamily(keyspace, alterTable.columnFamily);
                clientState.hasColumnFamilyAccess(alterTable.columnFamily, Permission.WRITE);
                QueryProcessor.validateSchemaAgreement();
                try {
                    QueryProcessor.applyMigrationOnStage(new UpdateColumnFamily(alterTable.getCfDef(keyspace)));
                }
                catch (ConfigurationException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                catch (IOException e) {
                    InvalidRequestException ex = new InvalidRequestException(e.getMessage());
                    ex.initCause((Throwable)e);
                    throw ex;
                }
                result.type = CqlResultType.VOID;
                return result;
            }
        }
        return null;
    }

    private static List<Column> extractThriftColumns(SelectStatement select, CFMetaData metadata, Row row) {
        ArrayList<Column> thriftColumns = new ArrayList<Column>();
        if (select.isColumnRange()) {
            if (select.isWildcard()) {
                thriftColumns.add(new Column(metadata.getKeyName()).setValue(row.key.key).setTimestamp(-1L));
            }
            for (IColumn c : row.cf.getSortedColumns()) {
                if (c.isMarkedForDelete()) continue;
                thriftColumns.add(QueryProcessor.thriftify(c));
            }
        } else {
            String keyString = QueryProcessor.getKeyString(metadata);
            for (Term term : select.getColumnNames()) {
                ByteBuffer name;
                if (term.getText().equalsIgnoreCase(keyString)) {
                    ByteBuffer requestedKey = ByteBufferUtil.bytes(term.getText());
                    thriftColumns.add(new Column(requestedKey).setValue(row.key.key).setTimestamp(-1L));
                    continue;
                }
                try {
                    name = term.getByteBuffer(metadata.comparator);
                }
                catch (InvalidRequestException e) {
                    throw new AssertionError((Object)e);
                }
                IColumn c = row.cf.getColumn(name);
                if (c == null || c.isMarkedForDelete()) {
                    thriftColumns.add(new Column().setName(name));
                    continue;
                }
                thriftColumns.add(QueryProcessor.thriftify(c));
            }
        }
        return thriftColumns;
    }

    private static Column thriftify(IColumn c) {
        ByteBuffer value = c instanceof CounterColumn ? ByteBufferUtil.bytes(CounterContext.instance().total(c.value())) : c.value();
        return new Column(c.name()).setValue(value).setTimestamp(c.timestamp());
    }

    private static String getKeyString(CFMetaData metadata) {
        String keyString;
        try {
            keyString = ByteBufferUtil.string(metadata.getKeyName());
        }
        catch (CharacterCodingException e) {
            throw new AssertionError((Object)e);
        }
        return keyString;
    }

    private static CQLStatement getStatement(String queryStr) throws InvalidRequestException, RecognitionException {
        ANTLRStringStream stream = new ANTLRStringStream(queryStr);
        CqlLexer lexer = new CqlLexer((CharStream)stream);
        CommonTokenStream tokenStream = new CommonTokenStream((TokenSource)lexer);
        CqlParser parser = new CqlParser((TokenStream)tokenStream);
        CQLStatement statement = parser.query();
        lexer.throwLastRecognitionError();
        parser.throwLastRecognitionError();
        return statement;
    }

    private static void validateSchemaIsSettled() throws SchemaDisagreementException {
        long limit = System.currentTimeMillis() + 10000L;
        block0: while (limit - System.currentTimeMillis() >= 0L) {
            String currentVersionId = DatabaseDescriptor.getDefsVersion().toString();
            for (String version : QueryProcessor.describeSchemaVersions().keySet()) {
                if (version.equals(currentVersionId)) continue;
                continue block0;
            }
            return;
        }
        throw new SchemaDisagreementException();
    }

    private static String bufferToString(ByteBuffer string) {
        try {
            return ByteBufferUtil.string(string);
        }
        catch (CharacterCodingException e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }
}

