/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.FileMark;
import org.apache.cassandra.utils.CLibrary;

public class BufferedRandomAccessFile
extends RandomAccessFile
implements FileDataInput {
    private static final long MAX_BYTES_IN_PAGE_CACHE = (long)Math.pow(2.0, 27.0);
    private final String filePath;
    public static final int DEFAULT_BUFFER_SIZE = 65535;
    private boolean isDirty;
    private boolean syncNeeded;
    private boolean hitEOF = false;
    private ByteBuffer buffer;
    private long bufferOffset;
    private long bufferEnd;
    private long current = 0L;
    private long maxBufferSize;
    private final long fileLength;
    private final FileChannel channel;
    private long markedPointer;
    private int fd;
    private final boolean skipCache;
    private long bytesSinceCacheFlush = 0L;
    private long minBufferOffset = Long.MAX_VALUE;

    public BufferedRandomAccessFile(String name, String mode) throws IOException {
        this(new File(name), mode, 0);
    }

    public BufferedRandomAccessFile(String name, String mode, int bufferSize) throws IOException {
        this(new File(name), mode, bufferSize);
    }

    public BufferedRandomAccessFile(File file, String mode) throws IOException {
        this(file, mode, 0);
    }

    public BufferedRandomAccessFile(File file, String mode, int bufferSize) throws IOException {
        this(file, mode, bufferSize, false);
    }

    public BufferedRandomAccessFile(File file, String mode, int bufferSize, boolean skipCache) throws IOException {
        super(file, mode);
        this.skipCache = skipCache;
        this.channel = super.getChannel();
        this.filePath = file.getAbsolutePath();
        this.maxBufferSize = Math.max(bufferSize, 65535);
        this.buffer = ByteBuffer.allocate((int)this.maxBufferSize);
        this.fileLength = mode.equals("r") ? this.channel.size() : -1L;
        this.bufferEnd = this.reBuffer();
        this.fd = CLibrary.getfd(this.getFD());
    }

    public void sync() throws IOException {
        if (this.syncNeeded) {
            this.flush();
            this.channel.force(true);
            if (this.skipCache) {
                CLibrary.trySkipCache(this.fd, 0, 0);
                this.minBufferOffset = Long.MAX_VALUE;
                this.bytesSinceCacheFlush = 0L;
            }
            this.syncNeeded = false;
        }
    }

    public void flush() throws IOException {
        if (this.isDirty) {
            if (this.channel.position() != this.bufferOffset) {
                this.channel.position(this.bufferOffset);
            }
            int lengthToWrite = (int)(this.bufferEnd - this.bufferOffset);
            super.write(this.buffer.array(), 0, lengthToWrite);
            if (this.skipCache) {
                this.bytesSinceCacheFlush += (long)lengthToWrite;
                if (this.bufferOffset < this.minBufferOffset) {
                    this.minBufferOffset = this.bufferOffset;
                }
                if (this.bytesSinceCacheFlush >= MAX_BYTES_IN_PAGE_CACHE) {
                    CLibrary.trySkipCache(this.fd, (int)this.minBufferOffset, 0);
                    this.minBufferOffset = this.bufferOffset;
                    this.bytesSinceCacheFlush = 0L;
                }
            }
            this.isDirty = false;
        }
    }

    private long reBuffer() throws IOException {
        this.flush();
        this.buffer.clear();
        this.bufferOffset = this.current;
        if (this.bufferOffset > this.channel.size()) {
            this.buffer.rewind();
            this.bufferEnd = this.bufferOffset;
            this.hitEOF = true;
            return 0L;
        }
        if (this.bufferOffset < this.minBufferOffset) {
            this.minBufferOffset = this.bufferOffset;
        }
        this.channel.position(this.bufferOffset);
        long bytesRead = this.channel.read(this.buffer);
        this.hitEOF = bytesRead < this.maxBufferSize;
        this.bufferEnd = this.bufferOffset + bytesRead;
        this.buffer.rewind();
        this.bytesSinceCacheFlush += bytesRead;
        if (this.skipCache && this.bytesSinceCacheFlush >= MAX_BYTES_IN_PAGE_CACHE) {
            CLibrary.trySkipCache(this.fd, (int)this.minBufferOffset, 0);
            this.bytesSinceCacheFlush = 0L;
            this.minBufferOffset = Long.MAX_VALUE;
        }
        return bytesRead;
    }

    @Override
    public int read() throws IOException {
        if (this.isEOF()) {
            return -1;
        }
        if (this.current < this.bufferOffset || this.current >= this.bufferEnd) {
            this.reBuffer();
            if (this.current == this.bufferEnd && this.hitEOF) {
                return -1;
            }
        }
        byte result = this.buffer.get();
        ++this.current;
        return result & 0xFF;
    }

    @Override
    public int read(byte[] buffer) throws IOException {
        return this.read(buffer, 0, buffer.length);
    }

    @Override
    public int read(byte[] buff, int offset, int length) throws IOException {
        int bytesCount = 0;
        while (length > 0) {
            int bytesRead = this.readAtMost(buff, offset, length);
            if (bytesRead == -1) {
                return -1;
            }
            offset += bytesRead;
            length -= bytesRead;
            bytesCount += bytesRead;
        }
        return bytesCount;
    }

    private int readAtMost(byte[] buff, int offset, int length) throws IOException {
        if ((long)length >= this.bufferEnd && this.hitEOF) {
            return -1;
        }
        int left = (int)this.maxBufferSize - this.buffer.position();
        if (this.current < this.bufferOffset || left < length) {
            this.reBuffer();
        }
        length = Math.min(length, (int)(this.maxBufferSize - (long)this.buffer.position()));
        this.buffer.get(buff, offset, length);
        this.current += (long)length;
        return length;
    }

    @Override
    public ByteBuffer readBytes(int length) throws IOException {
        assert (length >= 0) : "buffer length should not be negative: " + length;
        byte[] buff = new byte[length];
        this.readFully(buff);
        return ByteBuffer.wrap(buff);
    }

    @Override
    public void write(int val) throws IOException {
        byte[] b = new byte[]{(byte)val};
        this.write(b, 0, b.length);
    }

    @Override
    public void write(byte[] b) throws IOException {
        this.write(b, 0, b.length);
    }

    @Override
    public void write(byte[] buff, int offset, int length) throws IOException {
        while (length > 0) {
            int n = this.writeAtMost(buff, offset, length);
            offset += n;
            length -= n;
            this.isDirty = true;
            this.syncNeeded = true;
        }
    }

    private int writeAtMost(byte[] buff, int offset, int length) throws IOException {
        int left = (int)this.maxBufferSize - this.buffer.position();
        if (this.current < this.bufferOffset || left < length) {
            this.reBuffer();
        }
        length = Math.min(length, (int)(this.maxBufferSize - (long)this.buffer.position()));
        this.buffer.put(buff, offset, length);
        this.current += (long)length;
        if (this.current > this.bufferEnd) {
            this.bufferEnd = this.current;
        }
        return length;
    }

    @Override
    public void seek(long newPosition) throws IOException {
        this.current = newPosition;
        if (newPosition >= this.bufferEnd || newPosition < this.bufferOffset) {
            this.reBuffer();
        }
        int delta = (int)(newPosition - this.bufferOffset);
        this.buffer.position(delta);
    }

    @Override
    public int skipBytes(int count) throws IOException {
        if (count > 0) {
            long eof;
            long currentPos = this.getFilePointer();
            int newCount = (int)(currentPos + (long)count > (eof = this.length()) ? eof - currentPos : (long)count);
            this.seek(currentPos + (long)newCount);
            return newCount;
        }
        return 0;
    }

    @Override
    public long length() throws IOException {
        return this.fileLength == -1L ? Math.max(this.current, this.channel.size()) : this.fileLength;
    }

    @Override
    public long getFilePointer() {
        return this.bufferOffset + (long)this.buffer.position();
    }

    @Override
    public String getPath() {
        return this.filePath;
    }

    @Override
    public boolean isEOF() throws IOException {
        return this.getFilePointer() == this.length();
    }

    @Override
    public long bytesRemaining() throws IOException {
        return this.length() - this.getFilePointer();
    }

    @Override
    public void close() throws IOException {
        this.sync();
        this.buffer = null;
        if (this.skipCache && this.bytesSinceCacheFlush > 0L) {
            CLibrary.trySkipCache(this.fd, 0, 0);
        }
        super.close();
    }

    public void reset() throws IOException {
        this.seek(this.markedPointer);
    }

    public int bytesPastMark() {
        long bytes = this.getFilePointer() - this.markedPointer;
        assert (bytes >= 0L);
        if (bytes > Integer.MAX_VALUE) {
            throw new UnsupportedOperationException("Overflow: " + bytes);
        }
        return (int)bytes;
    }

    @Override
    public FileMark mark() {
        this.markedPointer = this.getFilePointer();
        return new BufferedRandomAccessFileMark(this.markedPointer);
    }

    @Override
    public void reset(FileMark mark) throws IOException {
        assert (mark instanceof BufferedRandomAccessFileMark);
        this.seek(((BufferedRandomAccessFileMark)mark).pointer);
    }

    @Override
    public int bytesPastMark(FileMark mark) {
        assert (mark instanceof BufferedRandomAccessFileMark);
        long bytes = this.getFilePointer() - ((BufferedRandomAccessFileMark)mark).pointer;
        assert (bytes >= 0L);
        if (bytes > Integer.MAX_VALUE) {
            throw new UnsupportedOperationException("Overflow: " + bytes);
        }
        return (int)bytes;
    }

    protected static class BufferedRandomAccessFileMark
    implements FileMark {
        long pointer;

        public BufferedRandomAccessFileMark(long pointer) {
            this.pointer = pointer;
        }
    }
}

