/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.docling;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.InvalidPayloadException;
import org.apache.camel.component.docling.DoclingConfiguration;
import org.apache.camel.component.docling.DoclingEndpoint;
import org.apache.camel.component.docling.DoclingOperations;
import org.apache.camel.support.DefaultProducer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DoclingProducer
extends DefaultProducer {
    private static final Logger LOG = LoggerFactory.getLogger(DoclingProducer.class);
    private DoclingEndpoint endpoint;
    private DoclingConfiguration configuration;

    public DoclingProducer(DoclingEndpoint endpoint) {
        super((Endpoint)endpoint);
        this.endpoint = endpoint;
        this.configuration = endpoint.getConfiguration();
    }

    public void process(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer processing exchange with message ID: {}", (Object)exchange.getExchangeId());
        DoclingOperations operation = this.getOperation(exchange);
        LOG.debug("DoclingProducer performing operation: {}", (Object)operation);
        switch (operation) {
            case CONVERT_TO_MARKDOWN: {
                this.processConvertToMarkdown(exchange);
                break;
            }
            case CONVERT_TO_HTML: {
                this.processConvertToHTML(exchange);
                break;
            }
            case CONVERT_TO_JSON: {
                this.processConvertToJSON(exchange);
                break;
            }
            case EXTRACT_TEXT: {
                this.processExtractText(exchange);
                break;
            }
            case EXTRACT_STRUCTURED_DATA: {
                this.processExtractStructuredData(exchange);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported operation: " + String.valueOf((Object)operation));
            }
        }
    }

    private DoclingOperations getOperation(Exchange exchange) {
        DoclingOperations operation = (DoclingOperations)((Object)exchange.getIn().getHeader("CamelDoclingOperation", DoclingOperations.class));
        if (operation == null) {
            operation = this.configuration.getOperation();
        }
        return operation;
    }

    private void processConvertToMarkdown(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer converting to markdown");
        String inputPath = this.getInputPath(exchange);
        exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "markdown", exchange));
    }

    private void processConvertToHTML(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer converting to HTML");
        String inputPath = this.getInputPath(exchange);
        exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "html", exchange));
    }

    private void processConvertToJSON(Exchange exchange) throws Exception {
        String inputPath = this.getInputPath(exchange);
        exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "json", exchange));
    }

    private void processExtractText(Exchange exchange) throws Exception {
        String inputPath = this.getInputPath(exchange);
        exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "text", exchange));
    }

    private void processExtractStructuredData(Exchange exchange) throws Exception {
        String inputPath = this.getInputPath(exchange);
        exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "json", exchange));
    }

    private String getInputPath(Exchange exchange) throws InvalidPayloadException, IOException {
        String inputPath = (String)exchange.getIn().getHeader("CamelDoclingInputFilePath", String.class);
        if (inputPath != null) {
            this.validateFileSize(inputPath);
            return inputPath;
        }
        Object body = exchange.getIn().getBody();
        if (body instanceof String) {
            String content = (String)body;
            if (content.startsWith("/") || content.contains("\\")) {
                this.validateFileSize(content);
                return content;
            }
            Path tempFile = Files.createTempFile("docling-", ".tmp", new FileAttribute[0]);
            Files.write(tempFile, content.getBytes(), new OpenOption[0]);
            this.validateFileSize(tempFile.toString());
            return tempFile.toString();
        }
        if (body instanceof byte[]) {
            byte[] content = (byte[])body;
            if ((long)content.length > this.configuration.getMaxFileSize()) {
                throw new IllegalArgumentException("File size exceeds maximum allowed size: " + this.configuration.getMaxFileSize());
            }
            Path tempFile = Files.createTempFile("docling-", ".tmp", new FileAttribute[0]);
            Files.write(tempFile, content, new OpenOption[0]);
            return tempFile.toString();
        }
        if (body instanceof File) {
            File file = (File)body;
            this.validateFileSize(file.getAbsolutePath());
            return file.getAbsolutePath();
        }
        throw new InvalidPayloadException(exchange, String.class);
    }

    private void validateFileSize(String filePath) throws IOException {
        long fileSize;
        Path path = Paths.get(filePath, new String[0]);
        if (Files.exists(path, new LinkOption[0]) && (fileSize = Files.size(path)) > this.configuration.getMaxFileSize()) {
            throw new IllegalArgumentException("File size (" + fileSize + " bytes) exceeds maximum allowed size: " + this.configuration.getMaxFileSize());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String executeDoclingCommand(String inputPath, String outputFormat, Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer executing Docling command for input: {} with format: {}", (Object)inputPath, (Object)outputFormat);
        Path tempOutputDir = Files.createTempDirectory("docling-output", new FileAttribute[0]);
        try {
            List<String> command = this.buildDoclingCommand(inputPath, outputFormat, exchange, tempOutputDir.toString());
            LOG.debug("Executing Docling command: {}", command);
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            if (this.configuration.getWorkingDirectory() != null) {
                processBuilder.directory(new File(this.configuration.getWorkingDirectory()));
            }
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            StringBuilder error = new StringBuilder();
            try (BufferedReader outputReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                 BufferedReader errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream()));){
                String line;
                while ((line = outputReader.readLine()) != null) {
                    LOG.debug("Docling output: {}", (Object)line);
                    output.append(line).append("\n");
                }
                while ((line = errorReader.readLine()) != null) {
                    error.append(line).append("\n");
                }
            }
            boolean finished = process.waitFor(this.configuration.getProcessTimeout(), TimeUnit.MILLISECONDS);
            if (!finished) {
                process.destroyForcibly();
                throw new RuntimeException("Docling process timed out after " + this.configuration.getProcessTimeout() + " milliseconds");
            }
            int exitCode = process.exitValue();
            if (exitCode != 0) {
                throw new RuntimeException("Docling process failed with exit code " + exitCode + ". Error: " + error.toString());
            }
            String result = this.readGeneratedOutputFile(tempOutputDir, inputPath, outputFormat);
            if (!this.configuration.isContentInBody()) {
                result = this.moveOutputFileToFinalLocation(tempOutputDir, inputPath, outputFormat);
            }
            String string = result;
            return string;
        }
        finally {
            if (this.configuration.isContentInBody()) {
                this.deleteDirectory(tempOutputDir);
            }
        }
    }

    private String readGeneratedOutputFile(Path outputDir, String inputPath, String outputFormat) throws IOException {
        Path inputFilePath = Paths.get(inputPath, new String[0]);
        String baseName = inputFilePath.getFileName().toString();
        int lastDot = baseName.lastIndexOf(46);
        if (lastDot > 0) {
            baseName = baseName.substring(0, lastDot);
        }
        String extension = this.getOutputFileExtension(outputFormat);
        String expectedFileName = baseName + "." + extension;
        Path outputFile = outputDir.resolve(expectedFileName);
        Path actualOutputFile = null;
        if (Files.exists(outputFile, new LinkOption[0])) {
            actualOutputFile = outputFile;
        } else {
            try (Stream<Path> stream = Files.list(outputDir);){
                actualOutputFile = stream.findFirst().orElse(null);
                if (actualOutputFile == null || !Files.isRegularFile(actualOutputFile, new LinkOption[0])) {
                    throw new RuntimeException("No output file generated in: " + String.valueOf(outputDir));
                }
            }
        }
        if (this.configuration.isContentInBody()) {
            String content = Files.readString(actualOutputFile);
            try {
                Files.delete(actualOutputFile);
                LOG.debug("Deleted output file: {}", (Object)actualOutputFile);
            }
            catch (IOException e) {
                LOG.warn("Failed to delete output file: {}", (Object)actualOutputFile, (Object)e);
            }
            return content;
        }
        return actualOutputFile.toString();
    }

    private String moveOutputFileToFinalLocation(Path tempOutputDir, String inputPath, String outputFormat) throws IOException {
        String extension;
        String expectedFileName;
        Path tempOutputFile;
        Path inputFilePath = Paths.get(inputPath, new String[0]);
        String baseName = inputFilePath.getFileName().toString();
        int lastDot = baseName.lastIndexOf(46);
        if (lastDot > 0) {
            baseName = baseName.substring(0, lastDot);
        }
        if (!Files.exists(tempOutputFile = tempOutputDir.resolve(expectedFileName = baseName + "." + (extension = this.getOutputFileExtension(outputFormat))), new LinkOption[0])) {
            try (Stream<Path> stream = Files.list(tempOutputDir);){
                tempOutputFile = stream.findFirst().orElse(null);
                if (tempOutputFile == null || !Files.isRegularFile(tempOutputFile, new LinkOption[0])) {
                    throw new RuntimeException("No output file generated in: " + String.valueOf(tempOutputDir));
                }
            }
        }
        Path finalOutputFile = inputFilePath.getParent().resolve(tempOutputFile.getFileName());
        int counter = 1;
        while (Files.exists(finalOutputFile, new LinkOption[0])) {
            String nameWithoutExt = baseName;
            String ext = extension;
            finalOutputFile = inputFilePath.getParent().resolve(nameWithoutExt + "_" + counter + "." + ext);
            ++counter;
        }
        Files.move(tempOutputFile, finalOutputFile, new CopyOption[0]);
        LOG.debug("Moved output file from {} to {}", (Object)tempOutputFile, (Object)finalOutputFile);
        return finalOutputFile.toString();
    }

    private String getOutputFileExtension(String outputFormat) {
        switch (outputFormat.toLowerCase()) {
            case "markdown": 
            case "md": {
                return "md";
            }
            case "html": {
                return "html";
            }
            case "json": {
                return "json";
            }
            case "text": {
                return "txt";
            }
        }
        return "md";
    }

    private void deleteDirectory(Path directory) {
        try {
            if (Files.exists(directory, new LinkOption[0])) {
                Files.walk(directory, new FileVisitOption[0]).sorted((a, b) -> b.compareTo((Path)a)).forEach(path -> {
                    try {
                        Files.delete(path);
                    }
                    catch (IOException e) {
                        LOG.warn("Failed to delete temporary file: {}", path, (Object)e);
                    }
                });
            }
        }
        catch (IOException e) {
            LOG.warn("Failed to clean up temporary directory: {}", (Object)directory, (Object)e);
        }
    }

    private List<String> buildDoclingCommand(String inputPath, String outputFormat, Exchange exchange, String outputDirectory) {
        ArrayList<String> command = new ArrayList<String>();
        command.add(this.configuration.getDoclingCommand());
        this.addCustomArguments(command, exchange);
        this.addOutputFormatArguments(command, outputFormat);
        this.addOcrArguments(command);
        this.addLayoutArguments(command);
        this.addOutputDirectoryArguments(command, exchange, outputDirectory);
        command.add(inputPath);
        return command;
    }

    private void addCustomArguments(List<String> command, Exchange exchange) {
        List customArgs = (List)exchange.getIn().getHeader("CamelDoclingCustomArguments", List.class);
        if (customArgs != null && !customArgs.isEmpty()) {
            LOG.debug("Adding custom Docling arguments: {}", (Object)customArgs);
            command.addAll(customArgs);
        }
    }

    private void addOutputFormatArguments(List<String> command, String outputFormat) {
        if (outputFormat != null && !outputFormat.isEmpty()) {
            command.add("--to");
            command.add(this.mapToDoclingFormat(outputFormat));
        }
    }

    private void addOcrArguments(List<String> command) {
        if (!this.configuration.isEnableOCR()) {
            command.add("--no-ocr");
        } else if (this.configuration.getOcrLanguage() != null) {
            command.add("--ocr-lang");
            command.add(this.configuration.getOcrLanguage());
        }
    }

    private void addLayoutArguments(List<String> command) {
        if (this.configuration.isIncludeLayoutInfo()) {
            command.add("--show-layout");
        }
    }

    private void addOutputDirectoryArguments(List<String> command, Exchange exchange, String outputDirectory) {
        String outputPath = (String)exchange.getIn().getHeader("CamelDoclingOutputFilePath", String.class);
        if (outputPath != null) {
            command.add("--output");
            command.add(outputPath);
        } else {
            command.add("--output");
            command.add(outputDirectory);
        }
    }

    private String mapToDoclingFormat(String outputFormat) {
        switch (outputFormat.toLowerCase()) {
            case "markdown": {
                return "md";
            }
            case "html": {
                return "html";
            }
            case "json": {
                return "json";
            }
            case "text": {
                return "text";
            }
        }
        return "md";
    }
}

