/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.gcp.bigtable;

import com.google.auth.Credentials;
import com.google.bigtable.v2.MutateRowResponse;
import com.google.bigtable.v2.Mutation;
import com.google.bigtable.v2.Row;
import com.google.bigtable.v2.RowFilter;
import com.google.bigtable.v2.SampleRowKeysResponse;
import com.google.cloud.bigtable.config.BigtableOptions;
import com.google.cloud.bigtable.config.CredentialOptions;
import com.google.cloud.bigtable.config.RetryOptions;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.protobuf.ByteString;
import io.grpc.Status;
import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.concurrent.ConcurrentLinkedQueue;
import javax.annotation.Nullable;
import org.apache.beam.sdk.Pipeline;
import org.apache.beam.sdk.annotations.Experimental;
import org.apache.beam.sdk.coders.Coder;
import org.apache.beam.sdk.coders.protobuf.ProtoCoder;
import org.apache.beam.sdk.io.BoundedSource;
import org.apache.beam.sdk.io.gcp.bigtable.BigtableService;
import org.apache.beam.sdk.io.gcp.bigtable.BigtableServiceImpl;
import org.apache.beam.sdk.io.range.ByteKey;
import org.apache.beam.sdk.io.range.ByteKeyRange;
import org.apache.beam.sdk.io.range.ByteKeyRangeTracker;
import org.apache.beam.sdk.options.GcpOptions;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.transforms.DoFn;
import org.apache.beam.sdk.transforms.PTransform;
import org.apache.beam.sdk.transforms.ParDo;
import org.apache.beam.sdk.transforms.SerializableFunction;
import org.apache.beam.sdk.transforms.display.DisplayData;
import org.apache.beam.sdk.transforms.display.HasDisplayData;
import org.apache.beam.sdk.util.ReleaseInfo;
import org.apache.beam.sdk.values.KV;
import org.apache.beam.sdk.values.PBegin;
import org.apache.beam.sdk.values.PCollection;
import org.apache.beam.sdk.values.PDone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Experimental
public class BigtableIO {
    private static final Logger LOG = LoggerFactory.getLogger(BigtableIO.class);

    @Experimental
    public static Read read() {
        return new Read(null, "", ByteKeyRange.ALL_KEYS, null, null);
    }

    @Experimental
    public static Write write() {
        return new Write(null, "", null);
    }

    private BigtableIO() {
    }

    private static String getUserAgent() {
        String javaVersion = System.getProperty("java.specification.version");
        ReleaseInfo info = ReleaseInfo.getReleaseInfo();
        return String.format("%s/%s (%s); %s", info.getName(), info.getVersion(), javaVersion, "0.3.0");
    }

    private static RetryOptions.Builder retryOptionsToBuilder(RetryOptions options) {
        RetryOptions.Builder builder = new RetryOptions.Builder();
        builder.setEnableRetries(options.enableRetries());
        builder.setInitialBackoffMillis(options.getInitialBackoffMillis());
        builder.setBackoffMultiplier(options.getBackoffMultiplier());
        builder.setMaxElapsedBackoffMillis(options.getMaxElaspedBackoffMillis());
        builder.setStreamingBufferSize(options.getStreamingBufferSize());
        builder.setStreamingBatchSize(options.getStreamingBatchSize());
        builder.setReadPartialRowTimeoutMillis(options.getReadPartialRowTimeoutMillis());
        builder.setMaxScanTimeoutRetries(options.getMaxScanTimeoutRetries());
        builder.setAllowRetriesWithoutTimestamp(options.allowRetriesWithoutTimestamp());
        for (Status.Code code : Status.Code.values()) {
            if (!options.isRetryable(code)) continue;
            builder.addStatusToRetryOn(code);
        }
        return builder;
    }

    static class BigtableWriteException
    extends IOException {
        public BigtableWriteException(KV<ByteString, Iterable<Mutation>> record, Throwable cause) {
            super(String.format("Error mutating row %s with mutations %s", ((ByteString)record.getKey()).toStringUtf8(), record.getValue()), cause);
        }
    }

    private static class BigtableReader
    extends BoundedSource.BoundedReader<Row> {
        private BigtableSource source;
        private BigtableService service;
        private BigtableService.Reader reader;
        private final ByteKeyRangeTracker rangeTracker;
        private long recordsReturned;

        public BigtableReader(BigtableSource source, BigtableService service) {
            this.source = source;
            this.service = service;
            this.rangeTracker = ByteKeyRangeTracker.of((ByteKeyRange)source.getRange());
        }

        public boolean start() throws IOException {
            boolean hasRecord;
            this.reader = this.service.createReader(this.getCurrentSource());
            boolean bl = hasRecord = this.reader.start() && this.rangeTracker.tryReturnRecordAt(true, ByteKey.of((ByteString)this.reader.getCurrentRow().getKey())) || this.rangeTracker.markDone();
            if (hasRecord) {
                ++this.recordsReturned;
            }
            return hasRecord;
        }

        public synchronized BigtableSource getCurrentSource() {
            return this.source;
        }

        public boolean advance() throws IOException {
            boolean hasRecord;
            boolean bl = hasRecord = this.reader.advance() && this.rangeTracker.tryReturnRecordAt(true, ByteKey.of((ByteString)this.reader.getCurrentRow().getKey())) || this.rangeTracker.markDone();
            if (hasRecord) {
                ++this.recordsReturned;
            }
            return hasRecord;
        }

        public Row getCurrent() throws NoSuchElementException {
            return this.reader.getCurrentRow();
        }

        public void close() throws IOException {
            LOG.info("Closing reader after reading {} records.", (Object)this.recordsReturned);
            if (this.reader != null) {
                this.reader.close();
                this.reader = null;
            }
        }

        public final Double getFractionConsumed() {
            return this.rangeTracker.getFractionConsumed();
        }

        public final long getSplitPointsConsumed() {
            return this.rangeTracker.getSplitPointsConsumed();
        }

        public final synchronized BigtableSource splitAtFraction(double fraction) {
            ByteKey splitKey;
            try {
                splitKey = this.rangeTracker.getRange().interpolateKey(fraction);
            }
            catch (IllegalArgumentException e) {
                LOG.info("%s: Failed to interpolate key for fraction %s.", (Object)this.rangeTracker.getRange(), (Object)fraction);
                return null;
            }
            LOG.debug("Proposing to split {} at fraction {} (key {})", new Object[]{this.rangeTracker, fraction, splitKey});
            BigtableSource primary = this.source.withEndKey(splitKey);
            BigtableSource residual = this.source.withStartKey(splitKey);
            if (!this.rangeTracker.trySplitAtPosition(splitKey)) {
                return null;
            }
            this.source = primary;
            return residual;
        }
    }

    static class BigtableSource
    extends BoundedSource<Row> {
        private final SerializableFunction<PipelineOptions, BigtableService> serviceFactory;
        private final String tableId;
        @Nullable
        private final RowFilter filter;
        private final ByteKeyRange range;
        @Nullable
        private Long estimatedSizeBytes;
        @Nullable
        private transient List<SampleRowKeysResponse> sampleRowKeys;

        public BigtableSource(SerializableFunction<PipelineOptions, BigtableService> serviceFactory, String tableId, @Nullable RowFilter filter, ByteKeyRange range, @Nullable Long estimatedSizeBytes) {
            this.serviceFactory = serviceFactory;
            this.tableId = tableId;
            this.filter = filter;
            this.range = range;
            this.estimatedSizeBytes = estimatedSizeBytes;
        }

        public String toString() {
            return MoreObjects.toStringHelper(BigtableSource.class).add("tableId", (Object)this.tableId).add("filter", (Object)this.filter).add("range", (Object)this.range).add("estimatedSizeBytes", (Object)this.estimatedSizeBytes).toString();
        }

        protected BigtableSource withStartKey(ByteKey startKey) {
            Preconditions.checkNotNull((Object)startKey, (Object)"startKey");
            return new BigtableSource(this.serviceFactory, this.tableId, this.filter, this.range.withStartKey(startKey), this.estimatedSizeBytes);
        }

        protected BigtableSource withEndKey(ByteKey endKey) {
            Preconditions.checkNotNull((Object)endKey, (Object)"endKey");
            return new BigtableSource(this.serviceFactory, this.tableId, this.filter, this.range.withEndKey(endKey), this.estimatedSizeBytes);
        }

        protected BigtableSource withEstimatedSizeBytes(Long estimatedSizeBytes) {
            Preconditions.checkNotNull((Object)estimatedSizeBytes, (Object)"estimatedSizeBytes");
            return new BigtableSource(this.serviceFactory, this.tableId, this.filter, this.range, estimatedSizeBytes);
        }

        private List<SampleRowKeysResponse> getSampleRowKeys(PipelineOptions pipelineOptions) throws IOException {
            return ((BigtableService)this.serviceFactory.apply((Object)pipelineOptions)).getSampleRowKeys(this);
        }

        public List<BigtableSource> splitIntoBundles(long desiredBundleSizeBytes, PipelineOptions options) throws Exception {
            long maximumNumberOfSplits = 4000L;
            long sizeEstimate = this.getEstimatedSizeBytes(options);
            desiredBundleSizeBytes = Math.max(sizeEstimate / maximumNumberOfSplits, desiredBundleSizeBytes);
            return this.splitIntoBundlesBasedOnSamples(desiredBundleSizeBytes, this.getSampleRowKeys(options));
        }

        private List<BigtableSource> splitIntoBundlesBasedOnSamples(long desiredBundleSizeBytes, List<SampleRowKeysResponse> sampleRowKeys) {
            if (sampleRowKeys.isEmpty()) {
                LOG.info("Not splitting source {} because no sample row keys are available.", (Object)this);
                return Collections.singletonList(this);
            }
            LOG.info("About to split into bundles of size {} with sampleRowKeys length {} first element {}", new Object[]{desiredBundleSizeBytes, sampleRowKeys.size(), sampleRowKeys.get(0)});
            ByteKey lastEndKey = ByteKey.EMPTY;
            long lastOffset = 0L;
            ImmutableList.Builder splits = ImmutableList.builder();
            for (SampleRowKeysResponse response : sampleRowKeys) {
                ByteKey splitEndKey;
                ByteKey responseEndKey = ByteKey.of((ByteString)response.getRowKey());
                long responseOffset = response.getOffsetBytes();
                Preconditions.checkState((responseOffset >= lastOffset ? 1 : 0) != 0, (String)"Expected response byte offset %s to come after the last offset %s", (Object[])new Object[]{responseOffset, lastOffset});
                if (!this.range.overlaps(ByteKeyRange.of((ByteKey)lastEndKey, (ByteKey)responseEndKey)).booleanValue()) {
                    lastOffset = responseOffset;
                    lastEndKey = responseEndKey;
                    continue;
                }
                ByteKey splitStartKey = lastEndKey;
                if (splitStartKey.compareTo(this.range.getStartKey()) < 0) {
                    splitStartKey = this.range.getStartKey();
                }
                if (!this.range.containsKey(splitEndKey = responseEndKey).booleanValue()) {
                    splitEndKey = this.range.getEndKey();
                }
                long sampleSizeBytes = responseOffset - lastOffset;
                List<BigtableSource> subSplits = this.splitKeyRangeIntoBundleSizedSubranges(sampleSizeBytes, desiredBundleSizeBytes, ByteKeyRange.of((ByteKey)splitStartKey, (ByteKey)splitEndKey));
                splits.addAll(subSplits);
                lastEndKey = responseEndKey;
                lastOffset = responseOffset;
            }
            if (!lastEndKey.isEmpty() && (this.range.getEndKey().isEmpty() || lastEndKey.compareTo(this.range.getEndKey()) < 0)) {
                splits.add((Object)this.withStartKey(lastEndKey).withEndKey(this.range.getEndKey()));
            }
            ImmutableList ret = splits.build();
            LOG.info("Generated {} splits. First split: {}", (Object)ret.size(), ret.get(0));
            return ret;
        }

        public long getEstimatedSizeBytes(PipelineOptions options) throws IOException {
            if (this.estimatedSizeBytes == null) {
                this.estimatedSizeBytes = this.getEstimatedSizeBytesBasedOnSamples(this.getSampleRowKeys(options));
            }
            return this.estimatedSizeBytes;
        }

        private long getEstimatedSizeBytesBasedOnSamples(List<SampleRowKeysResponse> samples) {
            long estimatedSizeBytes = 0L;
            long lastOffset = 0L;
            ByteKey currentStartKey = ByteKey.EMPTY;
            for (SampleRowKeysResponse response : samples) {
                ByteKey currentEndKey = ByteKey.of((ByteString)response.getRowKey());
                long currentOffset = response.getOffsetBytes();
                if (!currentStartKey.isEmpty() && currentStartKey.equals((Object)currentEndKey)) {
                    lastOffset = currentOffset;
                    continue;
                }
                if (this.range.overlaps(ByteKeyRange.of((ByteKey)currentStartKey, (ByteKey)currentEndKey)).booleanValue()) {
                    estimatedSizeBytes += currentOffset - lastOffset;
                }
                currentStartKey = currentEndKey;
                lastOffset = currentOffset;
            }
            return estimatedSizeBytes;
        }

        public BoundedSource.BoundedReader<Row> createReader(PipelineOptions options) throws IOException {
            return new BigtableReader(this, (BigtableService)this.serviceFactory.apply((Object)options));
        }

        public void validate() {
            Preconditions.checkArgument((!this.tableId.isEmpty() ? 1 : 0) != 0, (Object)"tableId cannot be empty");
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            super.populateDisplayData(builder);
            builder.add(DisplayData.item((String)"tableId", (String)this.tableId).withLabel("Table ID"));
            if (this.filter != null) {
                builder.add(DisplayData.item((String)"rowFilter", (String)this.filter.toString()).withLabel("Table Row Filter"));
            }
        }

        public Coder<Row> getDefaultOutputCoder() {
            return ProtoCoder.of(Row.class);
        }

        private List<BigtableSource> splitKeyRangeIntoBundleSizedSubranges(long sampleSizeBytes, long desiredBundleSizeBytes, ByteKeyRange range) {
            LOG.debug("Subsplit for sampleSizeBytes {} and desiredBundleSizeBytes {}", (Object)sampleSizeBytes, (Object)desiredBundleSizeBytes);
            if (sampleSizeBytes <= desiredBundleSizeBytes) {
                return Collections.singletonList(this.withStartKey(range.getStartKey()).withEndKey(range.getEndKey()));
            }
            Preconditions.checkArgument((sampleSizeBytes > 0L ? 1 : 0) != 0, (String)"Sample size %s bytes must be greater than 0.", (Object[])new Object[]{sampleSizeBytes});
            Preconditions.checkArgument((desiredBundleSizeBytes > 0L ? 1 : 0) != 0, (String)"Desired bundle size %s bytes must be greater than 0.", (Object[])new Object[]{desiredBundleSizeBytes});
            int splitCount = (int)Math.ceil((double)sampleSizeBytes / (double)desiredBundleSizeBytes);
            List splitKeys = range.split(splitCount);
            ImmutableList.Builder splits = ImmutableList.builder();
            Iterator keys = splitKeys.iterator();
            ByteKey prev = (ByteKey)keys.next();
            while (keys.hasNext()) {
                ByteKey next = (ByteKey)keys.next();
                splits.add((Object)this.withStartKey(prev).withEndKey(next).withEstimatedSizeBytes(sampleSizeBytes / (long)splitCount));
                prev = next;
            }
            return splits.build();
        }

        public ByteKeyRange getRange() {
            return this.range;
        }

        public RowFilter getRowFilter() {
            return this.filter;
        }

        public String getTableId() {
            return this.tableId;
        }
    }

    @Experimental
    public static class Write
    extends PTransform<PCollection<KV<ByteString, Iterable<Mutation>>>, PDone> {
        @Nullable
        private final BigtableOptions options;
        private final String tableId;
        @Nullable
        private final BigtableService bigtableService;

        private Write(@Nullable BigtableOptions options, String tableId, @Nullable BigtableService bigtableService) {
            this.options = options;
            this.tableId = (String)Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            this.bigtableService = bigtableService;
        }

        public Write withBigtableOptions(BigtableOptions options) {
            Preconditions.checkNotNull((Object)options, (Object)"options");
            return this.withBigtableOptions(options.toBuilder());
        }

        public Write withBigtableOptions(BigtableOptions.Builder optionsBuilder) {
            Preconditions.checkNotNull((Object)optionsBuilder, (Object)"optionsBuilder");
            BigtableOptions options = optionsBuilder.build();
            RetryOptions retryOptions = options.getRetryOptions();
            BigtableOptions.Builder clonedBuilder = options.toBuilder().setBulkOptions(options.getBulkOptions().toBuilder().setUseBulkApi(true).build()).setRetryOptions(BigtableIO.retryOptionsToBuilder(retryOptions).setStreamingBatchSize(Math.min(retryOptions.getStreamingBatchSize(), retryOptions.getStreamingBufferSize() / 2)).build());
            BigtableOptions optionsWithAgent = clonedBuilder.setUserAgent(BigtableIO.getUserAgent()).build();
            return new Write(optionsWithAgent, this.tableId, this.bigtableService);
        }

        public Write withTableId(String tableId) {
            Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            return new Write(this.options, tableId, this.bigtableService);
        }

        public BigtableOptions getBigtableOptions() {
            return this.options;
        }

        public String getTableId() {
            return this.tableId;
        }

        public PDone expand(PCollection<KV<ByteString, Iterable<Mutation>>> input) {
            input.apply((PTransform)ParDo.of((DoFn)new BigtableWriterFn(this.tableId, new SerializableFunction<PipelineOptions, BigtableService>(){

                public BigtableService apply(PipelineOptions options) {
                    return Write.this.getBigtableService(options);
                }
            })));
            return PDone.in((Pipeline)input.getPipeline());
        }

        public void validate(PCollection<KV<ByteString, Iterable<Mutation>>> input) {
            Preconditions.checkArgument((this.options != null ? 1 : 0) != 0, (Object)"BigtableOptions not specified");
            Preconditions.checkArgument((!this.tableId.isEmpty() ? 1 : 0) != 0, (Object)"Table ID not specified");
            try {
                Preconditions.checkArgument((boolean)this.getBigtableService(input.getPipeline().getOptions()).tableExists(this.tableId), (String)"Table %s does not exist", (Object[])new Object[]{this.tableId});
            }
            catch (IOException e) {
                LOG.warn("Error checking whether table {} exists; proceeding.", (Object)this.tableId, (Object)e);
            }
        }

        Write withBigtableService(BigtableService bigtableService) {
            Preconditions.checkNotNull((Object)bigtableService, (Object)"bigtableService");
            return new Write(this.options, this.tableId, bigtableService);
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            super.populateDisplayData(builder);
            builder.add(DisplayData.item((String)"tableId", (String)this.tableId).withLabel("Table ID"));
            if (this.options != null) {
                builder.add(DisplayData.item((String)"bigtableOptions", (String)this.options.toString()).withLabel("Bigtable Options"));
            }
        }

        public String toString() {
            return MoreObjects.toStringHelper(Write.class).add("options", (Object)this.options).add("tableId", (Object)this.tableId).toString();
        }

        @VisibleForTesting
        BigtableService getBigtableService(PipelineOptions pipelineOptions) {
            if (this.bigtableService != null) {
                return this.bigtableService;
            }
            BigtableOptions.Builder clonedOptions = this.options.toBuilder();
            if (this.options.getCredentialOptions().getCredentialType() == CredentialOptions.CredentialType.DefaultCredentials) {
                clonedOptions.setCredentialOptions(CredentialOptions.credential((Credentials)((GcpOptions)pipelineOptions.as(GcpOptions.class)).getGcpCredential()));
            }
            return new BigtableServiceImpl(clonedOptions.build());
        }

        private class BigtableWriterFn
        extends DoFn<KV<ByteString, Iterable<Mutation>>, Void> {
            private final String tableId;
            private final SerializableFunction<PipelineOptions, BigtableService> bigtableServiceFactory;
            private BigtableService.Writer bigtableWriter;
            private long recordsWritten;
            private final ConcurrentLinkedQueue<BigtableWriteException> failures;

            public BigtableWriterFn(String tableId, SerializableFunction<PipelineOptions, BigtableService> bigtableServiceFactory) {
                this.tableId = (String)Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
                this.bigtableServiceFactory = (SerializableFunction)Preconditions.checkNotNull(bigtableServiceFactory, (Object)"bigtableServiceFactory");
                this.failures = new ConcurrentLinkedQueue();
            }

            @DoFn.StartBundle
            public void startBundle(DoFn.Context c) throws IOException {
                if (this.bigtableWriter == null) {
                    this.bigtableWriter = ((BigtableService)this.bigtableServiceFactory.apply((Object)c.getPipelineOptions())).openForWriting(this.tableId);
                }
                this.recordsWritten = 0L;
            }

            @DoFn.ProcessElement
            public void processElement(DoFn.ProcessContext c) throws Exception {
                this.checkForFailures();
                Futures.addCallback(this.bigtableWriter.writeRecord((KV<ByteString, Iterable<Mutation>>)((KV)c.element())), (FutureCallback)new WriteExceptionCallback((KV<ByteString, Iterable<Mutation>>)((KV)c.element())));
                ++this.recordsWritten;
            }

            @DoFn.FinishBundle
            public void finishBundle(DoFn.Context c) throws Exception {
                this.bigtableWriter.flush();
                this.checkForFailures();
                LOG.info("Wrote {} records", (Object)this.recordsWritten);
            }

            @DoFn.Teardown
            public void tearDown() throws Exception {
                this.bigtableWriter.close();
                this.bigtableWriter = null;
            }

            public void populateDisplayData(DisplayData.Builder builder) {
                builder.delegate((HasDisplayData)Write.this);
            }

            private void checkForFailures() throws IOException {
                int i;
                if (this.failures.isEmpty()) {
                    return;
                }
                StringBuilder logEntry = new StringBuilder();
                for (i = 0; i < 10 && !this.failures.isEmpty(); ++i) {
                    BigtableWriteException exc = (BigtableWriteException)this.failures.remove();
                    logEntry.append("\n").append(exc.getMessage());
                    if (exc.getCause() == null) continue;
                    logEntry.append(": ").append(exc.getCause().getMessage());
                }
                String message = String.format("At least %d errors occurred writing to Bigtable. First %d errors: %s", i + this.failures.size(), i, logEntry.toString());
                LOG.error(message);
                throw new IOException(message);
            }

            private class WriteExceptionCallback
            implements FutureCallback<MutateRowResponse> {
                private final KV<ByteString, Iterable<Mutation>> value;

                public WriteExceptionCallback(KV<ByteString, Iterable<Mutation>> value) {
                    this.value = value;
                }

                public void onFailure(Throwable cause) {
                    BigtableWriterFn.this.failures.add(new BigtableWriteException(this.value, cause));
                }

                public void onSuccess(MutateRowResponse produced) {
                }
            }
        }
    }

    @Experimental
    public static class Read
    extends PTransform<PBegin, PCollection<Row>> {
        @Nullable
        private final BigtableOptions options;
        private final String tableId;
        private final ByteKeyRange keyRange;
        @Nullable
        private final RowFilter filter;
        @Nullable
        private final BigtableService bigtableService;

        public Read withBigtableOptions(BigtableOptions options) {
            Preconditions.checkNotNull((Object)options, (Object)"options");
            return this.withBigtableOptions(options.toBuilder());
        }

        public Read withBigtableOptions(BigtableOptions.Builder optionsBuilder) {
            Preconditions.checkNotNull((Object)optionsBuilder, (Object)"optionsBuilder");
            BigtableOptions options = optionsBuilder.build();
            RetryOptions retryOptions = options.getRetryOptions();
            BigtableOptions.Builder clonedBuilder = options.toBuilder().setDataChannelCount(1).setRetryOptions(BigtableIO.retryOptionsToBuilder(retryOptions).setStreamingBatchSize(Math.min(retryOptions.getStreamingBatchSize(), retryOptions.getStreamingBufferSize() / 2)).build());
            BigtableOptions optionsWithAgent = clonedBuilder.setUserAgent(BigtableIO.getUserAgent()).build();
            return new Read(optionsWithAgent, this.tableId, this.keyRange, this.filter, this.bigtableService);
        }

        public Read withRowFilter(RowFilter filter) {
            Preconditions.checkNotNull((Object)filter, (Object)"filter");
            return new Read(this.options, this.tableId, this.keyRange, filter, this.bigtableService);
        }

        public Read withKeyRange(ByteKeyRange keyRange) {
            Preconditions.checkNotNull((Object)keyRange, (Object)"keyRange");
            return new Read(this.options, this.tableId, keyRange, this.filter, this.bigtableService);
        }

        public Read withTableId(String tableId) {
            Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            return new Read(this.options, tableId, this.keyRange, this.filter, this.bigtableService);
        }

        public BigtableOptions getBigtableOptions() {
            return this.options;
        }

        public ByteKeyRange getKeyRange() {
            return this.keyRange;
        }

        public String getTableId() {
            return this.tableId;
        }

        public PCollection<Row> expand(PBegin input) {
            BigtableSource source = new BigtableSource(new SerializableFunction<PipelineOptions, BigtableService>(){

                public BigtableService apply(PipelineOptions options) {
                    return Read.this.getBigtableService(options);
                }
            }, this.tableId, this.filter, this.keyRange, null);
            return (PCollection)input.getPipeline().apply((PTransform)org.apache.beam.sdk.io.Read.from((BoundedSource)source));
        }

        public void validate(PBegin input) {
            Preconditions.checkArgument((this.options != null ? 1 : 0) != 0, (Object)"BigtableOptions not specified");
            Preconditions.checkArgument((!this.tableId.isEmpty() ? 1 : 0) != 0, (Object)"Table ID not specified");
            try {
                Preconditions.checkArgument((boolean)this.getBigtableService(input.getPipeline().getOptions()).tableExists(this.tableId), (String)"Table %s does not exist", (Object[])new Object[]{this.tableId});
            }
            catch (IOException e) {
                LOG.warn("Error checking whether table {} exists; proceeding.", (Object)this.tableId, (Object)e);
            }
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            super.populateDisplayData(builder);
            builder.add(DisplayData.item((String)"tableId", (String)this.tableId).withLabel("Table ID"));
            if (this.options != null) {
                builder.add(DisplayData.item((String)"bigtableOptions", (String)this.options.toString()).withLabel("Bigtable Options"));
            }
            builder.addIfNotDefault(DisplayData.item((String)"keyRange", (String)this.keyRange.toString()), (Object)ByteKeyRange.ALL_KEYS.toString());
            if (this.filter != null) {
                builder.add(DisplayData.item((String)"rowFilter", (String)this.filter.toString()).withLabel("Table Row Filter"));
            }
        }

        public String toString() {
            return MoreObjects.toStringHelper(Read.class).add("options", (Object)this.options).add("tableId", (Object)this.tableId).add("keyRange", (Object)this.keyRange).add("filter", (Object)this.filter).toString();
        }

        private Read(@Nullable BigtableOptions options, String tableId, ByteKeyRange keyRange, @Nullable RowFilter filter, @Nullable BigtableService bigtableService) {
            this.options = options;
            this.tableId = (String)Preconditions.checkNotNull((Object)tableId, (Object)"tableId");
            this.keyRange = (ByteKeyRange)Preconditions.checkNotNull((Object)keyRange, (Object)"keyRange");
            this.filter = filter;
            this.bigtableService = bigtableService;
        }

        Read withBigtableService(BigtableService bigtableService) {
            Preconditions.checkNotNull((Object)bigtableService, (Object)"bigtableService");
            return new Read(this.options, this.tableId, this.keyRange, this.filter, bigtableService);
        }

        @VisibleForTesting
        BigtableService getBigtableService(PipelineOptions pipelineOptions) {
            if (this.bigtableService != null) {
                return this.bigtableService;
            }
            BigtableOptions.Builder clonedOptions = this.options.toBuilder();
            if (this.options.getCredentialOptions().getCredentialType() == CredentialOptions.CredentialType.DefaultCredentials) {
                clonedOptions.setCredentialOptions(CredentialOptions.credential((Credentials)((GcpOptions)pipelineOptions.as(GcpOptions.class)).getGcpCredential()));
            }
            return new BigtableServiceImpl(clonedOptions.build());
        }
    }
}

