/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.gcp.datastore;

import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.BackOffUtils;
import com.google.api.client.util.Sleeper;
import com.google.auth.Credentials;
import com.google.auth.http.HttpCredentialsAdapter;
import com.google.cloud.hadoop.util.ChainingHttpRequestInitializer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.datastore.v1.CommitRequest;
import com.google.datastore.v1.Entity;
import com.google.datastore.v1.EntityResult;
import com.google.datastore.v1.Filter;
import com.google.datastore.v1.Key;
import com.google.datastore.v1.Mutation;
import com.google.datastore.v1.PartitionId;
import com.google.datastore.v1.PropertyFilter;
import com.google.datastore.v1.PropertyOrder;
import com.google.datastore.v1.Query;
import com.google.datastore.v1.QueryResultBatch;
import com.google.datastore.v1.RunQueryRequest;
import com.google.datastore.v1.RunQueryResponse;
import com.google.datastore.v1.Value;
import com.google.datastore.v1.client.Datastore;
import com.google.datastore.v1.client.DatastoreException;
import com.google.datastore.v1.client.DatastoreFactory;
import com.google.datastore.v1.client.DatastoreHelper;
import com.google.datastore.v1.client.DatastoreOptions;
import com.google.datastore.v1.client.QuerySplitter;
import com.google.protobuf.Int32Value;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import javax.annotation.Nullable;
import org.apache.beam.sdk.Pipeline;
import org.apache.beam.sdk.annotations.Experimental;
import org.apache.beam.sdk.io.gcp.datastore.AutoValue_DatastoreV1_Read;
import org.apache.beam.sdk.options.GcpOptions;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.transforms.Create;
import org.apache.beam.sdk.transforms.DoFn;
import org.apache.beam.sdk.transforms.Flatten;
import org.apache.beam.sdk.transforms.GroupByKey;
import org.apache.beam.sdk.transforms.MapElements;
import org.apache.beam.sdk.transforms.PTransform;
import org.apache.beam.sdk.transforms.ParDo;
import org.apache.beam.sdk.transforms.SimpleFunction;
import org.apache.beam.sdk.transforms.Values;
import org.apache.beam.sdk.transforms.display.DisplayData;
import org.apache.beam.sdk.util.FluentBackoff;
import org.apache.beam.sdk.util.RetryHttpRequestInitializer;
import org.apache.beam.sdk.values.KV;
import org.apache.beam.sdk.values.PBegin;
import org.apache.beam.sdk.values.PCollection;
import org.apache.beam.sdk.values.PDone;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Experimental(value=Experimental.Kind.SOURCE_SINK)
public class DatastoreV1 {
    @VisibleForTesting
    static final int DATASTORE_BATCH_UPDATE_LIMIT = 500;

    DatastoreV1() {
    }

    public Read read() {
        return new AutoValue_DatastoreV1_Read.Builder().setNumQuerySplits(0).build();
    }

    public Write write() {
        return new Write(null);
    }

    public DeleteEntity deleteEntity() {
        return new DeleteEntity(null);
    }

    public DeleteKey deleteKey() {
        return new DeleteKey(null);
    }

    static boolean isValidKey(Key key) {
        List elementList = key.getPathList();
        if (elementList.isEmpty()) {
            return false;
        }
        Key.PathElement lastElement = (Key.PathElement)elementList.get(elementList.size() - 1);
        return lastElement.getId() != 0L || !lastElement.getName().isEmpty();
    }

    @VisibleForTesting
    static class V1DatastoreFactory
    implements Serializable {
        V1DatastoreFactory() {
        }

        public Datastore getDatastore(PipelineOptions pipelineOptions, String projectId) {
            Credentials credential = ((GcpOptions)pipelineOptions.as(GcpOptions.class)).getGcpCredential();
            Object initializer = credential != null ? new ChainingHttpRequestInitializer(new HttpRequestInitializer[]{new HttpCredentialsAdapter(credential), new RetryHttpRequestInitializer()}) : new RetryHttpRequestInitializer();
            DatastoreOptions.Builder builder = new DatastoreOptions.Builder().projectId(projectId).initializer((HttpRequestInitializer)initializer);
            return DatastoreFactory.get().create(builder.build());
        }

        public QuerySplitter getQuerySplitter() {
            return DatastoreHelper.getQuerySplitter();
        }
    }

    @VisibleForTesting
    static class DeleteKeyFn
    extends SimpleFunction<Key, Mutation> {
        DeleteKeyFn() {
        }

        public Mutation apply(Key key) {
            Preconditions.checkArgument((boolean)DatastoreV1.isValidKey(key), (String)"Keys to be deleted from the Cloud Datastore must be complete:\n%s", (Object[])new Object[]{key});
            return DatastoreHelper.makeDelete((Key)key).build();
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            builder.add(DisplayData.item((String)"deleteKeyFn", ((Object)((Object)this)).getClass()).withLabel("Create Delete Mutation"));
        }
    }

    @VisibleForTesting
    static class DeleteEntityFn
    extends SimpleFunction<Entity, Mutation> {
        DeleteEntityFn() {
        }

        public Mutation apply(Entity entity) {
            Preconditions.checkArgument((boolean)DatastoreV1.isValidKey(entity.getKey()), (String)"Entities to be deleted from the Cloud Datastore must have complete keys:\n%s", (Object[])new Object[]{entity});
            return DatastoreHelper.makeDelete((Key)entity.getKey()).build();
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            builder.add(DisplayData.item((String)"deleteEntityFn", ((Object)((Object)this)).getClass()).withLabel("Create Delete Mutation"));
        }
    }

    @VisibleForTesting
    static class UpsertFn
    extends SimpleFunction<Entity, Mutation> {
        UpsertFn() {
        }

        public Mutation apply(Entity entity) {
            Preconditions.checkArgument((boolean)DatastoreV1.isValidKey(entity.getKey()), (String)"Entities to be written to the Cloud Datastore must have complete keys:\n%s", (Object[])new Object[]{entity});
            return DatastoreHelper.makeUpsert((Entity)entity).build();
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            builder.add(DisplayData.item((String)"upsertFn", ((Object)((Object)this)).getClass()).withLabel("Create Upsert Mutation"));
        }
    }

    @VisibleForTesting
    static class DatastoreWriterFn
    extends DoFn<Mutation, Void> {
        private static final Logger LOG = LoggerFactory.getLogger(DatastoreWriterFn.class);
        private final String projectId;
        private transient Datastore datastore;
        private final V1DatastoreFactory datastoreFactory;
        private final List<Mutation> mutations = new ArrayList<Mutation>();
        private static final int MAX_RETRIES = 5;
        private static final FluentBackoff BUNDLE_WRITE_BACKOFF = FluentBackoff.DEFAULT.withMaxRetries(5).withInitialBackoff(Duration.standardSeconds((long)5L));

        DatastoreWriterFn(String projectId) {
            this(projectId, new V1DatastoreFactory());
        }

        @VisibleForTesting
        DatastoreWriterFn(String projectId, V1DatastoreFactory datastoreFactory) {
            this.projectId = (String)Preconditions.checkNotNull((Object)projectId, (Object)"projectId");
            this.datastoreFactory = datastoreFactory;
        }

        @DoFn.StartBundle
        public void startBundle(DoFn.Context c) {
            this.datastore = this.datastoreFactory.getDatastore(c.getPipelineOptions(), this.projectId);
        }

        @DoFn.ProcessElement
        public void processElement(DoFn.ProcessContext c) throws Exception {
            this.mutations.add((Mutation)c.element());
            if (this.mutations.size() >= 500) {
                this.flushBatch();
            }
        }

        @DoFn.FinishBundle
        public void finishBundle(DoFn.Context c) throws Exception {
            if (!this.mutations.isEmpty()) {
                this.flushBatch();
            }
        }

        private void flushBatch() throws DatastoreException, IOException, InterruptedException {
            LOG.debug("Writing batch of {} mutations", (Object)this.mutations.size());
            Sleeper sleeper = Sleeper.DEFAULT;
            BackOff backoff = BUNDLE_WRITE_BACKOFF.backoff();
            while (true) {
                try {
                    CommitRequest.Builder commitRequest = CommitRequest.newBuilder();
                    commitRequest.addAllMutations(this.mutations);
                    commitRequest.setMode(CommitRequest.Mode.NON_TRANSACTIONAL);
                    this.datastore.commit(commitRequest.build());
                }
                catch (DatastoreException exception) {
                    LOG.error("Error writing to the Datastore ({}): {}", (Object)exception.getCode(), (Object)exception.getMessage());
                    if (BackOffUtils.next((Sleeper)sleeper, (BackOff)backoff)) continue;
                    LOG.error("Aborting after {} retries.", (Object)5);
                    throw exception;
                }
                break;
            }
            LOG.debug("Successfully wrote {} mutations", (Object)this.mutations.size());
            this.mutations.clear();
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            super.populateDisplayData(builder);
            builder.addIfNotNull(DisplayData.item((String)"projectId", (String)this.projectId).withLabel("Output Project"));
        }
    }

    private static abstract class Mutate<T>
    extends PTransform<PCollection<T>, PDone> {
        @Nullable
        private final String projectId;
        private final SimpleFunction<T, Mutation> mutationFn;

        Mutate(@Nullable String projectId, SimpleFunction<T, Mutation> mutationFn) {
            this.projectId = projectId;
            this.mutationFn = (SimpleFunction)Preconditions.checkNotNull(mutationFn);
        }

        public PDone expand(PCollection<T> input) {
            ((PCollection)input.apply("Convert to Mutation", (PTransform)MapElements.via(this.mutationFn))).apply("Write Mutation to Datastore", (PTransform)ParDo.of((DoFn)new DatastoreWriterFn(this.projectId)));
            return PDone.in((Pipeline)input.getPipeline());
        }

        public void validate(PCollection<T> input) {
            Preconditions.checkNotNull((Object)this.projectId, (Object)"projectId");
            Preconditions.checkNotNull(this.mutationFn, (Object)"mutationFn");
        }

        public String toString() {
            return MoreObjects.toStringHelper(((Object)((Object)this)).getClass()).add("projectId", (Object)this.projectId).add("mutationFn", (Object)this.mutationFn.getClass().getName()).toString();
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            super.populateDisplayData(builder);
            builder.addIfNotNull(DisplayData.item((String)"projectId", (String)this.projectId).withLabel("Output Project")).include("mutationFn", this.mutationFn);
        }

        public String getProjectId() {
            return this.projectId;
        }
    }

    public static class DeleteKey
    extends Mutate<Key> {
        DeleteKey(@Nullable String projectId) {
            super(projectId, new DeleteKeyFn());
        }

        public DeleteKey withProjectId(String projectId) {
            Preconditions.checkNotNull((Object)projectId, (Object)"projectId");
            return new DeleteKey(projectId);
        }
    }

    public static class DeleteEntity
    extends Mutate<Entity> {
        DeleteEntity(@Nullable String projectId) {
            super(projectId, new DeleteEntityFn());
        }

        public DeleteEntity withProjectId(String projectId) {
            Preconditions.checkNotNull((Object)projectId, (Object)"projectId");
            return new DeleteEntity(projectId);
        }
    }

    public static class Write
    extends Mutate<Entity> {
        Write(@Nullable String projectId) {
            super(projectId, new UpsertFn());
        }

        public Write withProjectId(String projectId) {
            Preconditions.checkNotNull((Object)projectId, (Object)"projectId");
            return new Write(projectId);
        }
    }

    public static abstract class Read
    extends PTransform<PBegin, PCollection<Entity>> {
        private static final Logger LOG = LoggerFactory.getLogger(Read.class);
        public static final int NUM_QUERY_SPLITS_MAX = 50000;
        static final int NUM_QUERY_SPLITS_MIN = 12;
        static final long DEFAULT_BUNDLE_SIZE_BYTES = 0x4000000L;
        static final int QUERY_BATCH_LIMIT = 500;

        @Nullable
        public abstract String getProjectId();

        @Nullable
        public abstract Query getQuery();

        @Nullable
        public abstract String getNamespace();

        public abstract int getNumQuerySplits();

        public abstract String toString();

        abstract Builder toBuilder();

        static int getEstimatedNumSplits(Datastore datastore, Query query, @Nullable String namespace) {
            int numSplits;
            try {
                long estimatedSizeBytes = Read.getEstimatedSizeBytes(datastore, query, namespace);
                LOG.info("Estimated size bytes for the query is: {}", (Object)estimatedSizeBytes);
                numSplits = (int)Math.min(50000L, Math.round((double)estimatedSizeBytes / 6.7108864E7));
            }
            catch (Exception e) {
                LOG.warn("Failed the fetch estimatedSizeBytes for query: {}", (Object)query, (Object)e);
                numSplits = 12;
            }
            return Math.max(numSplits, 12);
        }

        private static long queryLatestStatisticsTimestamp(Datastore datastore, @Nullable String namespace) throws DatastoreException {
            Query.Builder query = Query.newBuilder();
            if (namespace == null) {
                query.addKindBuilder().setName("__Stat_Total__");
            } else {
                query.addKindBuilder().setName("__Stat_Ns_Total__");
            }
            query.addOrder(DatastoreHelper.makeOrder((String)"timestamp", (PropertyOrder.Direction)PropertyOrder.Direction.DESCENDING));
            query.setLimit(Int32Value.newBuilder().setValue(1));
            RunQueryRequest request = Read.makeRequest(query.build(), namespace);
            RunQueryResponse response = datastore.runQuery(request);
            QueryResultBatch batch = response.getBatch();
            if (batch.getEntityResultsCount() == 0) {
                throw new NoSuchElementException("Datastore total statistics unavailable");
            }
            Entity entity = batch.getEntityResults(0).getEntity();
            return ((Value)entity.getProperties().get("timestamp")).getTimestampValue().getSeconds() * 1000000L;
        }

        static long getEstimatedSizeBytes(Datastore datastore, Query query, @Nullable String namespace) throws DatastoreException {
            String ourKind = query.getKind(0).getName();
            long latestTimestamp = Read.queryLatestStatisticsTimestamp(datastore, namespace);
            LOG.info("Latest stats timestamp for kind {} is {}", (Object)ourKind, (Object)latestTimestamp);
            Query.Builder queryBuilder = Query.newBuilder();
            if (namespace == null) {
                queryBuilder.addKindBuilder().setName("__Stat_Kind__");
            } else {
                queryBuilder.addKindBuilder().setName("__Stat_Ns_Kind__");
            }
            queryBuilder.setFilter(DatastoreHelper.makeAndFilter((Filter[])new Filter[]{DatastoreHelper.makeFilter((String)"kind_name", (PropertyFilter.Operator)PropertyFilter.Operator.EQUAL, (Value)DatastoreHelper.makeValue((String)ourKind).build()).build(), DatastoreHelper.makeFilter((String)"timestamp", (PropertyFilter.Operator)PropertyFilter.Operator.EQUAL, (Value)DatastoreHelper.makeValue((long)latestTimestamp).build()).build()}));
            RunQueryRequest request = Read.makeRequest(queryBuilder.build(), namespace);
            long now = System.currentTimeMillis();
            RunQueryResponse response = datastore.runQuery(request);
            LOG.debug("Query for per-kind statistics took {}ms", (Object)(System.currentTimeMillis() - now));
            QueryResultBatch batch = response.getBatch();
            if (batch.getEntityResultsCount() == 0) {
                throw new NoSuchElementException("Datastore statistics for kind " + ourKind + " unavailable");
            }
            Entity entity = batch.getEntityResults(0).getEntity();
            return ((Value)entity.getProperties().get("entity_bytes")).getIntegerValue();
        }

        static RunQueryRequest makeRequest(Query query, @Nullable String namespace) {
            RunQueryRequest.Builder requestBuilder = RunQueryRequest.newBuilder().setQuery(query);
            if (namespace != null) {
                requestBuilder.getPartitionIdBuilder().setNamespaceId(namespace);
            }
            return requestBuilder.build();
        }

        private static List<Query> splitQuery(Query query, @Nullable String namespace, Datastore datastore, QuerySplitter querySplitter, int numSplits) throws DatastoreException {
            PartitionId.Builder partitionBuilder = PartitionId.newBuilder();
            if (namespace != null) {
                partitionBuilder.setNamespaceId(namespace);
            }
            return querySplitter.getSplits(query, partitionBuilder.build(), numSplits, datastore);
        }

        public Read withProjectId(String projectId) {
            Preconditions.checkNotNull((Object)projectId, (Object)"projectId");
            return this.toBuilder().setProjectId(projectId).build();
        }

        public Read withQuery(Query query) {
            Preconditions.checkNotNull((Object)query, (Object)"query");
            Preconditions.checkArgument((!query.hasLimit() || query.getLimit().getValue() > 0 ? 1 : 0) != 0, (String)"Invalid query limit %s: must be positive", (Object[])new Object[]{query.getLimit().getValue()});
            return this.toBuilder().setQuery(query).build();
        }

        public Read withNamespace(String namespace) {
            return this.toBuilder().setNamespace(namespace).build();
        }

        public Read withNumQuerySplits(int numQuerySplits) {
            return this.toBuilder().setNumQuerySplits(Math.min(Math.max(numQuerySplits, 0), 50000)).build();
        }

        public PCollection<Entity> expand(PBegin input) {
            V1Options v1Options = V1Options.from(this.getProjectId(), this.getQuery(), this.getNamespace());
            PCollection queries = (PCollection)((PCollection)input.apply((PTransform)Create.of((Object[])new Query[]{this.getQuery()}))).apply((PTransform)ParDo.of((DoFn)new SplitQueryFn(v1Options, this.getNumQuerySplits())));
            PCollection shardedQueries = (PCollection)((PCollection)((PCollection)queries.apply((PTransform)GroupByKey.create())).apply((PTransform)Values.create())).apply((PTransform)Flatten.iterables());
            PCollection entities = (PCollection)shardedQueries.apply((PTransform)ParDo.of((DoFn)new ReadFn(v1Options)));
            return entities;
        }

        public void validate(PBegin input) {
            Preconditions.checkNotNull((Object)this.getProjectId(), (Object)"projectId");
            Preconditions.checkNotNull((Object)this.getQuery(), (Object)"query");
        }

        public void populateDisplayData(DisplayData.Builder builder) {
            super.populateDisplayData(builder);
            builder.addIfNotNull(DisplayData.item((String)"projectId", (String)this.getProjectId()).withLabel("ProjectId")).addIfNotNull(DisplayData.item((String)"namespace", (String)this.getNamespace()).withLabel("Namespace")).addIfNotNull(DisplayData.item((String)"query", (String)this.getQuery().toString()).withLabel("Query"));
        }

        @VisibleForTesting
        static class ReadFn
        extends DoFn<Query, Entity> {
            private final V1Options options;
            private final V1DatastoreFactory datastoreFactory;
            private transient Datastore datastore;

            public ReadFn(V1Options options) {
                this(options, new V1DatastoreFactory());
            }

            @VisibleForTesting
            ReadFn(V1Options options, V1DatastoreFactory datastoreFactory) {
                this.options = options;
                this.datastoreFactory = datastoreFactory;
            }

            @DoFn.StartBundle
            public void startBundle(DoFn.Context c) throws Exception {
                this.datastore = this.datastoreFactory.getDatastore(c.getPipelineOptions(), this.options.getProjectId());
            }

            @DoFn.ProcessElement
            public void processElement(DoFn.ProcessContext context) throws Exception {
                Query query = (Query)context.element();
                String namespace = this.options.getNamespace();
                int userLimit = query.hasLimit() ? query.getLimit().getValue() : Integer.MAX_VALUE;
                boolean moreResults = true;
                QueryResultBatch currentBatch = null;
                while (moreResults) {
                    Query.Builder queryBuilder = query.toBuilder().clone();
                    queryBuilder.setLimit(Int32Value.newBuilder().setValue(Math.min(userLimit, 500)));
                    if (currentBatch != null && !currentBatch.getEndCursor().isEmpty()) {
                        queryBuilder.setStartCursor(currentBatch.getEndCursor());
                    }
                    RunQueryRequest request = Read.makeRequest(queryBuilder.build(), namespace);
                    RunQueryResponse response = this.datastore.runQuery(request);
                    currentBatch = response.getBatch();
                    int numFetch = currentBatch.getEntityResultsCount();
                    if (query.hasLimit()) {
                        Verify.verify((userLimit >= numFetch ? 1 : 0) != 0, (String)"Expected userLimit %s >= numFetch %s, because query limit %s must be <= userLimit", (Object[])new Object[]{userLimit, numFetch, query.getLimit()});
                        userLimit -= numFetch;
                    }
                    for (EntityResult entityResult : currentBatch.getEntityResultsList()) {
                        context.output((Object)entityResult.getEntity());
                    }
                    moreResults = userLimit > 0 && (numFetch == 500 || currentBatch.getMoreResults() == QueryResultBatch.MoreResultsType.NOT_FINISHED);
                }
            }
        }

        @VisibleForTesting
        static class SplitQueryFn
        extends DoFn<Query, KV<Integer, Query>> {
            private final V1Options options;
            private final int numSplits;
            private final V1DatastoreFactory datastoreFactory;
            private transient Datastore datastore;
            private transient QuerySplitter querySplitter;

            public SplitQueryFn(V1Options options, int numSplits) {
                this(options, numSplits, new V1DatastoreFactory());
            }

            @VisibleForTesting
            SplitQueryFn(V1Options options, int numSplits, V1DatastoreFactory datastoreFactory) {
                this.options = options;
                this.numSplits = numSplits;
                this.datastoreFactory = datastoreFactory;
            }

            @DoFn.StartBundle
            public void startBundle(DoFn.Context c) throws Exception {
                this.datastore = this.datastoreFactory.getDatastore(c.getPipelineOptions(), this.options.projectId);
                this.querySplitter = this.datastoreFactory.getQuerySplitter();
            }

            @DoFn.ProcessElement
            public void processElement(DoFn.ProcessContext c) throws Exception {
                Object querySplits;
                int key = 1;
                Query query = (Query)c.element();
                if (query.hasLimit()) {
                    c.output((Object)KV.of((Object)key, (Object)query));
                    return;
                }
                int estimatedNumSplits = this.numSplits <= 0 ? Read.getEstimatedNumSplits(this.datastore, query, this.options.getNamespace()) : this.numSplits;
                LOG.info("Splitting the query into {} splits", (Object)estimatedNumSplits);
                try {
                    querySplits = Read.splitQuery(query, this.options.getNamespace(), this.datastore, this.querySplitter, estimatedNumSplits);
                }
                catch (Exception e) {
                    LOG.warn("Unable to parallelize the given query: {}", (Object)query, (Object)e);
                    querySplits = ImmutableList.of((Object)query);
                }
                Iterator iterator = querySplits.iterator();
                while (iterator.hasNext()) {
                    Query subquery = (Query)iterator.next();
                    c.output((Object)KV.of((Object)key++, (Object)subquery));
                }
            }

            public void populateDisplayData(DisplayData.Builder builder) {
                super.populateDisplayData(builder);
                builder.addIfNotNull(DisplayData.item((String)"projectId", (String)this.options.getProjectId()).withLabel("ProjectId")).addIfNotNull(DisplayData.item((String)"namespace", (String)this.options.getNamespace()).withLabel("Namespace")).addIfNotNull(DisplayData.item((String)"query", (String)this.options.getQuery().toString()).withLabel("Query"));
            }
        }

        @VisibleForTesting
        static class V1Options
        implements Serializable {
            private final Query query;
            private final String projectId;
            @Nullable
            private final String namespace;

            private V1Options(String projectId, Query query, @Nullable String namespace) {
                this.projectId = (String)Preconditions.checkNotNull((Object)projectId, (Object)"projectId");
                this.query = (Query)Preconditions.checkNotNull((Object)query, (Object)"query");
                this.namespace = namespace;
            }

            public static V1Options from(String projectId, Query query, @Nullable String namespace) {
                return new V1Options(projectId, query, namespace);
            }

            public Query getQuery() {
                return this.query;
            }

            public String getProjectId() {
                return this.projectId;
            }

            @Nullable
            public String getNamespace() {
                return this.namespace;
            }
        }

        static abstract class Builder {
            Builder() {
            }

            abstract Builder setProjectId(String var1);

            abstract Builder setQuery(Query var1);

            abstract Builder setNamespace(String var1);

            abstract Builder setNumQuerySplits(int var1);

            abstract Read build();
        }
    }
}

