/*
 * Decompiled with CFR 0.152.
 */
package alluxio.util.compression;

import alluxio.util.executor.ExecutorServiceFactories;
import alluxio.util.executor.ExecutorServiceUtils;
import alluxio.util.io.FileUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Enumeration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.compress.archivers.zip.ParallelScatterZipCreator;
import org.apache.commons.compress.archivers.zip.Zip64Mode;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.compress.parallel.FileBasedScatterGatherBackingStore;
import org.apache.commons.compress.parallel.InputStreamSupplier;
import org.apache.commons.compress.parallel.ScatterGatherBackingStore;
import org.apache.commons.compress.parallel.ScatterGatherBackingStoreSupplier;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.NullInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ParallelZipUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ParallelZipUtils.class);

    public static void compress(Path dirPath, OutputStream outputStream, int poolSize, int compressionLevel) throws IOException, InterruptedException {
        LOG.info("compress in parallel for path {}", (Object)dirPath);
        ExecutorService executor = ExecutorServiceFactories.fixedThreadPool("parallel-zip-compress-pool", poolSize).create();
        ParallelScatterZipCreator parallelScatterZipCreator = new ParallelScatterZipCreator(executor, (ScatterGatherBackingStoreSupplier)new BasicBackingStoreSupplier(), compressionLevel);
        ZipArchiveOutputStream zipArchiveOutputStream = new ZipArchiveOutputStream(outputStream);
        zipArchiveOutputStream.setUseZip64(Zip64Mode.Always);
        try {
            try (Stream<Path> stream = Files.walk(dirPath, new FileVisitOption[0]);){
                for (Path subPath : stream.collect(Collectors.toList())) {
                    if (Thread.interrupted()) {
                        throw new InterruptedException();
                    }
                    File file = subPath.toFile();
                    InputStreamSupplier inputStreamSupplier = () -> {
                        try {
                            if (file.exists() && file.isFile()) {
                                return new FileInputStream(file);
                            }
                            return new NullInputStream(0L);
                        }
                        catch (FileNotFoundException e) {
                            LOG.warn("Can't find file when parallel zip, path = {}", (Object)subPath);
                            return new NullInputStream(0L);
                        }
                    };
                    String entryName = dirPath.relativize(subPath).toString();
                    if (file.isDirectory()) {
                        entryName = entryName + File.separator;
                    }
                    ZipArchiveEntry zipArchiveEntry = new ZipArchiveEntry(entryName);
                    zipArchiveEntry.setMethod(8);
                    parallelScatterZipCreator.addArchiveEntry(zipArchiveEntry, inputStreamSupplier);
                }
            }
            parallelScatterZipCreator.writeTo(zipArchiveOutputStream);
            zipArchiveOutputStream.finish();
            zipArchiveOutputStream.flush();
        }
        catch (ExecutionException e) {
            LOG.error("Parallel compress rocksdb failed", (Throwable)e);
            throw new IOException(e);
        }
        finally {
            if (!executor.isTerminated()) {
                LOG.info("ParallelScatterZipCreator failed to shut down the thread pool, cleaning up now.");
                ExecutorServiceUtils.shutdownAndAwaitTermination(executor);
            }
        }
        LOG.info("Completed parallel compression for path {}, statistics: {}", (Object)dirPath, (Object)parallelScatterZipCreator.getStatisticsMessage().toString());
    }

    public static void decompress(Path dirPath, String backupPath, int poolSize) throws IOException {
        LOG.info("decompress in parallel from path {} to {}", (Object)backupPath, (Object)dirPath);
        ExecutorService executor = ExecutorServiceFactories.fixedThreadPool("parallel-zip-decompress-pool", poolSize).create();
        ExecutorCompletionService<Boolean> completionService = new ExecutorCompletionService<Boolean>(executor);
        try (ZipFile zipFile = new ZipFile(backupPath);){
            Enumeration entries = zipFile.getEntries();
            int taskCount = 0;
            while (entries.hasMoreElements()) {
                ++taskCount;
                ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
                completionService.submit(() -> {
                    ParallelZipUtils.unzipEntry(zipFile, entry, dirPath);
                    return true;
                });
            }
            for (int i = 0; i < taskCount; ++i) {
                completionService.take().get();
            }
        }
        catch (ExecutionException e) {
            LOG.error("Parallel decompress rocksdb fail", (Throwable)e);
            FileUtils.deletePathRecursively(dirPath.toString());
            throw new IOException(e);
        }
        catch (InterruptedException e) {
            LOG.info("Parallel decompress rocksdb interrupted");
            FileUtils.deletePathRecursively(dirPath.toString());
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        finally {
            ExecutorServiceUtils.shutdownAndAwaitTermination(executor);
        }
    }

    private static void unzipEntry(ZipFile zipFile, ZipArchiveEntry entry, Path dirPath) throws Exception {
        File outputFile = new File(dirPath.toFile(), entry.getName());
        outputFile.getParentFile().mkdirs();
        if (entry.isDirectory()) {
            outputFile.mkdir();
        } else {
            try (InputStream inputStream = zipFile.getInputStream(entry);
                 FileOutputStream fileOutputStream = new FileOutputStream(outputFile);){
                IOUtils.copy((InputStream)inputStream, (OutputStream)fileOutputStream);
            }
        }
    }

    private ParallelZipUtils() {
    }

    private static class BasicBackingStoreSupplier
    implements ScatterGatherBackingStoreSupplier {
        final AtomicInteger mStoreNum = new AtomicInteger(0);

        private BasicBackingStoreSupplier() {
        }

        public ScatterGatherBackingStore get() throws IOException {
            File tempFile = File.createTempFile("zipUtilsParallelScatter", "n" + this.mStoreNum.incrementAndGet());
            return new FileBasedScatterGatherBackingStore(tempFile);
        }
    }
}

