/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.opensymphony.xwork.util;

import com.opensymphony.xwork.config.Configuration;
import com.opensymphony.xwork.config.ConfigurationManager;
import ognl.Node;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import static java.util.Collections.unmodifiableSet;

/**
 * The default implementation of {@link OgnlGuard}.
 *
 * @since Struts 6.4.0
 */
public class DefaultOgnlGuard implements OgnlGuard {

    private static final Log LOG = LogFactory.getLog(DefaultOgnlGuard.class);

    private static final Set<String> BLOCKED_VAR_REFS = unmodifiableSet(new HashSet<>(Arrays.asList(
            "#_memberAccess",
            "#_classResolver",
            "#_typeConverter",
            "#context",
            "#request",
            "#parameters",
            "#session",
            "#application",
            "#attr"
    )));

    protected Set<String> excludedNodeTypes;
    private int maxExprLength;

    public DefaultOgnlGuard() {
        loadConfig(ConfigurationManager.getConfiguration());
    }

    public void loadConfig(Configuration configuration) {
        excludedNodeTypes = configuration.getOgnlExcludedNodeTypes();
        maxExprLength = configuration.getOgnlExpressionMaxLength();
    }

    @Override
    public boolean isRawExpressionBlocked(String expr) {
        return isExceedsMaxExprLength(expr);
    }

    protected boolean isExceedsMaxExprLength(String expr) {
        return expr != null && expr.length() > maxExprLength;
    }

    @Override
    public boolean isParsedTreeBlocked(Object tree) {
        if (!(tree instanceof Node) || skipTreeCheck((Node) tree)) {
            return false;
        }
        return recurseNodes((Node) tree);
    }

    protected boolean skipTreeCheck(Node tree) {
        return false;
    }

    protected boolean recurseNodes(Node node) {
        if (checkNode(node)) {
            return true;
        }
        for (int i = 0; i < node.jjtGetNumChildren(); i++) {
            if (recurseNodes(node.jjtGetChild(i))) {
                return true;
            }
        }
        return false;
    }

    protected boolean checkNode(Node node) {
        return containsExcludedNodeType(node) || isBlockedVarRef(node);
    }

    protected boolean containsExcludedNodeType(Node node) {
        String nodeClassName = node.getClass().getName();
        if (excludedNodeTypes.contains(nodeClassName)) {
            LOG.warn("Expression contains blocked node type [" + nodeClassName + "]");
            return true;
        }
        return false;
    }

    protected boolean isBlockedVarRef(Node node) {
        String nodeClassName = node.getClass().getName();
        if ("ognl.ASTVarRef".equals(nodeClassName)) {
            String varRefValue = node.toString();
            if (BLOCKED_VAR_REFS.contains(varRefValue)) {
                if (!"#attr".equals(varRefValue)) {
                    // Prevent excessive logging due to unnecessary usage of #attr by Struts Core
                    LOG.warn("Expression contains blocked var ref [" + varRefValue + "]");
                }
                return true;
            }
        }
        return false;
    }
}
