/*
 * Copyright (c) 2002-2003 by OpenSymphony
 * All rights reserved.
 */
package com.opensymphony.xwork.config.impl;

import com.opensymphony.xwork.config.Configuration;
import com.opensymphony.xwork.config.ConfigurationException;
import com.opensymphony.xwork.config.ConfigurationManager;
import com.opensymphony.xwork.config.ConfigurationProvider;
import com.opensymphony.xwork.config.RuntimeConfiguration;
import com.opensymphony.xwork.config.entities.ActionConfig;
import com.opensymphony.xwork.config.entities.PackageConfig;
import com.opensymphony.xwork.config.entities.ResultTypeConfig;
import com.opensymphony.xwork.config.providers.InterceptorBuilder;
import com.opensymphony.xwork.util.OgnlUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.opensymphony.xwork.config.providers.XmlConfigurationProvider.XWORK_ALLOWLIST_ENABLE;
import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;


/**
 * DefaultConfiguration
 *
 * @author Jason Carreira
 *         Created Feb 24, 2003 7:38:06 AM
 */
public class DefaultConfiguration implements Configuration {
    //~ Static fields/initializers /////////////////////////////////////////////

    protected static final Log LOG = LogFactory.getLog(DefaultConfiguration.class);

    //~ Instance fields ////////////////////////////////////////////////////////

    // Programmatic Action Conifigurations
    private Map packageContexts = new HashMap();
    private RuntimeConfiguration runtimeConfiguration;

    private Set<String> excludedClasses = emptySet();
    private Set<String> excludedPackageNames = emptySet();
    private Set<String> excludedPackageExemptClasses = emptySet();
    private boolean allowStaticMethodAccess = true;
    private boolean allowStaticFieldAccess = true;
    private boolean disallowProxyMemberAccess = false;
    private boolean disallowDefaultPackageAccess = false;
    private Set<String> staticMemberAllowedClasses = emptySet();

    private int ognlExpressionMaxLength = 200;
    private Set<String> ognlExcludedNodeTypes = emptySet();

    private boolean enforceAllowlistEnabled = false;
    private Set<String> allowlistClasses = emptySet();
    private Set<String> allowlistPackageNames = emptySet();

    //~ Constructors ///////////////////////////////////////////////////////////

    public DefaultConfiguration() {
    }

    //~ Methods ////////////////////////////////////////////////////////////////

    public PackageConfig getPackageConfig(String name) {
        return (PackageConfig) packageContexts.get(name);
    }

    public Set getPackageConfigNames() {
        return packageContexts.keySet();
    }

    public Map getPackageConfigs() {
        return packageContexts;
    }

    public RuntimeConfiguration getRuntimeConfiguration() {
        return runtimeConfiguration;
    }

    public void addPackageConfig(String name, PackageConfig packageContext) {
        packageContexts.put(name, packageContext);
    }

    private static Set<String> toNewUnmodifiableSet(Collection<String> first, Collection<String> second) {
        Set<String> result = new HashSet<>();
        result.addAll(first);
        result.addAll(second);
        return unmodifiableSet(result);
    }

    @Override
    public void addExcludedClasses(Collection<String> newExcludedClasses) {
        this.excludedClasses = toNewUnmodifiableSet(this.excludedClasses, newExcludedClasses);
    }

    @Override
    public Set<String> getExcludedClasses() {
        return excludedClasses;
    }

    @Override
    public void addExcludedPackageNames(Collection<String> newExcludedPackageNames) {
        this.excludedPackageNames = toNewUnmodifiableSet(this.excludedPackageNames, newExcludedPackageNames);
    }


    @Override
    public Set<String> getExcludedPackageNames() {
        return excludedPackageNames;
    }

    @Override
    public void setExcludedPackageExemptClasses(Collection<String> newExcludedPackageExemptClasses) {
        this.excludedPackageExemptClasses = unmodifiableSet(new HashSet<>(newExcludedPackageExemptClasses));
    }

    @Override
    public Set<String> getExcludedPackageExemptClasses() {
        return excludedPackageExemptClasses;
    }

    @Override
    public void setAllowStaticMethodAccess(boolean allowStaticMethodAccess) {
        this.allowStaticMethodAccess = allowStaticMethodAccess;
    }

    @Override
    public boolean isAllowStaticMethodAccess() {
        return allowStaticMethodAccess;
    }

    @Override
    public void setAllowStaticFieldAccess(boolean allowStaticFieldAccess) {
        this.allowStaticFieldAccess = allowStaticFieldAccess;
    }

    @Override
    public boolean isAllowStaticFieldAccess() {
        return allowStaticFieldAccess;
    }

    @Override
    public void setDisallowProxyMemberAccess(boolean disallowProxyMemberAccess) {
        this.disallowProxyMemberAccess = disallowProxyMemberAccess;
    }

    @Override
    public boolean isDisallowProxyMemberAccess() {
        return disallowProxyMemberAccess;
    }

    @Override
    public void setDisallowDefaultPackageAccess(boolean disallowDefaultPackageAccess) {
        this.disallowDefaultPackageAccess = disallowDefaultPackageAccess;
    }

    @Override
    public boolean isDisallowDefaultPackageAccess() {
        return disallowDefaultPackageAccess;
    }

    @Override
    public void setStaticMemberAllowedClasses(Collection<String> newStaticMemberAllowedClasses) {
        this.staticMemberAllowedClasses = unmodifiableSet(new HashSet<>(newStaticMemberAllowedClasses));
    }

    @Override
    public Set<String> getStaticMemberAllowedClasses() {
        return staticMemberAllowedClasses;
    }

    @Override
    public void setOgnlExpressionMaxLength(int ognlExpressionMaxLength) {
        this.ognlExpressionMaxLength = ognlExpressionMaxLength;
    }

    @Override
    public int getOgnlExpressionMaxLength() { return ognlExpressionMaxLength; }

    @Override
    public void addOgnlExcludedNodeTypes(Collection<String> newExcludedNodeTypes) {
        this.ognlExcludedNodeTypes = toNewUnmodifiableSet(this.ognlExcludedNodeTypes, newExcludedNodeTypes);
    }

    @Override
    public Set<String> getOgnlExcludedNodeTypes() {
        return ognlExcludedNodeTypes;
    }

    @Override
    public void setEnforceAllowlistEnabled(boolean XWorkAllowlistEnable) {
        this.enforceAllowlistEnabled = XWorkAllowlistEnable;
    }

    @Override
    public boolean isEnforceAllowlistEnabled() {
        return enforceAllowlistEnabled;
    }

    @Override
    public void setAllowlistClasses(Collection<String> newAllowlistClasses) {
        this.allowlistClasses = unmodifiableSet(new HashSet<>(newAllowlistClasses));
    }

    @Override
    public Set<String> getAllowlistClasses() {
        return allowlistClasses;
    }

    @Override
    public void setAllowlistPackageNames(Collection<String> newAllowlistPackageNames) {
        this.allowlistPackageNames = unmodifiableSet(new HashSet<>(newAllowlistPackageNames));
    }

    @Override
    public Set<String> getAllowlistPackageNames() {
        return allowlistPackageNames;
    }

    /**
     * Allows the configuration to clean up any resources used
     */
    public void destroy() {
    }

    public void rebuildRuntimeConfiguration() {
        runtimeConfiguration = buildRuntimeConfiguration();
    }

    /**
     * Calls the ConfigurationProviderFactory.getConfig() to tell it to reload the configuration and then calls
     * buildRuntimeConfiguration().
     *
     * @throws ConfigurationException
     */
    public synchronized void reload() throws ConfigurationException {
        packageContexts.clear();

        for (Object o : ConfigurationManager.getConfigurationProviders()) {
            ConfigurationProvider provider = (ConfigurationProvider) o;
            provider.init(this);
        }

        // Allowlist can also be enabled by System property
        setEnforceAllowlistEnabled(isEnforceAllowlistEnabled() || Boolean.getBoolean(XWORK_ALLOWLIST_ENABLE));

        refreshSecurityConfig();

        rebuildRuntimeConfiguration();
    }

    public synchronized void refreshSecurityConfig() {
        OgnlUtil.retrieveSecurityMemberAccess().loadConfig(this);
        OgnlUtil.retrieveOgnlGuard().loadConfig(this);
    }

    public void removePackageConfig(String name) {
        PackageConfig toBeRemoved = (PackageConfig) packageContexts.get(name);

        if (toBeRemoved != null) {
            for (Iterator iterator = packageContexts.values().iterator();
                    iterator.hasNext();) {
                PackageConfig packageContext = (PackageConfig) iterator.next();
                packageContext.removeParent(toBeRemoved);
            }
        }
    }

    /**
     * This methodName builds the internal runtime configuration used by Xwork for finding and configuring Actions from the
     * programmatic configuration data structures. All of the old runtime configuration will be discarded and rebuilt.
     */
    protected synchronized RuntimeConfiguration buildRuntimeConfiguration() throws ConfigurationException {
        Map namespaceActionConfigs = new HashMap();

        for (Iterator iterator = packageContexts.values().iterator();
                iterator.hasNext();) {
            PackageConfig packageContext = (PackageConfig) iterator.next();

            if (!packageContext.isAbstract()) {
                String namespace = packageContext.getNamespace();
                Map configs = (Map) namespaceActionConfigs.get(namespace);

                if (configs == null) {
                    configs = new HashMap();
                }

                Map actionConfigs = packageContext.getAllActionConfigs();

                for (Iterator actionIterator = actionConfigs.keySet().iterator();
                        actionIterator.hasNext();) {
                    String actionName = (String) actionIterator.next();
                    ActionConfig baseConfig = (ActionConfig) actionConfigs.get(actionName);
                    configs.put(actionName, buildFullActionConfig(packageContext, baseConfig));
                }

                namespaceActionConfigs.put(namespace, configs);
            }
        }

        return new RuntimeConfigurationImpl(namespaceActionConfigs);
    }

    private void setDefaultResults(Map results, PackageConfig packageContext) {
        String defaultResult = packageContext.getFullDefaultResultType();

        for (Iterator iterator = results.entrySet().iterator();
                iterator.hasNext();) {
            Map.Entry entry = (Map.Entry) iterator.next();

            if (entry.getValue() == null) {
                ResultTypeConfig resultTypeConfig = (ResultTypeConfig) packageContext.getAllResultTypeConfigs().get(defaultResult);
                entry.setValue(resultTypeConfig.getClazz());
            }
        }
    }

    /**
     * Builds the full runtime actionconfig with all of the defaults and inheritance
     *
     * @param packageContext the PackageConfig which holds the base config we're building from
     * @param baseConfig     the ActionConfig which holds only the configuration specific to itself, without the defaults
     *                       and inheritance
     * @return a full ActionConfig for runtime configuration with all of the inherited and default params
     */
    private ActionConfig buildFullActionConfig(PackageConfig packageContext, ActionConfig baseConfig) throws ConfigurationException {
        Map params = new HashMap(baseConfig.getParams());
        Map results = new HashMap(packageContext.getAllGlobalResults());
        results.putAll(baseConfig.getResults());

        setDefaultResults(results, packageContext);

        List interceptors = new ArrayList(baseConfig.getInterceptors());

        if (interceptors.size() <= 0) {
            String defaultInterceptorRefName = packageContext.getFullDefaultInterceptorRef();

            if (defaultInterceptorRefName != null) {
                interceptors.addAll(InterceptorBuilder.constructInterceptorReference(packageContext, defaultInterceptorRefName, new HashMap()));
            }
        }

        List externalRefs = baseConfig.getExternalRefs();

        ActionConfig config = new ActionConfig(baseConfig.getMethodName(), baseConfig.getClassName(), params, results, interceptors, externalRefs, packageContext.getName());

        return config;
    }

    //~ Inner Classes //////////////////////////////////////////////////////////

    private class RuntimeConfigurationImpl implements RuntimeConfiguration {
        private Map namespaceActionConfigs;

        public RuntimeConfigurationImpl(Map namespaceActionConfigs) {
            this.namespaceActionConfigs = namespaceActionConfigs;
        }

        /**
         * Gets the configuration information for an action name, or returns null if the
         * name is not recognized.
         *
         * @param name      the name of the action
         * @param namespace the namespace for the action or null for the empty namespace, ""
         * @return the configuration information for action requested
         */
        public synchronized ActionConfig getActionConfig(String namespace, String name) {
            ActionConfig config = null;
            Map actions = (Map) namespaceActionConfigs.get((namespace == null) ? "" : namespace);

            if (actions != null) {
                config = (ActionConfig) actions.get(name);
            }

            // fail over to empty namespace
            if ((config == null) && (namespace != null) && (!namespace.trim().equals(""))) {
                actions = (Map) namespaceActionConfigs.get("");

                if (actions != null) {
                    config = (ActionConfig) actions.get(name);
                }
            }

            return config;
        }

        /**
         * Gets the configuration settings for every action.
         *
         * @return a Map of namespace - > Map of ActionConfig objects, with the key being the action name
         */
        public synchronized Map getActionConfigs() {
            return namespaceActionConfigs;
        }

        public String toString() {
            StringBuffer buff = new StringBuffer("RuntimeConfiguration - actions are\n");

            for (Iterator iterator = namespaceActionConfigs.keySet().iterator();
                    iterator.hasNext();) {
                String namespace = (String) iterator.next();
                Map actionConfigs = (Map) namespaceActionConfigs.get(namespace);

                for (Iterator iterator2 = actionConfigs.keySet().iterator();
                        iterator2.hasNext();) {
                    buff.append(namespace + "/" + iterator2.next() + "\n");
                }
            }

            return buff.toString();
        }
    }
}
