package nl.bimbase.bimworks.client;

import java.util.UUID;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import nl.sascom.backplanepublic.common.Request;
import nl.sascom.backplanepublic.common.Response;

public class FileSystemApi {
	
	private final BimWorksClient bimWorksClient;

	public FileSystemApi(BimWorksClient bimWorksClient) {
		this.bimWorksClient = bimWorksClient;
	}
	
	public UUID createNode(UUID parentNodeUuid, String name, boolean isFolder) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("CreateNode");
		request.put("parent_node_uuid", parentNodeUuid.toString());
		request.put("name", name);
		request.put("folder", isFolder);
		return UUID.fromString(bimWorksClient.executeAsyncTask(request).asText());
	}

	public void deleteNodeIfExists(String path) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("DeleteNodeIfExists");
		request.put("path", path);
		bimWorksClient.executeAsyncTask(request);
	}

	public void deleteNode(UUID nodeUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("DeleteNode");
		request.put("node_uuid", nodeUuid.toString());
		bimWorksClient.executeAsyncTask(request);
	}

	public void deleteNodeVersion(UUID nodeUuid, UUID versionUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("DeleteNodeVersion");
		request.put("node_uuid", nodeUuid.toString());
		request.put("version_uuid", versionUuid.toString());
		bimWorksClient.executeAsyncTask(request);
	}
	
	public void deleteNodeError(UUID nodeUuid, UUID errorUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("DeleteNodeError");
		request.put("node_uuid", nodeUuid.toString());
		request.put("error_uuid", errorUuid.toString());
		bimWorksClient.executeAsyncTask(request);
	}

	public void undeleteNode(UUID nodeUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("UndeleteNode");
		request.put("node_uuid", nodeUuid.toString());
		bimWorksClient.executeAsyncTask(request);
	}

	public TreeNode getRootNode() throws BimWorksException {
		return getNodeByPath("/");
	}

	public UUID createNodeIfNotExists(UUID parentNodeUuid, String name, boolean isFolder) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("CreateNodeIfNotExists");
		request.put("name", name);
		request.put("parent_node_uuid", parentNodeUuid.toString());
		request.put("folder", isFolder);
		return UUID.fromString(bimWorksClient.executeAsyncTask(request).asText());
	}

	public TreeNode getNodeByUuid(UUID nodeUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("GetNodeByUuid");
		request.getObjectInput().put("node_uuid", nodeUuid.toString());
		return Response.OBJECT_MAPPER.convertValue(bimWorksClient.executeAsyncTask(request), TreeNode.class);
	}

	public TreeNodeVersion getNodeVersion(UUID versionUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("GetNodeVersion");
		request.getObjectInput().put("version_uuid", versionUuid.toString());
		return Response.OBJECT_MAPPER.convertValue(bimWorksClient.executeAsyncTask(request), TreeNodeVersion.class);
	}

	public TreeNode getNodeByVersionUuid(UUID versionUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("GetNodeByVersionUuid");
		request.getObjectInput().put("version_uuid", versionUuid.toString());
		return Response.OBJECT_MAPPER.convertValue(bimWorksClient.executeAsyncTask(request), TreeNode.class);
	}

	public ObjectNode listNodes(UUID nodeUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("ListNodes");
		request.getObjectInput().put("parent_node_uuid", nodeUuid.toString());
		request.setTimeOut(1, TimeUnit.MINUTES);
		return (ObjectNode) bimWorksClient.executeAsyncTask(request);
	}

	public ArrayNode listNodeErrors(UUID nodeUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("ListNodeErrors");
		request.getObjectInput().put("node_uuid", nodeUuid.toString());
		request.setTimeOut(1, TimeUnit.MINUTES);
		return (ArrayNode) bimWorksClient.executeAsyncTask(request);
	}

	public ArrayNode listNodeVersions(UUID nodeUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("ListNodeVersions");
		request.getObjectInput().put("node_uuid", nodeUuid.toString());
		request.setTimeOut(1, TimeUnit.MINUTES);
		return (ArrayNode) bimWorksClient.executeAsyncTask(request);
	}

	public ObjectNode listNodesRecursive(UUID nodeUuid) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("ListNodesRecursive");
		request.getObjectInput().put("node_uuid", nodeUuid.toString());
		request.setTimeOut(1, TimeUnit.MINUTES);
		return (ObjectNode) bimWorksClient.executeAsyncTask(request);
	}

	public TreeNode getNodeByPath(String path) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("GetNodeByPath");
		request.getObjectInput().put("path", path);
		request.setTimeOut(1, TimeUnit.MINUTES);
		return BimWorksClient.OBJECT_MAPPER.convertValue(bimWorksClient.executeAsyncTask(request), TreeNode.class);
	}

	public boolean exists(String path) throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("NodeExists");
		request.put("path", path);
		request.setTimeOut(1, TimeUnit.MINUTES);
		return bimWorksClient.executeAsyncTask(request).asBoolean();
	}

	public UUID getRootNodeUuid() throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("GetNodeByPath");
		request.put("path", "/");
		return UUID.fromString(bimWorksClient.executeAsyncTask(request).get("uuid").asText());
	}

	public int clearNodeChildren() throws BimWorksException {
		Request request = bimWorksClient.createRequest();
		request.setTaskName("ClearNodeChildren");
		request.put("node_uuid", getRootNodeUuid().toString());
		request.setTimeOut(10, TimeUnit.MINUTES);
		return bimWorksClient.executeAsyncTask(request).get("uuid").asInt();
	}
}