package nl.bimbase.bimworks.client;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import nl.sascom.backplane.appbase.library.DefaultErrorCode;
import nl.sascom.backplanepublic.client.NodeClient;
import nl.sascom.backplanepublic.common.ClientTask;
import nl.sascom.backplanepublic.common.ExecuteException;
import nl.sascom.backplanepublic.common.LightContainerInterface;
import nl.sascom.backplanepublic.common.NodeClientException;
import nl.sascom.backplanepublic.common.NodeTransport;
import nl.sascom.backplanepublic.common.Request;
import nl.sascom.backplanepublic.common.Response;
import nl.sascom.backplanepublic.common.StreamAlreadyRegisteredException;
import nl.sascom.backplanepublic.common.StreamManager;

public class BimWorksClient implements AutoCloseable {

	public static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

	static {
		OBJECT_MAPPER.findAndRegisterModules();
	}
	
	private final NodeClient nodeClient;
	
	public final FileSystemApi fs;
	public final GisApi gis;
	public final AuthApi auth;
	public final IfcApi ifc;
	public final UploadApi upload;
	public final TokensApi tokens;

	public BimWorksClient(NodeClient nodeClient) {
		this.nodeClient = nodeClient;
		
		this.fs = new FileSystemApi(this);
		this.gis = new GisApi(this);
		this.auth = new AuthApi(this);
		this.ifc = new IfcApi(this);
		this.upload = new UploadApi(this);
		this.tokens = new TokensApi(this);
	}
	
	public BimWorksClient(String connectionUrl) throws Exception {
		this(new NodeClient(connectionUrl));
	}

	public BimWorksClient(NodeTransport nodeTransport) throws Exception {
		this(new NodeClient(nodeTransport));
	}

	public BimWorksClient(NodeTransport nodeTransport, StreamManager streamManager) throws Exception {
		this(new NodeClient(nodeTransport, null, streamManager));
	}

	public BimWorksClient(NodeTransport nodeTransport, LightContainerInterface nodeInterface) throws Exception {
		this(new NodeClient(nodeTransport, nodeInterface));
	}

	@Override
	public void close() throws InterruptedException {
		nodeClient.close();
	}

	public NodeClient getNodeClient() {
		return nodeClient;
	}
	
	public ObjectNode loginWithUsernamePassword(String username, String password) throws BimWorksException {
		try {
			return nodeClient.loginApp(username, password);
		} catch (NodeClientException e) {
			throw new BimWorksException(e);
		}
	}

	public void loginWithApiToken(String apiToken) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("LoginWithApiToken");
		ObjectNode input = request.createObject();
		input.put("token", apiToken);
		request.setInput(input);
		try {
			Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
			if (response.getOutput() != null && response.getOutput().has("auth")) {
				ObjectNode auth = (ObjectNode) response.getOutput().get("auth");
				nodeClient.setAuth(auth);
				nodeClient.getNodeTransport().connectAsync(auth);
			} else {
				throw new BimWorksException("No auth in response to login call " + (response.getErrorNode() != null ? response.getErrorNode().toString() : "") + response);
			}
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
	}



	public ArrayNode query(BimQuery bimQuery, String[] paths, Set<UUID> versionUuids) throws BimWorksException, InterruptedException {
		return query(bimQuery, paths, versionUuids.toArray(new UUID[0]));
	}
	
	public ArrayNode query(BimQuery bimQuery, String[] paths, UUID... versionUuids) throws BimWorksException, InterruptedException {
		Request request = nodeClient.createRequest();
		request.setTaskName("TableExport");
		request.put("type", "JSON");
		request.put("output", "RESPONSE");
		request.getObjectInput().set("query", bimQuery.toJson());
		ArrayNode pathsNode = Response.createArray();
		for (String path : paths) {
			pathsNode.add(path);
		}
		request.getObjectInput().set("paths", pathsNode);
		ArrayNode versionUuidsNode = Response.createArray();
		for (UUID versionUuid : versionUuids) {
			versionUuidsNode.add(versionUuid.toString());
		}
		request.getObjectInput().set("models", versionUuidsNode);
		request.setTimeOut(1, TimeUnit.MINUTES);
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			return task.await(1, TimeUnit.MINUTES).getArrayOutput();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
	}

	public ObjectNode queryDelegated(BimQuery query, String[] paths, Duration validFor, UUID... versionUuids) throws BimWorksException, InterruptedException {
		Request request = nodeClient.createRequest();
		request.setTaskName("GenerateQueryToken");
		request.getObjectInput().set("query", query.toJson());
		request.getObjectInput().put("expires_seconds", validFor.toMillis());
		request.getObjectInput().put("type", "JSON");
		request.getObjectInput().put("output", "RESPONSE");
		ArrayNode pathsNode = Response.createArray();
		for (String path : paths) {
			pathsNode.add(path);
		}
		request.getObjectInput().set("paths", pathsNode);

		ArrayNode versionUuidsNode = Response.createArray();
		for (UUID versionUuid : versionUuids) {
			versionUuidsNode.add(versionUuid.toString());
		}
		request.getObjectInput().set("models", versionUuidsNode);
		request.setTimeOut(1, TimeUnit.MINUTES);
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			return task.await(1, TimeUnit.MINUTES).getObjectOutput();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
	}

	public ObjectNode generateUploadToken(UUID parentUuid, Duration duration, Path path) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("GenerateUploadModelToTreeToken");
		ObjectNode input = request.createObject();
		input.put("node_uuid", parentUuid.toString());
		input.put("expires_seconds", duration.getSeconds());
		input.put("filename", path.getFileName().toString());
		try {
			input.put("filesize", Files.size(path));
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		input.put("contentType", "application/ifc");
		request.setInput(input);
		try {
			Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
			return (ObjectNode) response.getOutput();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
		return null;
	}

	public UploadModelResponse uploadModelWithToken(Path path, ObjectNode payload) throws IOException, StreamAlreadyRegisteredException, NodeAlreadyExistsException, InterruptedException, BimWorksException {
		Request request = nodeClient.createRequest();
		request.setInput(payload.get("input"));
		request.setTask((ObjectNode)payload.get("task"));
		request.setTimeOut(4, TimeUnit.HOURS);
		String streamId = nodeClient.registerStream(com.google.common.io.Files.asByteSource(path.toFile()));
		request.attachStream(streamId);
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			return new UploadModelResponse((ObjectNode) task.await(4, TimeUnit.HOURS).getOutput());
		} catch (ExecuteException e) {
			if (e.getErrorCode().name().equals("DefaultErrorCode") && e.getErrorCode().getCode() == 29) {
				throw new NodeAlreadyExistsException(e.getUserSafeMessage());
			}
			throw new BimWorksException(e);
		}
	}

	public void auth(ObjectNode auth) {
		this.nodeClient.setAuth(auth);
		this.nodeClient.connectAsync(auth);
	}

	public ObjectNode generateDownloadToken(UUID newNodeUuid, Duration duration) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("GenerateDownloadToken");
		ObjectNode input = request.createObject();
		input.put("node_uuid", newNodeUuid.toString());
		input.put("expires_seconds", duration.getSeconds());
		request.setInput(input);
		try {
			Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
			return (ObjectNode) response.getOutput();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
		return null;
	}

	public ObjectNode generateQueryToken(Set<UUID> versionUuids, BimQuery bimQuery, String[] paths, Duration duration) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("GenerateQueryToken");
		ObjectNode input = request.createObject();
		input.put("type", "JSON");
		input.put("output", "RESPONSE");
		ArrayNode versionUuidsNode = Response.createArray();
		for (UUID versionUuid : versionUuids) {
			versionUuidsNode.add(versionUuid.toString());
		}
		input.set("models", versionUuidsNode);
		input.set("query", bimQuery.toJson());
		ArrayNode pathsNode = Response.createArray();
		for (String path : paths) {
			pathsNode.add(path);
		}
		input.set("paths", pathsNode);
		input.put("expires_seconds", duration.getSeconds());
		request.setInput(input);
		try {
			Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
			return (ObjectNode) response.getOutput();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		}
		return null;
	}
	
	public void downloadModelWithToken(ObjectNode objectNode) {
		
	}

	public String getApplicationVersion() throws IOException, ExecuteException {
		Request request = nodeClient.createRequest();
		request.setProject("BimRepository");
		request.setTaskName("GetApplicationVersion");
		Response response = nodeClient.executeSync(request, 10, TimeUnit.SECONDS);
		return response.getOutput().asText();
	}

	public String createApiToken(UUID nodeUuid, String name, Set<String> allowedCorsDomains, Set<String> whiteListedIpAddresses, Set<ApiTokenCredential> credentials) throws ExecuteException, InterruptedException {
		Request request = nodeClient.createRequest();
		request.setTaskName("CreateApiToken");
		request.getObjectInput().put("node_uuid", nodeUuid.toString());
		request.getObjectInput().put("name", name);
		if (allowedCorsDomains != null) {
			request.put("only_cors_enabled_addresses", true);
			ArrayNode allowedCorsDomainsNode = Response.createArray();
			for (String domain : allowedCorsDomains) {
				allowedCorsDomainsNode.add(domain);
			}
			request.set("cors_enabled_domains", allowedCorsDomainsNode);
		}
		if (whiteListedIpAddresses != null) {
			request.put("only_white_listed_ip_addresses", true);
			ArrayNode whiteListedIpAddressesNode = Response.createArray();
			for (String ip : whiteListedIpAddresses) {
				whiteListedIpAddressesNode.add(ip);
			}
			request.set("white_listed_ip_addresses", whiteListedIpAddressesNode);
		}
		if (credentials != null) {
			ArrayNode credentialsNode = Response.createArray();
			for (ApiTokenCredential credential : credentials) {
				credentialsNode.add(credential.name());
			}
			request.set("credentials", credentialsNode);
		}
		request.setTimeOut(1, TimeUnit.MINUTES);
		ClientTask task = nodeClient.createAsyncTask(request);
		task.exec();
		return task.await(1, TimeUnit.MINUTES).getObjectOutput().get("token").asText();
	}

	public ArrayNode queryWithToken(ObjectNode payload) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setInput(payload.get("input"));
		request.setTask((ObjectNode)payload.get("task"));
		request.setTimeOut(4, TimeUnit.HOURS);
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			return task.await(4, TimeUnit.HOURS).getArrayOutput();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			throw new BimWorksException(e.getMessage());
		}
	}

	public ArrayNode listModelsForGis() throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setTaskName("ListModelsForGis");
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			Response response = task.await(30, TimeUnit.SECONDS);
			return response.getArrayOutput();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		return null;
	}

	public NodeClient getApi() {
		return this.nodeClient;
	}

	public String getLastScreenshotNew(UUID lastVersionUuid) throws BimWorksException {
		Request request = nodeClient.createRequest();
		request.setTaskName("GetLastScreenshotNew");
		ArrayNode modelsNode = Response.createArray();
		modelsNode.add(lastVersionUuid.toString());
		request.getObjectInput().set("models", modelsNode);
		ClientTask task = nodeClient.createAsyncTask(request);
		try {
			task.exec();
			Response response = task.await(30, TimeUnit.SECONDS);
			if (response.hasStreams()) {
				
			}
			return null;
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		return null;
	}

	public JsonNode executeAsyncTask(Request request) throws BimWorksException {
		ClientTask task = this.nodeClient.createAsyncTask(request);
		try {
			task.exec();
			Response response = task.await(30, TimeUnit.SECONDS);
			return response.getOutput();
		} catch (ExecuteException e) {
			throw new BimWorksException(e);
		} catch (InterruptedException e) {
			throw new BimWorksException(DefaultErrorCode.INTERRUPTED);
		}
	}

	public Request createRequest() {
		return nodeClient.createRequest();
	}

	public ClientTask createAsyncTask(Request request) {
		return nodeClient.createAsyncTask(request);
	}

	public String registerStream(Path path) throws StreamAlreadyRegisteredException {
		return nodeClient.registerStream(path);
	}

	public String registerStream(String filename, String contentType, String url, long filesize) throws MalformedURLException, StreamAlreadyRegisteredException {
		return nodeClient.registerStream(filename, contentType, url, filesize);
	}

	public String registerStream(String filename, long filesize, String contentType, InputStream inputStream) throws StreamAlreadyRegisteredException {
		return nodeClient.registerStream(filename, filesize, contentType, inputStream);
	}
}