/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Date;
import net.snowflake.client.core.Constants;
import net.snowflake.client.core.ObjectMapperFactory;
import net.snowflake.client.jdbc.internal.fasterxml.jackson.databind.JsonNode;
import net.snowflake.client.jdbc.internal.fasterxml.jackson.databind.ObjectMapper;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;

class FileCacheManager {
    private static final SFLogger LOGGER = SFLoggerFactory.getLogger(FileCacheManager.class);
    private static final ObjectMapper OBJECT_MAPPER = ObjectMapperFactory.getObjectMapper();
    private static final Charset DEFAULT_FILE_ENCODING = Charset.forName("UTF-8");
    private String cacheDirectorySystemProperty;
    private String cacheDirectoryEnvironmentVariable;
    private String baseCacheFileName;
    private long cacheExpirationInMilliseconds;
    private long cacheFileLockExpirationInMilliseconds;
    private File cacheFile;
    private File cacheLockFile;
    private File cacheDir;

    private FileCacheManager() {
    }

    static FileCacheManager builder() {
        return new FileCacheManager();
    }

    FileCacheManager setCacheDirectorySystemProperty(String cacheDirectorySystemProperty) {
        this.cacheDirectorySystemProperty = cacheDirectorySystemProperty;
        return this;
    }

    FileCacheManager setCacheDirectoryEnvironmentVariable(String cacheDirectoryEnvironmentVariable) {
        this.cacheDirectoryEnvironmentVariable = cacheDirectoryEnvironmentVariable;
        return this;
    }

    FileCacheManager setBaseCacheFileName(String baseCacheFileName) {
        this.baseCacheFileName = baseCacheFileName;
        return this;
    }

    FileCacheManager setCacheExpirationInSeconds(long cacheExpirationInSeconds) {
        this.cacheExpirationInMilliseconds = cacheExpirationInSeconds * 1000L;
        return this;
    }

    FileCacheManager setCacheFileLockExpirationInSeconds(long cacheFileLockExpirationInSeconds) {
        this.cacheFileLockExpirationInMilliseconds = cacheFileLockExpirationInSeconds * 1000L;
        return this;
    }

    void overrideCacheFile(File newCacheFile) {
        this.cacheFile = newCacheFile;
        this.cacheDir = newCacheFile.getParentFile();
        this.baseCacheFileName = newCacheFile.getName();
    }

    FileCacheManager build() {
        String cacheDirPath;
        String string = cacheDirPath = this.cacheDirectorySystemProperty != null ? System.getProperty(this.cacheDirectorySystemProperty) : null;
        if (cacheDirPath == null) {
            try {
                cacheDirPath = this.cacheDirectoryEnvironmentVariable != null ? System.getenv(this.cacheDirectoryEnvironmentVariable) : null;
            }
            catch (Throwable ex) {
                LOGGER.info("Cannot get environment variable for cache directory, skip using cache");
                return this;
            }
        }
        if (cacheDirPath != null) {
            this.cacheDir = new File(cacheDirPath);
        } else {
            String homeDir = System.getProperty("user.home");
            if (homeDir == null) {
                homeDir = System.getProperty("java.io.tmpdir");
            }
            this.cacheDir = Constants.getOS() == Constants.OS.WINDOWS ? new File(new File(new File(new File(homeDir, "AppData"), "Local"), "Snowflake"), "Caches") : (Constants.getOS() == Constants.OS.MAC ? new File(new File(new File(homeDir, "Library"), "Caches"), "Snowflake") : new File(new File(homeDir, ".cache"), "snowflake"));
        }
        if (!this.cacheDir.exists() && !this.cacheDir.mkdirs()) {
            throw new RuntimeException(String.format("Failed to locate or create the cache directory: %s", this.cacheDir));
        }
        File cacheFileTmp = new File(this.cacheDir, this.baseCacheFileName).getAbsoluteFile();
        try {
            cacheFileTmp.createNewFile();
            this.cacheFile = cacheFileTmp.getCanonicalFile();
            this.cacheLockFile = new File(this.cacheFile.getParentFile(), this.baseCacheFileName + ".lck");
        }
        catch (IOException | SecurityException ex) {
            throw new RuntimeException(String.format("Failed to touch the cache file: %s", cacheFileTmp.getAbsoluteFile()));
        }
        return this;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    JsonNode readCacheFile() {
        if (this.cacheFile == null) return null;
        if (!this.checkCacheLockFile()) {
            return null;
        }
        try {
            if (!this.cacheFile.exists()) {
                LOGGER.debug("Cache file doesn't exists. File: {}", this.cacheFile);
                return null;
            }
            try (InputStreamReader reader = new InputStreamReader((InputStream)new FileInputStream(this.cacheFile), DEFAULT_FILE_ENCODING);){
                JsonNode jsonNode = OBJECT_MAPPER.readTree(reader);
                return jsonNode;
            }
        }
        catch (IOException ex) {
            LOGGER.debug("Failed to read the cache file. No worry. File: {}, Err: {}", this.cacheFile, ex);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void writeCacheFile(JsonNode input) {
        LOGGER.debug("Writing cache file. File={}", this.cacheFile);
        if (this.cacheFile == null || !this.tryLockCacheFile()) {
            return;
        }
        try {
            if (input == null) {
                return;
            }
            try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(this.cacheFile), DEFAULT_FILE_ENCODING);){
                writer.write(input.toString());
            }
        }
        catch (IOException ex) {
            LOGGER.debug("Failed to write the cache file. File: {}", this.cacheFile);
        }
        finally {
            if (!this.unlockCacheFile()) {
                LOGGER.debug("Failed to unlock cache file");
            }
        }
    }

    void deleteCacheFile() {
        LOGGER.debug("Deleting cache file. File={}, Lock File={}", this.cacheFile, this.cacheLockFile);
        if (this.cacheFile == null) {
            return;
        }
        this.unlockCacheFile();
        if (!this.cacheFile.delete()) {
            LOGGER.debug("Failed to delete the file: {}", this.cacheFile);
        }
    }

    private boolean tryLockCacheFile() {
        boolean locked = false;
        for (int cnt = 0; cnt < 100 && !(locked = this.lockCacheFile()); ++cnt) {
            try {
                Thread.sleep(100L);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        if (!locked) {
            LOGGER.debug("Failed to lock the cache file.");
        }
        return locked;
    }

    private boolean lockCacheFile() {
        return this.cacheLockFile.mkdirs();
    }

    private boolean unlockCacheFile() {
        return this.cacheLockFile.delete();
    }

    private boolean checkCacheLockFile() {
        long currentTime = new Date().getTime();
        long cacheFileTs = FileCacheManager.fileCreationTime(this.cacheFile);
        if (!this.cacheLockFile.exists() && cacheFileTs > 0L && currentTime - this.cacheExpirationInMilliseconds <= cacheFileTs) {
            LOGGER.debug("No cache file lock directory exists and cache file is up to date.");
            return true;
        }
        long lockFileTs = FileCacheManager.fileCreationTime(this.cacheLockFile);
        if (lockFileTs < 0L) {
            return false;
        }
        if (lockFileTs < currentTime - this.cacheFileLockExpirationInMilliseconds) {
            if (!this.cacheLockFile.delete()) {
                LOGGER.debug("Failed to delete the directory. Dir: {}", this.cacheLockFile);
                return false;
            }
            LOGGER.debug("Deleted the cache lock directory, because it was old.");
            return currentTime - this.cacheExpirationInMilliseconds <= cacheFileTs;
        }
        LOGGER.debug("Failed to lock the file. Ignored.");
        return false;
    }

    private static long fileCreationTime(File targetFile) {
        if (!targetFile.exists()) {
            LOGGER.debug("File not exists. File: {}", targetFile);
            return -1L;
        }
        try {
            Path cacheFileLockPath = Paths.get(targetFile.getAbsolutePath(), new String[0]);
            BasicFileAttributes attr = Files.readAttributes(cacheFileLockPath, BasicFileAttributes.class, new LinkOption[0]);
            return attr.creationTime().toMillis();
        }
        catch (IOException ex) {
            LOGGER.debug("Failed to get creation time. File/Dir: {}, Err: {}", targetFile, ex);
            return -1L;
        }
    }
}

