/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core;

import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.Date;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import net.snowflake.client.core.SFException;
import net.snowflake.client.jdbc.ErrorCode;
import net.snowflake.client.jdbc.SnowflakeUtil;
import net.snowflake.client.jdbc.internal.apache.commons.codec.binary.Base64;
import net.snowflake.client.jdbc.internal.com.nimbusds.jose.JOSEException;
import net.snowflake.client.jdbc.internal.com.nimbusds.jose.JWSAlgorithm;
import net.snowflake.client.jdbc.internal.com.nimbusds.jose.JWSHeader;
import net.snowflake.client.jdbc.internal.com.nimbusds.jose.crypto.RSASSASigner;
import net.snowflake.client.jdbc.internal.com.nimbusds.jwt.JWTClaimsSet;
import net.snowflake.client.jdbc.internal.com.nimbusds.jwt.SignedJWT;
import net.snowflake.client.jdbc.internal.google.common.base.Strings;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JceOpenSSLPKCS8DecryptorProviderBuilder;
import org.bouncycastle.operator.InputDecryptorProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.pkcs.PKCS8EncryptedPrivateKeyInfo;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.util.io.pem.PemReader;

class SessionUtilKeyPair {
    static final SFLogger logger = SFLoggerFactory.getLogger(SessionUtilKeyPair.class);
    private final String userName;
    private final String accountName;
    private final PrivateKey privateKey;
    private PublicKey publicKey = null;
    private boolean isFipsMode = false;
    private Provider SecurityProvider = null;
    private static final String ISSUER_FMT = "%s.%s.%s";
    private static final String SUBJECT_FMT = "%s.%s";
    private static final int JWT_DEFAULT_AUTH_TIMEOUT = 10;
    private boolean isBouncyCastleProviderEnabled = false;

    SessionUtilKeyPair(PrivateKey privateKey, String privateKeyFile, String privateKeyFilePwd, String accountName, String userName) throws SFException {
        this.userName = userName.toUpperCase();
        this.accountName = accountName.toUpperCase();
        String enableBouncyCastleJvm = System.getProperty("net.snowflake.jdbc.enableBouncyCastle");
        if (enableBouncyCastleJvm != null) {
            this.isBouncyCastleProviderEnabled = enableBouncyCastleJvm.equalsIgnoreCase("true");
        }
        for (Provider p : Security.getProviders()) {
            if (!"BCFIPS".equals(p.getName())) continue;
            this.isFipsMode = true;
            this.SecurityProvider = p;
            break;
        }
        if (!Strings.isNullOrEmpty(privateKeyFile) && privateKey != null) {
            throw new SFException(ErrorCode.INVALID_OR_UNSUPPORTED_PRIVATE_KEY, "Cannot have both private key value and private key file.");
        }
        PrivateKey privateKey2 = this.privateKey = Strings.isNullOrEmpty(privateKeyFile) ? privateKey : this.extractPrivateKeyFromFile(privateKeyFile, privateKeyFilePwd);
        if (this.privateKey instanceof RSAPrivateCrtKey) {
            RSAPrivateCrtKey rsaPrivateCrtKey = (RSAPrivateCrtKey)this.privateKey;
            RSAPublicKeySpec rsaPublicKeySpec = new RSAPublicKeySpec(rsaPrivateCrtKey.getModulus(), rsaPrivateCrtKey.getPublicExponent());
            try {
                this.publicKey = this.getKeyFactoryInstance().generatePublic(rsaPublicKeySpec);
            }
            catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                throw new SFException(e, ErrorCode.INTERNAL_ERROR, "Error retrieving public key");
            }
        } else {
            throw new SFException(ErrorCode.INVALID_OR_UNSUPPORTED_PRIVATE_KEY, "Use java.security.interfaces.RSAPrivateCrtKey.class for the private key");
        }
    }

    private KeyFactory getKeyFactoryInstance() throws NoSuchAlgorithmException {
        if (this.isFipsMode) {
            return KeyFactory.getInstance("RSA", this.SecurityProvider);
        }
        return KeyFactory.getInstance("RSA");
    }

    private SecretKeyFactory getSecretKeyFactory(String algorithm) throws NoSuchAlgorithmException {
        if (this.isFipsMode) {
            return SecretKeyFactory.getInstance(algorithm, this.SecurityProvider);
        }
        return SecretKeyFactory.getInstance(algorithm);
    }

    private PrivateKey extractPrivateKeyFromFile(String privateKeyFile, String privateKeyFilePwd) throws SFException {
        if (this.isBouncyCastleProviderEnabled) {
            try {
                return this.extractPrivateKeyWithBouncyCastle(privateKeyFile, privateKeyFilePwd);
            }
            catch (IOException | OperatorCreationException | PKCSException e) {
                logger.error("Could not extract private key using Bouncy Castle provider", e);
                throw new SFException(e, ErrorCode.INVALID_OR_UNSUPPORTED_PRIVATE_KEY, e.getCause());
            }
        }
        try {
            return this.extractPrivateKeyWithJdk(privateKeyFile, privateKeyFilePwd);
        }
        catch (IOException | IllegalArgumentException | NullPointerException | InvalidKeyException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            logger.error("Could not extract private key. Try setting the JVM argument: -D{}=TRUE", "net.snowflake.jdbc.enableBouncyCastle");
            throw new SFException(e, ErrorCode.INVALID_OR_UNSUPPORTED_PRIVATE_KEY, privateKeyFile + ": " + e.getMessage());
        }
    }

    public String issueJwtToken() throws SFException {
        JWTClaimsSet.Builder builder = new JWTClaimsSet.Builder();
        String sub = String.format(SUBJECT_FMT, this.accountName, this.userName);
        String iss = String.format(ISSUER_FMT, this.accountName, this.userName, this.calculatePublicKeyFingerprint(this.publicKey));
        Date iat = new Date(System.currentTimeMillis());
        Date exp = new Date(iat.getTime() + 60000L);
        JWTClaimsSet claimsSet = builder.issuer(iss).subject(sub).issueTime(iat).expirationTime(exp).build();
        SignedJWT signedJWT = new SignedJWT(new JWSHeader(JWSAlgorithm.RS256), claimsSet);
        RSASSASigner signer = new RSASSASigner(this.privateKey);
        try {
            signedJWT.sign(signer);
        }
        catch (JOSEException e) {
            throw new SFException(e, ErrorCode.FAILED_TO_GENERATE_JWT, new Object[0]);
        }
        logger.debug("JWT:\n'{'\niss: {}\nsub: {}\niat: {}\nexp: {}\n'}'", iss, sub, String.valueOf(iat.getTime() / 1000L), String.valueOf(exp.getTime() / 1000L));
        return signedJWT.serialize();
    }

    private String calculatePublicKeyFingerprint(PublicKey publicKey) throws SFException {
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            byte[] sha256Hash = md.digest(publicKey.getEncoded());
            return "SHA256:" + Base64.encodeBase64String(sha256Hash);
        }
        catch (NoSuchAlgorithmException e) {
            throw new SFException(e, ErrorCode.INTERNAL_ERROR, "Error when calculating fingerprint");
        }
    }

    public static int getTimeout() {
        String jwtAuthTimeoutStr = SnowflakeUtil.systemGetEnv("JWT_AUTH_TIMEOUT");
        int jwtAuthTimeout = 10;
        if (jwtAuthTimeoutStr != null) {
            jwtAuthTimeout = Integer.parseInt(jwtAuthTimeoutStr);
        }
        return jwtAuthTimeout;
    }

    private PrivateKey extractPrivateKeyWithBouncyCastle(String privateKeyFile, String privateKeyFilePwd) throws IOException, PKCSException, OperatorCreationException {
        PrivateKeyInfo privateKeyInfo = null;
        PEMParser pemParser = new PEMParser((Reader)new FileReader(Paths.get(privateKeyFile, new String[0]).toFile()));
        Object pemObject = pemParser.readObject();
        if (pemObject instanceof PKCS8EncryptedPrivateKeyInfo) {
            PKCS8EncryptedPrivateKeyInfo encryptedPrivateKeyInfo = (PKCS8EncryptedPrivateKeyInfo)pemObject;
            InputDecryptorProvider pkcs8Prov = new JceOpenSSLPKCS8DecryptorProviderBuilder().build(privateKeyFilePwd.toCharArray());
            privateKeyInfo = encryptedPrivateKeyInfo.decryptPrivateKeyInfo(pkcs8Prov);
        } else if (pemObject instanceof PEMKeyPair) {
            privateKeyInfo = ((PEMKeyPair)pemObject).getPrivateKeyInfo();
        } else if (pemObject instanceof PrivateKeyInfo) {
            privateKeyInfo = (PrivateKeyInfo)pemObject;
        }
        pemParser.close();
        JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider(this.isFipsMode ? "BCFIPS" : "BC");
        return converter.getPrivateKey(privateKeyInfo);
    }

    private PrivateKey extractPrivateKeyWithJdk(String privateKeyFile, String privateKeyFilePwd) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException, InvalidKeyException {
        String privateKeyContent = new String(Files.readAllBytes(Paths.get(privateKeyFile, new String[0])));
        if (Strings.isNullOrEmpty(privateKeyFilePwd)) {
            return this.generatePrivateKey(false, privateKeyContent, privateKeyFilePwd);
        }
        return this.generatePrivateKey(true, privateKeyContent, privateKeyFilePwd);
    }

    private PrivateKey generatePrivateKey(boolean isEncrypted, String privateKeyContent, String privateKeyFilePwd) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException, InvalidKeyException {
        if (isEncrypted) {
            try (PemReader pr = new PemReader((Reader)new StringReader(privateKeyContent));){
                byte[] decoded = pr.readPemObject().getContent();
                pr.close();
                EncryptedPrivateKeyInfo pkInfo = new EncryptedPrivateKeyInfo(decoded);
                PBEKeySpec keySpec = new PBEKeySpec(privateKeyFilePwd.toCharArray());
                SecretKeyFactory pbeKeyFactory = this.getSecretKeyFactory(pkInfo.getAlgName());
                PKCS8EncodedKeySpec encodedKeySpec = pkInfo.getKeySpec(pbeKeyFactory.generateSecret(keySpec));
                KeyFactory keyFactory = this.getKeyFactoryInstance();
                PrivateKey privateKey = keyFactory.generatePrivate(encodedKeySpec);
                return privateKey;
            }
        }
        try (PemReader pr = new PemReader((Reader)new StringReader(privateKeyContent));){
            byte[] decoded = pr.readPemObject().getContent();
            pr.close();
            PKCS8EncodedKeySpec encodedKeySpec = new PKCS8EncodedKeySpec(decoded);
            KeyFactory keyFactory = this.getKeyFactoryInstance();
            PrivateKey privateKey = keyFactory.generatePrivate(encodedKeySpec);
            return privateKey;
        }
    }
}

