/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.security.impl;

import java.security.InvalidAlgorithmParameterException;
import java.security.SecureRandom;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import org.apache.commons.codec.binary.Hex;

import net.shibboleth.shared.security.RandomIdentifierParameterSpec;

/**
 * A specialized subclass of {@link RandomIdentifierGenerationStrategy} which constrains the supplied
 * random number generator to be an instance of {@link SecureRandom}.
 */
@ThreadSafe
public class SecureRandomIdentifierGenerationStrategy extends RandomIdentifierGenerationStrategy {

    /**
     * Default constructor.
     * 
     * <p>Initializes the random number source to a new {@link SecureRandom}, size of identifier is set to 16
     * bytes, and the encoder is set to a {@link Hex}.</p>
     */
    public SecureRandomIdentifierGenerationStrategy() {
        
    }

    /**
     * Constructor.
     * 
     * @param params parameter object, must be a {@link RandomIdentifierParameterSpec}
     * 
     * @throws InvalidAlgorithmParameterException if the parameters are invalid
     */
    public SecureRandomIdentifierGenerationStrategy(@Nonnull final ParameterSpec params)
            throws InvalidAlgorithmParameterException {
        super(params);
        
        if (((RandomIdentifierParameterSpec) params).source() != null) {
            if (!(((RandomIdentifierParameterSpec) params).source() instanceof SecureRandom)) {
                throw new InvalidAlgorithmParameterException("Random source was not an instance of SecureRandom");
            }
        }
    }

}