/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.saml.xmlobject.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.xml.namespace.QName;

import org.opensaml.core.xml.AbstractXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.util.AttributeMap;
import org.opensaml.core.xml.util.IndexedXMLObjectChildrenList;
import org.opensaml.core.xml.util.XMLObjectChildrenList;

import net.shibboleth.oidc.saml.xmlobject.DefaultAcrValue;
import net.shibboleth.oidc.saml.xmlobject.OAuthRPExtensions;
import net.shibboleth.oidc.saml.xmlobject.PostLogoutRedirectUri;
import net.shibboleth.oidc.saml.xmlobject.RequestUri;

/**
 * Implementation for {@link OAuthRPExtensions}.
 */
public class OAuthRPExtensionsImpl extends AbstractXMLObject implements OAuthRPExtensions {

    /** Token endpoint authentication method for this entity. */
    private String tokenEndpointAuthMethod;
    
    /** Response types for this entity. */
    private String responseTypes;
    
    /** Grant types for this entity. */
    private String grantTypes;
    
    /** Application type for this entity. */
    private String applicationType;
    
    /** Client URI for this entity. */
    private String clientUri;
    
    /** Scopes for this entity. */
    private String scopes;
    
    /** Software identifier for this entity. */
    private String softwareId;
    
    /** Software version for this entity. */
    private String softwareVersion;
    
    /** Sector identifier URI for this entity. */
    private String sectorIdentifierUri;
    
    /** ID token signed response algorithm for this entity. */
    private String idTokenSignedResponseAlg;
    
    /** ID token encrypted response algorithm for this entity. */
    private String idTokenEncryptedResponseAlg;

    /** ID token encrypted response encoder for this entity. */
    private String idTokenEncryptedResponseEnc;

    /** User info signed response algorithm for this entity. */
    private String userInfoSignedResponseAlg;
    
    /** User info encrypted response algorithm for this entity. */
    private String userInfoEncryptedResponseAlg;

    /** User info encrypted response encoder for this entity. */
    private String userInfoEncryptedResponseEnc;

    /** Request object signing algorithm for this entity. */
    private String requestObjectSigningAlg;
    
    /** Request object encryption algorithm for this entity. */
    private String requestObjectEncryptionAlg;

    /** Request object encryption encoder for this entity. */
    private String requestObjectEncryptionEnc;
    
    /** Token endpoint authentication signing algorithm for this entity. */
    private String tokenEndpointAuthSigningAlg;

    /** Initiate login URI for this entity. */
    private String initiateLoginUri;

    /** Default max age for this entity. */
    private int defaultMaxAge;
    
    /** Require auth time flag for this entity. */
    private boolean requireAuthTime;
    
    /** Default ACR values for this entity. */
    private final XMLObjectChildrenList<DefaultAcrValue> defaultAcrValues;
        
    /** Request URIs for this entity. */
    private final XMLObjectChildrenList<RequestUri> requestUris;
    
    /** Post logout redirect URIs for this entity. */
    private final XMLObjectChildrenList<PostLogoutRedirectUri> postLogoutRedirectUris;
    
    /** "anyAttribute" attributes. */
    private final AttributeMap unknownAttributes;
    
    /** "any" children. */
    private final IndexedXMLObjectChildrenList<XMLObject> unknownChildren;

    /**
     * Constructor.
     * 
     * @param namespaceURI the namespace the element is in
     * @param elementLocalName the local name of the XML element this Object represents
     * @param namespacePrefix the prefix for the given namespace
     */
    protected OAuthRPExtensionsImpl(final String namespaceURI, final String elementLocalName,
            final String namespacePrefix) {
        super(namespaceURI, elementLocalName, namespacePrefix);
        defaultAcrValues = new XMLObjectChildrenList<>(this);
        requestUris = new XMLObjectChildrenList<>(this);
        postLogoutRedirectUris = new XMLObjectChildrenList<>(this);
        unknownAttributes = new AttributeMap(this);
        unknownChildren = new IndexedXMLObjectChildrenList<>(this);
    }
    
    /** {@inheritDoc} */
    public String getResponseTypes() {
        return responseTypes;
    }
    
    /** {@inheritDoc} */
    public void setResponseTypes(final String types) {
        responseTypes = types;
    }

    /** {@inheritDoc} */
    public String getGrantTypes() {
        return grantTypes;
    }

    /** {@inheritDoc} */
    public void setGrantTypes(final String types) {
        grantTypes = types;
    }

    /** {@inheritDoc} */
    public String getApplicationType() {
        return applicationType;
    }

    /** {@inheritDoc} */
    public void setApplicationType(final String type) {
        applicationType = type;
    }

    /** {@inheritDoc} */
    public String getTokenEndpointAuthMethod() {
        return tokenEndpointAuthMethod;
    }

    /** {@inheritDoc} */
    public void setTokenEndpointAuthMethod(final String method) {
        tokenEndpointAuthMethod = method;
    }

    /** {@inheritDoc} */
    public String getClientUri() {
        return clientUri;
    }

    /** {@inheritDoc} */
    public void setClientUri(final String uri) {
        clientUri = uri;
    }

    /** {@inheritDoc} */
    public String getScopes() {
        return scopes;
    }
    
    /** {@inheritDoc} */
    public void setScopes(final String values) {
        scopes = values;
    }

    /** {@inheritDoc} */
    public String getSoftwareId() {
        return softwareId;
    }

    /** {@inheritDoc} */
    public void setSoftwareId(final String id) {
        softwareId = id;
    }

    /** {@inheritDoc} */
    public String getSoftwareVersion() {
        return softwareVersion;
    }

    /** {@inheritDoc} */
    public void setSoftwareVersion(final String version) {
        softwareVersion = version;
    }

    /** {@inheritDoc} */
    public String getSectorIdentifierUri() {
        return sectorIdentifierUri;
    }

    /** {@inheritDoc} */
    public void setSectorIdentifierUri(final String uri) {
        sectorIdentifierUri = uri;
    }

    /** {@inheritDoc} */
    public String getIdTokenSignedResponseAlg() {
        return idTokenSignedResponseAlg;
    }

    /** {@inheritDoc} */
    public void setIdTokenSignedResponseAlg(final String algorithm) {
        idTokenSignedResponseAlg = algorithm;
    }

    /** {@inheritDoc} */
    public String getIdTokenEncryptedResponseAlg() {
        return  idTokenEncryptedResponseAlg;
    }

    /** {@inheritDoc} */
    public void setIdTokenEncryptedResponseAlg(final String algorithm) {
        idTokenEncryptedResponseAlg = algorithm;
    }

    /** {@inheritDoc} */
    public String getIdTokenEncryptedResponseEnc() {
        return idTokenEncryptedResponseEnc;
    }

    /** {@inheritDoc} */
    public void setIdTokenEncryptedResponseEnc(final String algorithm) {
        idTokenEncryptedResponseEnc = algorithm;
    }

    /** {@inheritDoc} */
    public String getUserInfoSignedResponseAlg() {
        return userInfoSignedResponseAlg;
    }

    /** {@inheritDoc} */
    public void setUserInfoSignedResponseAlg(final String algorithm) {
        userInfoSignedResponseAlg = algorithm;
    }

    /** {@inheritDoc} */
    public String getUserInfoEncryptedResponseAlg() {
        return userInfoEncryptedResponseAlg;
    }

    /** {@inheritDoc} */
    public void setUserInfoEncryptedResponseAlg(final String algorithm) {
        userInfoEncryptedResponseAlg = algorithm;
    }

    /** {@inheritDoc} */
    public String getUserInfoEncryptedResponseEnc() {
        return userInfoEncryptedResponseEnc;
    }

    /** {@inheritDoc} */
    public void setUserInfoEncryptedResponseEnc(final String algorithm) {
        userInfoEncryptedResponseEnc = algorithm;
    }

    /** {@inheritDoc} */
    public String getRequestObjectSigningAlg() {
        return requestObjectSigningAlg;
    }

    /** {@inheritDoc} */
    public void setRequestObjectSigningAlg(final String algorithm) {
        requestObjectSigningAlg = algorithm;
    }

    /** {@inheritDoc} */
    public String getRequestObjectEncryptionAlg() {
        return requestObjectEncryptionAlg;
    }

    /** {@inheritDoc} */
    public void setRequestObjectEncryptionAlg(final String algorithm) {
        requestObjectEncryptionAlg = algorithm;
    }

    /** {@inheritDoc} */
    public String getRequestObjectEncryptionEnc() {
        return requestObjectEncryptionEnc;
    }

    /** {@inheritDoc} */
    public void setRequestObjectEncryptionEnc(final String algorithm) {
        requestObjectEncryptionEnc = algorithm;
    }

    /** {@inheritDoc} */
    public String getTokenEndpointAuthSigningAlg() {
        return tokenEndpointAuthSigningAlg;
    }

    /** {@inheritDoc} */
    public void setTokenEndpointAuthSigningAlg(final String algorithm) {
        tokenEndpointAuthSigningAlg = algorithm;
    }


    /** {@inheritDoc} */
    public String getInitiateLoginUri() {
        return initiateLoginUri;
    }

    /** {@inheritDoc} */
    public void setInitiateLoginUri(final String uri) {
        initiateLoginUri = uri;
    }

    /** {@inheritDoc} */
    public int getDefaultMaxAge() {
        return defaultMaxAge;
    }

    /** {@inheritDoc} */
    public void setDefaultMaxAge(final int age) {
        defaultMaxAge = age;
    }

    /** {@inheritDoc} */
    public boolean isRequireAuthTime() {
        return requireAuthTime;
    }

    /** {@inheritDoc} */
    public void setRequireAuthTime(final boolean flag) {
        requireAuthTime = flag;
    }

    /** {@inheritDoc} */
    public List<DefaultAcrValue> getDefaultAcrValues() {
        return defaultAcrValues;
    }

    /** {@inheritDoc} */
    public List<RequestUri> getRequestUris() {
        return requestUris;
    }

    /** {@inheritDoc} */
    public List<PostLogoutRedirectUri> getPostLogoutRedirectUris() {
        return postLogoutRedirectUris;
    }
    
    /** {@inheritDoc} */
    @Override
    public List<XMLObject> getOrderedChildren() {
        final ArrayList<XMLObject> children = new ArrayList<>();

        children.addAll(getDefaultAcrValues());
        children.addAll(getRequestUris());
        children.addAll(getPostLogoutRedirectUris());
        children.addAll(getUnknownXMLObjects());

        return Collections.unmodifiableList(children);
    }
    
    /** {@inheritDoc} */
    @Override
    public AttributeMap getUnknownAttributes() {
        return unknownAttributes;
    }
    
    /**
     * {@inheritDoc}
     */
    public List<XMLObject> getUnknownXMLObjects() {
        return unknownChildren;
    }
    
    /** {@inheritDoc} */
    public List<XMLObject> getUnknownXMLObjects(final QName typeOrName) {
        return (List<XMLObject>) unknownChildren.subList(typeOrName);
    }

}
