/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.saml.xmlobject;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.opensaml.core.xml.AttributeExtensibleXMLObject;
import org.opensaml.core.xml.ElementExtensibleXMLObject;
import org.opensaml.saml.common.SAMLObject;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * SAML 2.0 Metadata extension OAuthRPExtensions.
 */
public interface OAuthRPExtensions extends SAMLObject, AttributeExtensibleXMLObject, ElementExtensibleXMLObject {
    
    /** Local name of the XSI type. */
    @Nonnull @NotEmpty public static final String TYPE_LOCAL_NAME = "OAuthRPExtensions";

    /** QName of the XSI type. */
    @Nonnull public static final QName TYPE_NAME = new QName(MetadataValueSAMLObject.SAML20MDOIDCMD_NS, TYPE_LOCAL_NAME,
            MetadataValueSAMLObject.SAML20MDOIDCMD_PREFIX);
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String TOKEN_ENDPOINT_AUTH_METHOD_ATTRIB_NAME = "token_endpoint_auth_method";

    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String GRANT_TYPES_ATTRIB_NAME = "grant_types";

    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String RESPONSE_TYPES_ATTRIB_NAME = "response_types";

    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String APPLICATION_TYPE_ATTRIB_NAME = "application_type";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String CLIENT_URI_ATTRIB_NAME = "client_uri";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String SCOPES_ATTRIB_NAME = "scopes";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String SOFTWARE_ID_ATTRIB_NAME = "software_id";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String SOFTWARE_VERSION_ATTRIB_NAME = "software_version";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String SECTOR_IDENTIFIER_URI_ATTRIB_NAME = "sector_identifier_uri";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String ID_TOKEN_SIGNED_RESPONSE_ALG_ATTRIB_NAME =
            "id_token_signed_response_alg";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String ID_TOKEN_ENCRYPTED_RESPONSE_ALG_ATTRIB_NAME =
            "id_token_encrypted_response_alg";

    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String ID_TOKEN_ENCRYPTED_RESPONSE_ENC_ATTRIB_NAME =
            "id_token_encrypted_response_enc";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String USERINFO_SIGNED_RESPONSE_ALG_ATTRIB_NAME =
            "userinfo_signed_response_alg";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String USERINFO_ENCRYPTED_RESPONSE_ALG_ATTRIB_NAME =
            "userinfo_encrypted_response_alg";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String USERINFO_ENCRYPTED_RESPONSE_ENC_ATTRIB_NAME =
            "userinfo_encrypted_response_enc";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String REQUEST_OBJECT_SIGNING_ALG_ATTRIB_NAME =
            "request_object_signing_alg";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String REQUEST_OBJECT_ENCRYPTION_ALG_ATTRIB_NAME =
            "request_object_encryption_alg";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String REQUEST_OBJECT_ENCRYPTION_ENC_ATTRIB_NAME =
            "request_object_encryption_enc";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String TOKEN_ENDPOINT_AUTH_SIGNING_ALG_ATTRIB_NAME =
            "token_endpoint_auth_signing_alg";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String INITIATE_LOGIN_URI_ATTRIB_NAME = "initiate_login_uri";
    
    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String DEFAULT_MAX_AGE_ATTRIB_NAME = "default_max_age";

    /** Attribute local name. */
    @Nonnull @NotEmpty public static final String REQUIRE_AUTH_TIME_ATTRIB_NAME = "require_auth_time";
    
    /**
     * Get the token endpoint authentication method.
     * 
     * @return token endpoint authentication method
     */
    @Nullable public String getTokenEndpointAuthMethod();
    
    /**
     * Set the token endpoint authentication method.
     * 
     * @param method The token endpoint authentication method.
     */
    public void setTokenEndpointAuthMethod(@Nullable final String method);
    
    /**
     * Get the whitespace-delimited list of response types.
     * 
     * @return list of response types
     */
    @Nullable public String getResponseTypes();
    
    /**
     * Set the whitespace-delimited list of response types.
     * 
     * @param types response types
     */
    public void setResponseTypes(@Nullable final String types);
    
    /**
     * Get the whitespace-delimited list of grant types.
     * 
     * @return list of grant types
     */
    @Nullable public String getGrantTypes();
    
    /**
     * Set the whitespace-delimited list of grant types.
     * 
     * @param types grant types
     */
    public void setGrantTypes(@Nullable final String types);

    /**
     * Get the application type.
     * 
     * @return The application type.
     */
    @Nullable public String getApplicationType();
    
    /**
     * Set the application type.
     * 
     * @param type The application type.
     */
    public void setApplicationType(@Nullable final String type);

    /**
     * Get the client URI.
     * 
     * @return The client URI
     */
    @Nullable public String getClientUri();
    
    /**
     * Set the client URI.
     * 
     * @param uri The client URI.
     */
    public void setClientUri(@Nullable final String uri);
    
    /**
     * Get the whitespace-delimited list of scopes.
     * 
     * @return list of scopes
     */
    @Nullable public String getScopes();
    
    /**
     * Set the whitespace-delimited list of scopes.
     * 
     * @param scopes the scopes
     */
    public void setScopes(@Nullable final String scopes);
    
    /**
     * Get the software identifier.
     * 
     * @return software identifier
     */
    @Nullable public String getSoftwareId();
    
    /**
     * Set the software identifier.
     * 
     * @param id The software identifier.
     */
    public void setSoftwareId(@Nullable final String id);
    
    /**
     * Get the software version.
     * 
     * @return The software version.
     */
    @Nullable public String getSoftwareVersion();
    
    /**
     * Set the software version.
     * 
     * @param version The software version.
     */
    public void setSoftwareVersion(@Nullable final String version);
    
    /**
     * Get the sector identifier URI.
     * 
     * @return he sector identifier URI.
     */
    @Nullable public String getSectorIdentifierUri();
    
    /**
     * Set the sector identifier URI.
     * 
     * @param uri The sector identifier URI.
     */
    public void setSectorIdentifierUri(@Nullable final String uri);

    /**
     * Get the JWS alg algorithm for ID token signature.
     * 
     * @return The JWS alg algorithm for ID token signature.
     */
    @Nullable public String getIdTokenSignedResponseAlg();
    
    /**
     * Set the JWS alg algorithm for ID token signature.
     * 
     * @param algorithm The JWS alg algorithm for ID token signature.
     */
    public void setIdTokenSignedResponseAlg(@Nullable final String algorithm);

    /**
     * Get the JWE alg algorithm for ID token encryption.
     * 
     * @return The JWE alg algorithm for ID token encryption.
     */
    @Nullable public String getIdTokenEncryptedResponseAlg();
    
    /**
     * Set the JWE alg algorithm for ID token encryption.
     * 
     * @param algorithm The JWE alg algorithm for ID token encryption.
     */
    public void setIdTokenEncryptedResponseAlg(@Nullable final String algorithm);

    /**
     * Get the JWE enc algorithm for ID token encryption.
     * 
     * @return The JWE enc algorithm for ID token encryption.
     */
    @Nullable public String getIdTokenEncryptedResponseEnc();
    
    /**
     * Set the JWE enc algorithm for ID token encryption.
     * 
     * @param algorithm The JWE enc algorithm for ID token encryption.
     */
    public void setIdTokenEncryptedResponseEnc(@Nullable final String algorithm);

    /**
     * Get the JWS alg algorithm for UserInfo response signature.
     * 
     * @return The JWS alg algorithm for UserInfo response signature.
     */
    @Nullable public String getUserInfoSignedResponseAlg();
    
    /**
     * Set the JWS alg algorithm for UserInfo response signature.
     * 
     * @param algorithm The JWS alg algorithm for UserInfo response signature.
     */
    public void setUserInfoSignedResponseAlg(@Nullable final String algorithm);
    
    /**
     * Get the JWE alg algorithm for UserInfo response encryption.
     * 
     * @return The JWE alg algorithm for UserInfo response encryption.
     */
    @Nullable public String getUserInfoEncryptedResponseAlg();
    
    /**
     * Set the JWE alg algorithm for UserInfo response encryption.
     * 
     * @param algorithm The JWE alg algorithm for UserInfo response encryption.
     */
    public void setUserInfoEncryptedResponseAlg(@Nullable final String algorithm);

    /**
     * Get the JWE enc algorithm for UserInfo response encryption.
     * 
     * @return The JWE enc algorithm for UserInfo response encryption.
     */
    @Nullable public String getUserInfoEncryptedResponseEnc();
    
    /**
     * Set the JWE enc algorithm for UserInfo response encryption.
     * 
     * @param algorithm The JWE enc algorithm for UserInfo response encryption.
     */
    public void setUserInfoEncryptedResponseEnc(@Nullable final String algorithm);
    
    /**
     * Get the JWS alg algorithm for request object signature.
     * 
     * @return The JWS alg algorithm for request object signature.
     */
    @Nullable public String getRequestObjectSigningAlg();
    
    /**
     * Set the JWS alg algorithm for request object signature.
     * 
     * @param algorithm The JWS alg algorithm for request object signature.
     */
    public void setRequestObjectSigningAlg(@Nullable final String algorithm);
    
    /**
     * Get the JWE alg algorithm for request object encryption.
     * 
     * @return The JWE alg algorithm for request object encryption.
     */
    @Nullable public String getRequestObjectEncryptionAlg();
    
    /**
     * Set the JWE alg algorithm for request object encryption.
     * 
     * @param algorithm The JWE alg algorithm for request object encryption.
     */
    public void setRequestObjectEncryptionAlg(@Nullable final String algorithm);

    /**
     * Get the JWE enc algorithm for request object encryption.
     * 
     * @return The JWE enc algorithm for request object encryption.
     */
    @Nullable public String getRequestObjectEncryptionEnc();
    
    /**
     * Set the JWE enc algorithm for request object encryption.
     * 
     * @param algorithm The JWE enc algorithm for request object encryption.
     */
    public void setRequestObjectEncryptionEnc(@Nullable final String algorithm);
    
    /**
     * Get the JWS alg algorithm for token endpoint authentication signature.
     * 
     * @return The JWS alg algorithm for token endpoint authentication signature.
     */
    @Nullable public String getTokenEndpointAuthSigningAlg();
    
    /**
     * Set the JWS alg algorithm for token endpoint authentication signature.
     * 
     * @param algorithm The JWS alg algorithm for token endpoint authentication signature.
     */
    public void setTokenEndpointAuthSigningAlg(@Nullable final String algorithm);
    
    /**
     * Get the URI for initiating login.
     * 
     * @return The URI for initiating login.
     */
    @Nullable public String getInitiateLoginUri();
    
    /**
     * Set the URI for initiating login.
     * 
     * @param uri The URI for initiating login.
     */
    public void setInitiateLoginUri(@Nullable final String uri);

    /**
     * Get the value for default max age.
     * 
     * @return The value for default max age.
     */
    public int getDefaultMaxAge();
    
    /**
     * Set the value for default max age.
     * 
     * @param age The value for default max age.
     */
    public void setDefaultMaxAge(final int age);
    
    /**
     * Get the flag to require authentication time.
     * 
     * @return The flag to require authentication time.
     */
    public boolean isRequireAuthTime();
    
    /**
     * Set the flag to require authentication time.
     * 
     * @param flag The flag to require authentication time.
     */
    public void setRequireAuthTime(final boolean flag);

    /**
     * Get the list of default ACR values.
     * 
     * @return The list of default ACR values.
     */
    public List<DefaultAcrValue> getDefaultAcrValues();

    /**
     * Get the list of request URIs.
     * 
     * @return The list of request URIs.
     */
    public List<RequestUri> getRequestUris();
    
    /**
     * Get the list of post logout redirection URIs.
     * 
     * @return The list of post logout redirection URIs.
     */
    public List<PostLogoutRedirectUri> getPostLogoutRedirectUris();

}
