/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.logic;

import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.openid.connect.sdk.op.OIDCProviderMetadata;

import net.shibboleth.oidc.metadata.context.OIDCProviderMetadataContext;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * A function that returns the issuer id via the provider information stored in {@link OIDCProviderMetadataContext}.
 */
public class IssuerIDFromOIDCProviderMetadataContextLookupFunction
        implements BiFunction<ProfileRequestContext, JWTClaimsSet, String> {

    /** Strategy that will return {@link OIDCProviderMetadataContext}. */
    @Nonnull private Function<ProfileRequestContext, OIDCProviderMetadataContext> oidcMetadataContextLookupStrategy;

    /**
     * Constructor.
     */
    public IssuerIDFromOIDCProviderMetadataContextLookupFunction() {
        oidcMetadataContextLookupStrategy = new ChildContextLookup<>(OIDCProviderMetadataContext.class).compose(
                new InboundMessageContextLookup());
    }

    /**
     * Set the strategy used to return the {@link OIDCProviderMetadataContext}.
     * 
     * @param strategy The lookup strategy.
     */
    public void setOIDCMetadataContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCProviderMetadataContext> strategy) {
        oidcMetadataContextLookupStrategy =
                Constraint.isNotNull(strategy, "OIDCMetadataContext lookup strategy cannot be null");
    }

    @Override @Nullable
    public String apply(@Nonnull final ProfileRequestContext prc, @Nullable final JWTClaimsSet claimsSet) {
        final OIDCProviderMetadataContext oidcMetadataContext = oidcMetadataContextLookupStrategy.apply(prc);
        if (oidcMetadataContext == null) {
            return null;
        }
        final OIDCProviderMetadata providerInformation = oidcMetadataContext.getProviderInformation();
        if (providerInformation == null || providerInformation.getIssuer() == null) {
            return null;
        }
        return  providerInformation.getIssuer().toString();
    }
    
}