/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.impl;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.opensaml.messaging.encoder.MessageEncoder;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.oidc.profile.config.OIDCAuthorizationConfiguration;
import net.shibboleth.oidc.profile.config.OIDCAuthorizationConfiguration.OIDCHttpRequestMethod;
import net.shibboleth.oidc.profile.encoder.OIDCMessageEncoder;
import net.shibboleth.utilities.java.support.annotation.ParameterName;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.component.AbstractInitializableComponent;

/**
 * Message encoder factory function that returns the first encoder suitable for the given request method found in the
 * profile configuration.
 */
@ThreadSafe
public class AuthenticationRequestMessageEncoderFactory extends AbstractInitializableComponent
        implements Function<ProfileRequestContext, MessageEncoder> {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(AuthenticationRequestMessageEncoderFactory.class);

    /** The list of message encoders to choose from. */
    @Nonnull @Unmodifiable @NotLive private final List<OIDCMessageEncoder> encoders;

    /**
     * 
     * Constructor.
     *
     * @param encodersToUse the list of possible encoders to use.
     */
    public AuthenticationRequestMessageEncoderFactory(
            @Nullable @ParameterName(name = "encoders") final List<OIDCMessageEncoder> encodersToUse) {
        if (encodersToUse == null) {
            encoders = Collections.emptyList();
        } else {
            encoders = Collections.unmodifiableList(encodersToUse);
        }
    }

    @Override
    @Nullable public MessageEncoder apply(@Nonnull final ProfileRequestContext profileRequestContext) {

        final RelyingPartyContext rpCtx = profileRequestContext.getSubcontext(RelyingPartyContext.class);
        OIDCAuthorizationConfiguration profileConfiguration = null;

        if (rpCtx != null && rpCtx.getProfileConfig() instanceof OIDCAuthorizationConfiguration) {
            profileConfiguration = (OIDCAuthorizationConfiguration) rpCtx.getProfileConfig();
        }
        if (profileConfiguration == null) {
            log.warn("OIDCAuthorizationConfiguration not found, no encoders to lookup");
            return null;
        }

        final OIDCHttpRequestMethod requestMethodFromConfig =
                profileConfiguration.getHttpRequestMethod(profileRequestContext);

        if (requestMethodFromConfig == null) {
            log.warn("Authentication request method not found on profile, no encoders to lookup");
            return null;
        }

        final Optional<OIDCMessageEncoder> encoder =
                encoders.stream().filter(enc -> enc.test(requestMethodFromConfig)).findFirst();
        if (encoder.isPresent()) {
            log.trace("Returning OIDC message encoder of type '{}'", encoder.get().getClass());
        } else {
            log.warn("No message encoder was found for authentication request method type '{}'",
                    requestMethodFromConfig);
        }

        return encoder.orElse(null);
    }

}
