/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.encoder.impl;

import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletResponse;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.encoder.MessageEncoder;
import org.opensaml.messaging.encoder.MessageEncodingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.profile.config.OIDCAuthorizationConfiguration.OIDCHttpRequestMethod;
import net.shibboleth.oidc.profile.core.OIDCAuthenticationRequest;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.codec.HTMLEncoder;
import net.shibboleth.utilities.java.support.collection.Pair;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.net.HttpServletSupport;

/**
 * A {@link MessageEncoder message encoder} that encodes an OpenID authentication request by
 * HTTP Form POST Serialization.
 */
public class HTTPPostAuthnRequestEncoder extends AbstractOIDCMessageEncoder {
    
    /** Default template ID for using FORM POST request type. */
    @Nonnull @NotEmpty public static final String DEFAULT_TEMPLATE_ID = "/templates/oidc-request-form-post.vm";
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(HTTPPostAuthnRequestEncoder.class);
    
    /** Velocity engine used to evaluate the template when using FORM POST response mode. */
    @Nullable private VelocityEngine velocityEngine;

    /** ID of the Velocity template used when using FORM POST response mode. */
    @Nonnull @NotEmpty private String velocityTemplateId = DEFAULT_TEMPLATE_ID;
    
    /**
     * Set the Velocity template id.
     * 
     * <p>
     * Defaults to {@link #DEFAULT_TEMPLATE_ID}.
     * </p>
     * 
     * @param newVelocityTemplateId the new Velocity template id
     */
    public void setVelocityTemplateId(@Nonnull @NotEmpty final String newVelocityTemplateId) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        velocityTemplateId = 
                Constraint.isNotEmpty(newVelocityTemplateId, "Velocity template id must not not be null or empty");
    }

    /**
     * Set the VelocityEngine instance.
     * 
     * @param newVelocityEngine the new VelocityEngine instane
     */
    public void setVelocityEngine(@Nonnull final VelocityEngine newVelocityEngine) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        velocityEngine = Constraint.isNotNull(newVelocityEngine, "Velocity engine can not be null");
    }
    

    @Override
    public boolean test(@Nonnull final OIDCHttpRequestMethod requestMethod) {
       return requestMethod == OIDCHttpRequestMethod.POST;
    }   
    
    /**
     * Construct form POST.
     * 
     * @param request the authentication request.
     * @return response message as velocity context.
     */
    private VelocityContext doPostEncode(@Nonnull final OIDCAuthenticationRequest request) {
        final VelocityContext context = new VelocityContext();
        final List<Pair<String, String>> params = createParametersFromRequest(request);
        params.forEach(param -> context.put(param.getFirst(), param.getSecond()));
        context.put("action", HTMLEncoder.encodeForHTMLAttribute(request.getEndpointURI().toString()));
        log.trace("Velocity context {}", params);
        return context;
    }    

    @Override
    protected void doEncode() throws MessageEncodingException {
        
        if (velocityEngine == null) {
            throw new MessageEncodingException("VelocityEngine must be supplied for form post request mode");
        }
        
        log.debug("Encoding OIDC authentication request using HTTP Form Post Serialization");
        final MessageContext messageContext = getMessageContext();
        final Object outboundMessage = messageContext.getMessage();
        if (!(outboundMessage instanceof OIDCAuthenticationRequest)) {
            throw new MessageEncodingException("No outbound OIDC authentication request message "
                    + "contained in message context");
        }
        try {
            final HttpServletResponse response = getHttpServletResponse();
            HttpServletSupport.addNoCacheHeaders(response);
            HttpServletSupport.setUTF8Encoding(response);
            HttpServletSupport.setContentType(response, "text/html");
            final VelocityContext context = doPostEncode((OIDCAuthenticationRequest) outboundMessage);
            try (final Writer out = new OutputStreamWriter(response.getOutputStream(), StandardCharsets.UTF_8)) {
                velocityEngine.mergeTemplate(velocityTemplateId, "UTF-8", context, out);
                out.flush();
            }
        } catch (final Exception e) {
            log.error("Error creating authorization POST request: {}", e.getMessage());
            throw new MessageEncodingException("Error creating authorization POST request", e);
        }
        
        
    }


}
