/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.decoding.impl;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.decoder.MessageDecoder;
import org.opensaml.messaging.decoder.MessageDecodingException;
import org.opensaml.messaging.decoder.servlet.AbstractHttpServletRequestMessageDecoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.http.ServletUtils;
import com.nimbusds.openid.connect.sdk.AuthenticationResponse;
import com.nimbusds.openid.connect.sdk.AuthenticationResponseParser;

import net.shibboleth.oidc.profile.decoding.OIDCMessageDecoder;

/** 
 * A {@link MessageDecoder message decoder} that decodes an incoming {@link AuthenticationResponse}
 * when using a form_post response_type.
 */
//TODO this is identical to the HTTPRedirectAuthnDecoder because it uses the Nimbus parser.
// do we leave as a placeholder for when we create our own decoders?
public class HTTPPostAuthnResponseDecoder extends AbstractHttpServletRequestMessageDecoder
                    implements OIDCMessageDecoder {
    
    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(HTTPPostAuthnResponseDecoder.class);

    @Override
    protected void doDecode() throws MessageDecodingException {
        
        log.trace("Decoding incomming 'form_post' authentication response");
        
        final MessageContext messageContext = new MessageContext();
        final HttpServletRequest request = getHttpServletRequest();
        
        if (!"POST".equalsIgnoreCase(request.getMethod())) {
            throw new MessageDecodingException("This message decoder only supports the HTTP POST method");
        }
        
        try {
            final AuthenticationResponse inboundMessage = 
                    AuthenticationResponseParser.parse(ServletUtils.createHTTPRequest(request));
            messageContext.setMessage(inboundMessage);
        } catch (final ParseException | IOException e) {
            log.error("Unable to parse incomming authentication response",e);
        }
        setMessageContext(messageContext);        
    }
}
