/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.oauth2.config;

import java.time.Duration;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.google.common.base.Predicates;

import net.shibboleth.idp.profile.config.AbstractConditionalProfileConfiguration;
import net.shibboleth.idp.profile.config.AttributeResolvingProfileConfiguration;
import net.shibboleth.idp.profile.config.OverriddenIssuerProfileConfiguration;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Interface for OAuth 2 token "audience" profile configuration.
 * 
 * <p>This applies to behavior controlling the characteristics of tokens issued
 * to the parties intended to process them, as distinct from the clients that use them.</p>
 */
public class OAuth2TokenAudienceConfiguration  extends AbstractConditionalProfileConfiguration
        implements OAuth2ProfileConfiguration, OverriddenIssuerProfileConfiguration, 
                    AttributeResolvingProfileConfiguration {

    /** OAuth2 Token Revocation URI. */
    @Nonnull @NotEmpty public static final String PROTOCOL_URI = "https://tools.ietf.org/html/rfc6749";

    /** ID for this profile configuration. */
    @Nonnull @NotEmpty 
    public static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/oauth2/token/audience";
    
    /** Lookup function to override issuer value. */
    @Nonnull private Function<ProfileRequestContext,String> issuerLookupStrategy;

    /** Whether attributes should be resolved in the course of the profile. */
    @Nonnull private Predicate<ProfileRequestContext> resolveAttributesPredicate;
    
    /** Whether encryption is optional in the face of no key, etc. */
    @Nonnull private Predicate<ProfileRequestContext> encryptionOptionalPredicate;

    /** Lookup functioon to supply access token type. */
    @Nonnull private Function<ProfileRequestContext,String> accessTokenTypeLookupStrategy;

    /** Lookup function to supply access token lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> accessTokenLifetimeLookupStrategy;

    /** Constructor. */
    public OAuth2TokenAudienceConfiguration() {
        this(PROFILE_ID);
    }

    /**
     * Creates a new configuration instance.
     *
     * @param profileId Unique profile identifier.
     */
    public OAuth2TokenAudienceConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);

        issuerLookupStrategy = FunctionSupport.constant(null);
        
        resolveAttributesPredicate = Predicates.alwaysTrue();
        encryptionOptionalPredicate = Predicates.alwaysFalse();
        
        accessTokenTypeLookupStrategy = FunctionSupport.constant(null);
        accessTokenLifetimeLookupStrategy = FunctionSupport.constant(Duration.ofMinutes(10));
    }
    
    /** {@inheritDoc} */
    @Override
    @Nullable @NotEmpty public String getIssuer(@Nullable final ProfileRequestContext profileRequestContext) {
        return issuerLookupStrategy.apply(profileRequestContext);
    }
    
    /**
     * Set overridden issuer value.
     * 
     * @param issuer issuer value
     */
    public void setIssuer(@Nullable @NotEmpty final String issuer) {
        issuerLookupStrategy = FunctionSupport.constant(issuer);
    }
    
    /**
     * Sets lookup strategy for overridden issuer value.
     * 
     * @param strategy lookup strategy
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        issuerLookupStrategy = Constraint.isNotNull(strategy, "Issuer lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    public boolean isResolveAttributes(@Nullable final ProfileRequestContext profileRequestContext) {
        return resolveAttributesPredicate.test(profileRequestContext);
    }

    /**
     * Set whether attributes should be resolved during the profile.
     * 
     * @param flag flag to set
     */
    public void setResolveAttributes(final boolean flag) {
        resolveAttributesPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }
    
    /**
     * Set a condition to determine whether attributes should be resolved during the profile.
     * 
     * @param condition condition to set
     */
    public void setResolveAttributesPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        resolveAttributesPredicate = Constraint.isNotNull(condition, "Resolve attributes predicate cannot be null");
    }
    
    /**
     * Get whether encryption is optional in the face of a missing key, etc.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return true iff encryption is optional
     */
    public boolean isEncryptionOptional(@Nullable final ProfileRequestContext profileRequestContext) {
        return encryptionOptionalPredicate.test(profileRequestContext);
    }
    
    /**
     * Set whether encryption is optional in the face of a missing key, etc.
     * 
     * @param flag  flag to set
     */
    public void setEncryptionOptional(final boolean flag) {
        encryptionOptionalPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set a condition to determine whether encryption is optional in the face of a missing key, etc.
     *
     * @param condition condition to set
     */
    public void setEncryptionOptionalPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        encryptionOptionalPredicate = Constraint.isNotNull(condition, "Encryption optional predicate cannot be null");
    }
    
    /**
     * Get access token type.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return access token type, or null for unspecified/opaque
     */
     @Nullable @NotEmpty public String getAccessTokenType(@Nullable final ProfileRequestContext profileRequestContext) {
         return accessTokenTypeLookupStrategy.apply(profileRequestContext);
     }
    
    /**
     * Set access token type.
     * 
     * @param type token type, or null for unspecified/opaque
     */
     public void setAccessTokenType(@Nullable @NotEmpty final String type) {
         accessTokenTypeLookupStrategy = FunctionSupport.constant(StringSupport.trimOrNull(type));
     }
    
    /**
     * Set lookup strategy for access token type.
     * 
     * @param strategy lookup strategy
     */
     public void setAccessTokenTypeLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
         accessTokenTypeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
     }

     /**
      * Get access token lifetime.
      * 
      * <p>Defaults to 10 minutes.</p>
      * 
      * @param profileRequestContext profile request context
      * 
      * @return access token lifetime
      */
     @Positive @Nonnull
     public Duration getAccessTokenLifetime(@Nullable final ProfileRequestContext profileRequestContext) {
         final Duration lifetime = accessTokenLifetimeLookupStrategy.apply(profileRequestContext);
         
         Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                 "Access token lifetime must be greater than 0");
         return lifetime;
     }
     
     /**
      * Set the lifetime of an access token.
      * 
      * @param lifetime lifetime of an access token in milliseconds
      */
     public void setAccessTokenLifetime(@Positive @Nonnull final Duration lifetime) {
         Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                 "Access token lifetime must be greater than 0");
         
         accessTokenLifetimeLookupStrategy = FunctionSupport.constant(lifetime);
     }
     
     /**
      * Set a lookup strategy for the access token lifetime.
      *
      * @param strategy lookup strategy
      */
     public void setAccessTokenLifetimeLookupStrategy(
             @Nullable final Function<ProfileRequestContext,Duration> strategy) {
         accessTokenLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
     }

}