/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.oauth2.config;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.profile.config.OverriddenIssuerProfileConfiguration;
import net.shibboleth.oidc.jwt.claims.ClaimsValidator;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;

/**
 * Base class for profile configurations that validate previously issued OAuth2 tokens.
 * 
 * @since 2.0.0
 */
public class AbstractOAuth2TokenValidatingConfiguration extends AbstractOAuth2ClientAuthenticableProfileConfiguration
        implements OverriddenIssuerProfileConfiguration {

    /** Lookup function to override issuer value. */
    @Nonnull private Function<ProfileRequestContext,String> issuerLookupStrategy;
    
    /** Validation of JWT claims for issued tokens. */
    @Nonnull private Function<ProfileRequestContext,ClaimsValidator> issuedClaimsValidatorLookupStrategy;

    /**
     * Creates a new configuration instance.
     *
     * @param profileId Unique profile identifier.
     */
    public AbstractOAuth2TokenValidatingConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);
        
        issuerLookupStrategy = FunctionSupport.constant(null);
        issuedClaimsValidatorLookupStrategy = FunctionSupport.constant(null);
    }

    /** {@inheritDoc} */
    @Override
    @Nullable @NotEmpty public String getIssuer(@Nullable final ProfileRequestContext profileRequestContext) {
        return issuerLookupStrategy.apply(profileRequestContext);
    }
    
    /**
     * Set overridden issuer value.
     * 
     * @param issuer issuer value
     */
    public void setIssuer(@Nullable @NotEmpty final String issuer) {
        issuerLookupStrategy = FunctionSupport.constant(issuer);
    }
    
    /**
     * Sets lookup strategy for overridden issuer value.
     * 
     * @param strategy lookup strategy
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        issuerLookupStrategy = Constraint.isNotNull(strategy, "Issuer lookup strategy cannot be null");
    }
    
    /**
     * Get the {@link ClaimsValidator} to apply to issued JWT-based tokens being validated by this profile.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return the validator to use
     */
    @Nullable public ClaimsValidator getIssuedClaimsValidator(
            @Nullable final ProfileRequestContext profileRequestContext) {
        return issuedClaimsValidatorLookupStrategy.apply(profileRequestContext);
    }
    
    /**
     * Set the {@link ClaimsValidator} to apply to issued JWT-based tokens being validated by this profile.
     * 
     * @param validator validator to use
     */
    public void setIssuedClaimsValidator(@Nullable final ClaimsValidator validator) {
        issuedClaimsValidatorLookupStrategy = FunctionSupport.constant(validator);
    }
    

    /**
     * Set a lookup strategy for the {@link ClaimsValidator} to apply to issued JWT-based tokens being validated by
     * this profile.
     *
     * @param strategy  lookup strategy
     */
    public void setIssuedClaimsValidatorLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,ClaimsValidator> strategy) {
        issuedClaimsValidatorLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

}