/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.oauth2.config;

import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.google.common.base.Predicates;

import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Base class for OIDC protocol configuration, containing configuration bits shared by all flow aware OIDC protocol
 * configurations.
 */
public abstract class AbstractOAuth2FlowAwareProfileConfiguration
        extends AbstractOAuth2ClientAuthenticableProfileConfiguration {

    /** Predicate used to indicate whether authorization code flow is supported by this profile. Default true. */
    @Nonnull private Predicate<ProfileRequestContext> authorizationCodeFlowPredicate;

    /** Predicate used to indicate whether implicit flow is supported by this profile. Default true. */
    @Nonnull private Predicate<ProfileRequestContext> implicitFlowPredicate;

    /** Predicate used to indicate whether hybrid flow is supported by this profile. Default true. */
    @Nonnull private Predicate<ProfileRequestContext> hybridFlowPredicate;

    /** Predicate used to indicate whether refresh tokens are supported by this profile. Default true. */
    @Nonnull private Predicate<ProfileRequestContext> refreshTokensPredicate;

    /**
     * Constructor.
     *
     * @param profileId Unique profile identifier.
     */
    protected AbstractOAuth2FlowAwareProfileConfiguration(final String profileId) {
        super(profileId);
        authorizationCodeFlowPredicate = Predicates.alwaysTrue();
        implicitFlowPredicate = Predicates.alwaysTrue();
        hybridFlowPredicate = Predicates.alwaysTrue();
        refreshTokensPredicate = Predicates.alwaysTrue();
    }

    /**
     * Get whether authorization code flow is supported by this profile.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return whether authorization code flow is supported by this profile
     */
    public boolean isAuthorizationCodeFlowEnabled(@Nullable final ProfileRequestContext profileRequestContext) {
        return authorizationCodeFlowPredicate.test(profileRequestContext);
    }

    /**
     * Set whether authorization code flow is supported by this profile.
     * 
     * @param flag flag to set
     */
    public void setAuthorizationCodeFlowEnabled(final boolean flag) {
        authorizationCodeFlowPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set condition used to indicate whether authorization code flow is supported by this profile.
     * 
     * @param condition condition to set
     */
    public void setAuthorizationCodeFlowEnabledPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        authorizationCodeFlowPredicate = Constraint.isNotNull(condition,
                "Condition used to indicate whether authorization code flow is supported cannot be null");
    }

    /**
     * Get whether implicit flow is supported by this profile.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return whether implicit flow is supported by this profile
     */
    public boolean isHybridFlowEnabled(@Nullable final ProfileRequestContext profileRequestContext) {
        return hybridFlowPredicate.test(profileRequestContext);
    }
    
    /**
     * Set whether implicit flow is supported by this profile.
     * 
     * @param flag flag to set
     */
    public void setHybridFlowEnabled(final boolean flag) {
        hybridFlowPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set condition used to indicate whether implicit flow is supported by this profile.
     * 
     * @param condition condition to set
     */
    public void setHybridFlowEnabledPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        hybridFlowPredicate = Constraint.isNotNull(condition,
                "Condition used to indicate whether implicit flow is supported cannot be null");
    }

    /**
     * Get whether hybrid flow is supported by this profile.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return whether hybrid flow is supported by this profile
     */
    public boolean isImplicitFlowEnabled(@Nullable final ProfileRequestContext profileRequestContext) {
        return implicitFlowPredicate.test(profileRequestContext);
    }
    
    /**
     * Set whether hybrid flow is supported by this profile.
     * 
     * @param flag flag to set
     */
    public void setImplicitFlowEnabled(final boolean flag) {
        implicitFlowPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set condition used to indicate whether hybrid flow is supported by this profile.
     * 
     * @param condition condition to set.
     */
    public void setImplicitFlowEnabledPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        implicitFlowPredicate = Constraint.isNotNull(condition,
                "Condition used to indicate whether hybrid flow is supported cannot be null");
    }

    /**
     * Get whether refresh tokens are supported by this profile.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return whether refresh tokens are supported by this profile
     */
    public boolean isRefreshTokensEnabled(@Nullable final ProfileRequestContext profileRequestContext) {
        return refreshTokensPredicate.test(profileRequestContext);
    }

    /**
     * Set whether refresh tokens are supported by this profile.
     * 
     * @param flag flag to set
     */
    public void setRefreshTokensEnabled(final boolean flag) {
        refreshTokensPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set condition used to indicate whether refresh tokens are supported by this profile.
     * 
     * @param condition condition to set
     */
    public void setRefreshTokensEnabledPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        refreshTokensPredicate = Constraint.isNotNull(condition,
                "Condition used to indicate whether refresh tokens are supported cannot be null");
    }

}