/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.core;

import javax.annotation.Nonnull;

import com.nimbusds.oauth2.sdk.ErrorObject;
import com.nimbusds.oauth2.sdk.OAuth2Error;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * Supplemental error objects for any missing values in the Nimbus library.
 */
public final class OidcError {

    /**
     * The resource indicator is invalid.
     */
    @Nonnull @NotEmpty public static final String INVALID_TARGET_CODE = "invalid_target";
    
    /**
     * The resource indicator is invalid.
     */
    public static final ErrorObject INVALID_TARGET =
        new ErrorObject(INVALID_TARGET_CODE, "Improper or disallowed resource indicator", HTTPResponse.SC_FORBIDDEN);

    /**
     * The request is missing a required PKCE code challenge.
     */
    public static final ErrorObject MISSING_PKCE_CODE_CHALLENGE = 
        new ErrorObject(OAuth2Error.INVALID_REQUEST_CODE, "PKCE code challenge required", HTTPResponse.SC_BAD_REQUEST);

    /**
     * The request contains an unsupported PKCE transform algorithm.
     */
    public static final ErrorObject INVALID_PKCE_TRANSFORMATION_METHOD =
            new ErrorObject(OAuth2Error.INVALID_REQUEST_CODE, "PKCE transform algorithm not supported",
                    HTTPResponse.SC_BAD_REQUEST);
    
    /**
     * Constructor.
     */
    private OidcError() {
        // no op
    }

}