/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.core;

import java.net.URI;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.nimbusds.jwt.JWT;
import com.nimbusds.oauth2.sdk.id.ClientID;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/**
 * OpenID Connect Authentication Request. Extends the OAuth 2.0 authorization request.
 * <p> This class is mutable and not thread-safe.</p> 
 */
public class OIDCAuthenticationRequest extends OAuthAuthorizationRequest { 
    
    /** The default OpenID scope. */
    @Nonnull @NotEmpty private static final String DEFAULT_OPENID_SCOPE = "openid";

    /** The request object. Optional. */
    @Nullable private JWT requestObject;
    
    /** The request URI. Optional. */
    @Nullable private URI requestURI;
    
    /**
     * 
     * Constructor.
     *
     * @param id the clientID.
     */
    public OIDCAuthenticationRequest(@Nonnull final ClientID id) {
        super(id);
        // Must contain the openid scope. 
        getScope().add(DEFAULT_OPENID_SCOPE);
    }


    /**
     * Get the request object.
     * 
     * @return Returns the request object.
     */
    @Nullable public JWT getRequestObject() {
        return requestObject;
    }

    /**
     * Set the request object.
     * 
     * @param object The request object to set.
     */
    public void setRequestObject(@Nullable final JWT object) {
        requestObject = object;
    }

    /**
     * Get the URI to fetch the request object from.
     * 
     * @return Returns the requestURI.
     */
    @Nullable public URI getRequestURI() {
        return requestURI;
    }

    /**
     * Set the URI to fetch the request object from.
     * 
     * @param uri The requestURI to set.
     */
    public void setRequestURI(@Nullable final URI uri) {
        requestURI = uri;
    }
    
    //TODO others relating to sections 5.2, 5.5, 6, and 7.2.1
    
    

}
