/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.config.navigate;

import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.profile.context.navigate.AbstractRelyingPartyLookupFunction;
import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.oidc.metadata.policy.MetadataPolicyResolver;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * A lookup function for the map of {@link MetadataPolicy} objects, resolved via configurable
 * {@link MetadataPolicyResolver} and optionally configurable {@link CriteriaSet}. First object returned by the
 * resolver is returned.
 */
public class ResolverBasedRegistrationMetadataPolicyLookupFunction
    extends AbstractRelyingPartyLookupFunction<Map<String,MetadataPolicy>> {
    
    /** Class logger. */
    @Nonnull 
    private final Logger log = LoggerFactory.getLogger(ResolverBasedRegistrationMetadataPolicyLookupFunction.class);

    /** The metadata policy resolver. */
    @Nullable private MetadataPolicyResolver metadataPolicyResolver;

    /** The lookup strategy for the criteria set used for the metadata policy resolver. */
    @Nullable private Function<ProfileRequestContext, CriteriaSet> criteriaSetLookupStrategy;
    
    /**
     * Set the metadata policy resolver.
     * 
     * @param resolver What to set.
     */
    public void setMetadataPolicyResolver(@Nullable final MetadataPolicyResolver resolver) {
        metadataPolicyResolver = resolver;
    }
    
    /**
     * Set the lookup strategy for the criteria set used for the metadata policy resolver.
     * 
     * @param strategy What to set.
     */
    public void setCriteriaSetLookupStrategy(@Nullable final Function<ProfileRequestContext, CriteriaSet> strategy) {
        criteriaSetLookupStrategy = strategy;
    }
    
    /** {@inheritDoc} */
    @Override @Nullable public Map<String, MetadataPolicy> apply(@Nullable final ProfileRequestContext input) {
        if (metadataPolicyResolver == null) {
            log.debug("No metadata policy resolver configured, returning null");
            return null;
        }
        final CriteriaSet criteriaSet =
                criteriaSetLookupStrategy == null ? null : criteriaSetLookupStrategy.apply(input);
        log.debug("Starting to resolve single metadata policy");
        try {
            return metadataPolicyResolver.resolveSingle(criteriaSet == null ? new CriteriaSet() : criteriaSet);
        } catch (final ResolverException e) {
            log.error("Resolver exception catched while resolving metadata policy", e);
        }
        return null;
    }
    
}
