/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.config;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.google.common.base.Predicates;

import net.shibboleth.idp.profile.config.AttributeResolvingProfileConfiguration;
import net.shibboleth.oidc.profile.oauth2.config.AbstractOAuth2TokenValidatingConfiguration;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Profile configuration for the OpenID Connect core protocol userinfo endpoint.
 */
public class OIDCUserInfoConfiguration extends AbstractOAuth2TokenValidatingConfiguration
        implements AttributeResolvingProfileConfiguration {

    /** OIDC base protocol URI. */
    @Nonnull @NotEmpty public static final String PROTOCOL_URI = "http://openid.net/specs/openid-connect-core-1_0.html";

    /** ID for this profile configuration. */
    @Nonnull @NotEmpty public static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/oidc/userinfo";

    /** Whether attributes should be resolved in the course of the profile. */
    @Nonnull private Predicate<ProfileRequestContext> resolveAttributesPredicate;

    /** Whether encryption is optional in the face of no key, etc. */
    @Nonnull private Predicate<ProfileRequestContext> encryptionOptionalPredicate;

    /** Lookup function to supply attribute IDs to omit from UserInfo token. */
    @Nonnull private Function<ProfileRequestContext,Set<String>> deniedUserInfoAttributesLookupStrategy;
    
    /**
     * Constructor.
     */
    public OIDCUserInfoConfiguration() {
        this(PROFILE_ID);
    }

    /**
     * Creates a new configuration instance.
     *
     * @param profileId Unique profile identifier.
     */
    public OIDCUserInfoConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);
        
        resolveAttributesPredicate = Predicates.alwaysTrue();
        encryptionOptionalPredicate = Predicates.alwaysTrue();
        deniedUserInfoAttributesLookupStrategy = FunctionSupport.constant(null);
    }

    /** {@inheritDoc} */
    @Override
    public boolean isResolveAttributes(@Nullable final ProfileRequestContext profileRequestContext) {
        return resolveAttributesPredicate.test(profileRequestContext);
    }

    /**
     * Set whether attributes should be resolved during the profile.
     * 
     * @param flag flag to set
     */
    public void setResolveAttributes(final boolean flag) {
        resolveAttributesPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }
    
    /**
     * Set a condition to determine whether attributes should be resolved during the profile.
     * 
     * @param condition condition to set
     */
    public void setResolveAttributesPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        resolveAttributesPredicate = Constraint.isNotNull(condition, "Resolve attributes predicate cannot be null");
    }
    
    /**
     * Get whether encryption is optional in the face of a missing key, etc.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return true iff encryption is optional
     */
    public boolean isEncryptionOptional(@Nullable final ProfileRequestContext profileRequestContext) {
        return encryptionOptionalPredicate.test(profileRequestContext);
    }
    
    /**
     * Set whether encryption is optional in the face of a missing key, etc.
     * 
     * @param flag  flag to set
     */
    public void setEncryptionOptional(final boolean flag) {
        encryptionOptionalPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set a condition to determine whether encryption is optional in the face of a missing key, etc.
     *
     * @param condition condition to set
     */
    public void setEncryptionOptionalPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        encryptionOptionalPredicate = Constraint.isNotNull(condition, "Encryption optional predicate cannot be null");
    }
    
    /**
     * Get the set of attribute IDs which should be omitted from the UserInfo token.
     * 
     * <p>Default behavior is to include all claims, but omiited claims also affect the set that
     * may need to be embedded for recovery into the access/refresh tokens.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the attribute IDs to omit from UserInfo token
     */
    @Nonnull @NonnullElements @NotLive public Set<String> getDeniedUserInfoAttributes(
            @Nullable final ProfileRequestContext profileRequestContext) {
        
        final Set<String> attributes = deniedUserInfoAttributesLookupStrategy.apply(profileRequestContext);
        if (attributes != null) {
            return Set.copyOf(attributes);
        }
        return Collections.emptySet();
    }

    /**
     * Set the set of attribute IDs which should be omitted from the UserInfo token.
     * 
     * <p>Default behavior is to include all claims, but omiited claims also affect the set that
     * may need to be embedded for recovery into the access/refresh tokens.</p>
     * 
     * @param attributes the attribute IDs to omit from UserInfo token
     */
    public void setDeniedUserInfoAttributes(@Nullable @NonnullElements final Collection<String> attributes) {

        if (attributes == null || attributes.isEmpty()) {
            deniedUserInfoAttributesLookupStrategy = FunctionSupport.constant(null);
        } else {
            deniedUserInfoAttributesLookupStrategy = FunctionSupport.constant(
                    Set.copyOf(StringSupport.normalizeStringCollection(attributes)));
        }
    }

    /**
     * Set a lookup strategy for the set of attribute IDs which should be omitted from the UserInfo token.
     *
     * @param strategy  lookup strategy
     */
    public void setDeniedUserInfoAttributesLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Set<String>> strategy) {
        deniedUserInfoAttributesLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

}