/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.config;

import java.time.Duration;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.profile.config.OverriddenIssuerProfileConfiguration;
import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.oidc.profile.oauth2.config.AbstractOAuth2FlowAwareProfileConfiguration;
import net.shibboleth.utilities.java.support.annotation.constraint.NonNegative;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;

/**
 * Profile configuration for the OpenID Connect dynamic client registration.
 */
public class OIDCDynamicRegistrationConfiguration extends AbstractOAuth2FlowAwareProfileConfiguration
        implements OverriddenIssuerProfileConfiguration {

    /** OIDC base protocol URI. */
    public static final String PROTOCOL_URI = "https://openid.net/specs/openid-connect-registration-1_0.html";

    /** ID for this profile configuration. */
    public static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/oidc/registration";

    /** Lookup function to override issuer value. */
    @Nonnull private Function<ProfileRequestContext,String> issuerLookupStrategy;

    /** Lookup function to supply registration validity period. */
    @Nonnull private Function<ProfileRequestContext,Duration> registrationValidityPeriodLookupStrategy;

    /** Lookup function to supply client secret expiration period. */
    @Nullable private Function<ProfileRequestContext,Duration> secretExpirationPeriodLookupStrategy;

    /** Lookup function to the default metadata policy. */
    @Nullable private Function<ProfileRequestContext,Map<String,MetadataPolicy>> metadataPolicyLookupStrategy;

    /**
     * Constructor.
     */
    public OIDCDynamicRegistrationConfiguration() {
        this(PROFILE_ID);
    }

    /**
     * Creates a new configuration instance.
     *
     * @param profileId Unique profile identifier.
     */
    public OIDCDynamicRegistrationConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);
        issuerLookupStrategy = FunctionSupport.constant(null);
        setRegistrationValidityPeriod(Duration.ofHours(24));
        setSecretExpirationPeriod(Duration.ofDays(365));
        metadataPolicyLookupStrategy = FunctionSupport.constant(null);
    }

    /** {@inheritDoc} */
    @Override
    @Nullable @NotEmpty public String getIssuer(@Nullable final ProfileRequestContext profileRequestContext) {
        return issuerLookupStrategy.apply(profileRequestContext);
    }
    
    /**
     * Set overridden issuer value.
     * 
     * @param issuer issuer value
     */
    public void setIssuer(@Nullable @NotEmpty final String issuer) {
        issuerLookupStrategy = FunctionSupport.constant(issuer);
    }
    
    /**
     * Sets lookup strategy for overridden issuer value.
     * 
     * @param strategy lookup strategy
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        issuerLookupStrategy = Constraint.isNotNull(strategy, "Issuer lookup strategy cannot be null");
    }
    
    /**
     * Get dynamic registration validity period.
     * 
     * <p>A 0 is interpreted as an unlimited period.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return dynamic registration validity period
     */
    @Nullable @NonNegative
    public Duration getRegistrationValidityPeriod(@Nullable final ProfileRequestContext profileRequestContext) {
        
        final Duration period = registrationValidityPeriodLookupStrategy.apply(profileRequestContext);
        
        Constraint.isFalse(period != null && period.isNegative(), "Validity period cannot be negative");
        return period;
    }

    /**
     * Sets the registration validity period.
     * 
     * <p>A 0 is interpreted as an unlimited period.</p>
     * 
     * @param period registration validity period
     */
    public void setRegistrationValidityPeriod(@Nullable @NonNegative final Duration period) {
        Constraint.isFalse(period != null && period.isNegative(), "Validity period cannot be negative");

        registrationValidityPeriodLookupStrategy = FunctionSupport.constant(period);
    }

    /**
     * Set a lookup strategy for the registration validity period.
     * 
     * <p>A null or 0 is interpreted as an unlimited period.</p>
     * 
     * @param strategy lookup strategy
     */
    public void setRegistrationValidityPeriodLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Duration> strategy) {
        registrationValidityPeriodLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get client secret expiration period.
     * 
     * <p>A 0 is interpreted as an unlimited period.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return client secret expiration period
     */
    @Nullable @NonNegative
    public Duration getSecretExpirationPeriod(@Nullable final ProfileRequestContext profileRequestContext) {
        final Duration period = secretExpirationPeriodLookupStrategy.apply(profileRequestContext);
        
        Constraint.isFalse(period != null && period.isNegative(), "Validity period cannot be negative");
        return period;
    }

    /**
     * Sets the client secret expiration period.
     * 
     * <p>A 0 is interpreted as an unlimited period.</p>
     * 
     * @param period registration validity period
     */
    public void setSecretExpirationPeriod(@Nullable @NonNegative final Duration period) {
        Constraint.isFalse(period != null && period.isNegative(), "Validity period cannot be negative");
        
        secretExpirationPeriodLookupStrategy = FunctionSupport.constant(period);
    }

    /**
     * Set a lookup strategy for the client secret expiration period.
     * 
     * <p>A null or 0 is interpreted as an unlimited period.</p>
     * 
     * @param strategy lookup strategy
     */
    public void setSecretExpirationPeriodLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Duration> strategy) {
        secretExpirationPeriodLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get the metadata policy.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return metadata policy
     */
    @Nullable
    public Map<String,MetadataPolicy> getMetadataPolicy(@Nullable final ProfileRequestContext profileRequestContext) {
        return metadataPolicyLookupStrategy.apply(profileRequestContext);
    }

    /**
     * Sets the metadata policy.
     * 
     * @param policy metadata policy
     */
    public void setMetadataPolicy(@Nonnull final Map<String,MetadataPolicy> policy) {
        metadataPolicyLookupStrategy = FunctionSupport.constant(policy);
    }

    /**
     * Set a lookup strategy for the metadata policy.
     * 
     * @param strategy lookup strategy
     */
    public void setMetadataPolicyLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Map<String,MetadataPolicy>> strategy) {
        metadataPolicyLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

}