/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.spring.relyingparty.metadata.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.beans.factory.BeanCreationException;
import org.springframework.context.ApplicationContext;

import net.shibboleth.oidc.metadata.ProviderMetadataProviderContainer;
import net.shibboleth.oidc.metadata.ProviderMetadataResolver;
import net.shibboleth.oidc.metadata.impl.ChainingProviderMetadataResolver;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.resolver.ResolverException;
import net.shibboleth.utilities.java.support.service.ServiceException;
import net.shibboleth.utilities.java.support.service.ServiceableComponent;

/**
 * Strategy for summoning up a {@link ProviderMetadataResolver} from a populated {@link ApplicationContext}.
 * 
 * <p>
 * The logic is the same as in 
 * net.shibboleth.idp.profile.spring.relyingparty.metadata.impl.MetadataResolverServiceStrategy.
 * </p>
 */
public class ProviderMetadataResolverServiceStrategy extends AbstractIdentifiableInitializableComponent
        implements Function<ApplicationContext, ServiceableComponent<ProviderMetadataResolver>> {

    /** {@inheritDoc} */
    @Nullable
    public ServiceableComponent<ProviderMetadataResolver> apply(@Nonnull final ApplicationContext appContext) {
        final Collection<ProviderMetadataProviderContainer> containers =
                appContext.getBeansOfType(ProviderMetadataProviderContainer.class).values();

        if (containers.isEmpty()) {
            throw new ServiceException(
                    "Reload did not produce any bean of type " + ProviderMetadataProviderContainer.class.getName());
        }
        if (1 == containers.size()) {
            // done
            return containers.iterator().next();
        }
        
        // initialize so we can sort
        for (final ProviderMetadataProviderContainer container : containers) {
            try {
                container.initialize();
            } catch (final ComponentInitializationException e) {
                throw new BeanCreationException("could not preinitialize , provider metadata provider " 
                        + container.getId(), e);
            }
        }
        // wrap many containers into a single top level container with a chain of all the individual chains/resolvers.
        // The result of the strategy needs a single serviceable component.
        
        final List<ProviderMetadataProviderContainer> containerList = new ArrayList<>(containers.size());
        containerList.addAll(containers);
        Collections.sort(containerList);
        final ChainingProviderMetadataResolver chain = new ChainingProviderMetadataResolver();
        try {
            chain.setResolvers(containerList.stream().
                    map(ProviderMetadataProviderContainer::getEmbeddedResolver).
                    collect(Collectors.toList()));
            chain.setId("MultiFileResolverFor:" + containers.size() + ":Resources");
            chain.initialize();
            final ProviderMetadataProviderContainer result = new ProviderMetadataProviderContainer();
            result.setEmbeddedResolver(chain);
            result.setApplicationContext(appContext);
            result.initialize();
            return result;
        } catch (final ResolverException | ComponentInitializationException e) {
            throw new ServiceException("Chaining constructor create failed", e);
        }
        
    }

}