/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.policy.impl;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.ResponseHandler;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/** The response handler for parsing the metadata policy document into a map. */
public class DefaultMetadataPolicyResponseHandler extends AbstractIdentifiableInitializableComponent
        implements ResponseHandler<Map<String, MetadataPolicy>> {        
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultMetadataPolicyResponseHandler.class);
    
    /** The parsing strategy used for parsing metadata policies. */
    @NonnullAfterInit private Function<byte[], List<Map<String, MetadataPolicy>>> parsingStrategy;
    
    /**
     * Set the parsing strategy used for parsing metadata policies.
     * 
     * @param strategy What to set.
     */
    public void setParsingStrategy(@Nonnull final Function<byte[], List<Map<String, MetadataPolicy>>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        parsingStrategy = Constraint.isNotNull(strategy, "Parsing strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        if (parsingStrategy == null) {
            throw new ComponentInitializationException("Parsing strategy cannot be null");
        }
    }
    
    /** {@inheritDoc} */
    @Override
    @Nullable public Map<String, MetadataPolicy> handleResponse(final HttpResponse response) throws IOException {
        
        final int httpStatusCode = response.getStatusLine().getStatusCode();
        
        final String currentRequestURI = 
                MDC.get(MetadataPolicyViaLocationFetchingStrategy.MDC_ATTRIB_CURRENT_REQUEST_URI);
        
        if (httpStatusCode != HttpStatus.SC_OK) {
            log.warn("Non-ok status code '{}' returned from remote metadata source: {}",
                    httpStatusCode, currentRequestURI);
            return null;
        }
        
        final List<Map<String, MetadataPolicy>> parsedResponse;
        
        try {
            // this should convert the entity with the character set from the entity.
            parsedResponse = parsingStrategy.apply(EntityUtils.toByteArray(response.getEntity()));
            
        } catch (final Exception e) {
            // catch any of the many exceptions
            log.error("Error parsing HTTP response stream", e);
            return null;
        }
        
        if (parsedResponse == null || parsedResponse.isEmpty()) {
            log.error("No parsed values found from the response.");
            return null;
        }
        
        if (parsedResponse.size() > 1) {
            log.warn("More than one metadata policies found, returning first of the list.");            
        }
        return parsedResponse.get(0);
        
    }
    
}
