/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.policy.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.BiFunction;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.utilities.java.support.collection.Pair;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;

/**
 * <p>A function that applies the given {@link MetadataPolicy} to the given object. The input is given as a {@link
 * Pair} of the object and the policy. The policy is applied to the incoming object in the following way, as specified
 * in the OIDC federation federation specification 1.0 (draft 17 / September 2021):</p>
 * 
 * <ul>
 * <li>If there is a value operator in the policy, apply that and you are done.</li>
 * <li>Add whatever value is specified in an add operator.</li>
 * <li>If the parameter still has no value apply the default if there is one.</li>
 * <li>Do the essential check. If essential is missing as an operator essential is to be treated as if set to false.
 * If essential is defined to be true, then the claim MUST have a value by now. Otherwise applying the operator MUST
 * fail.</li>
 * <li>Do the other checks. Verified that the value is one_of or that the values are subset_of/superset_of. If the
 * parameter values do not fall within the allowed boundaries, applying the operator MUST fail.</li>
 * </ul>
 * 
 * <p>In addition to the checks above, we also support regular expression validation.</p>
 * 
 * <p>The function returns a {@link Pair} of the object for which the value modifiers of the metadata policy have
 * been applied to, and a flag indicating if the object was compatible with the value checks of the metadata policy.
 * </p>
 */
public class DefaultMetadataPolicyEnforcer implements BiFunction<Object,MetadataPolicy,Pair<Object,Boolean>> {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultMetadataPolicyEnforcer.class);

    /** {@inheritDoc} */
    @Override
    @Nonnull public Pair<Object, Boolean> apply(@Nullable final Object candidate, 
            @Nullable final MetadataPolicy policy) {
        if (policy == null) {
            return new Pair<>(candidate, Boolean.TRUE);
        }
        final Object value = policy.getValue();
        if (value != null) {
            return new Pair<>(value, Boolean.TRUE);
        }
        final Object result;
        final Object add = policy.getAdd();
        
        if (add != null) {
            try {
                result = applyAddOperator(candidate, add);
            } catch (final ConstraintViolationException e) {
                log.warn("Could not add values to candidate: {}", e.getMessage());
                return new Pair<>(candidate, Boolean.FALSE);
            }
        } else {
            result = candidate == null ? policy.getDefaultValue() : candidate;
        }
        
        final boolean validation = doValueChecks(result, policy);
        return new Pair<>(result, Boolean.valueOf(validation));
    }

    /**
     * Applies the given add value modifier for the given candidate and returns the result of the operation.
     * 
     * @param candidate The candidate for which the add operation is applied.
     * @param add The value(s) to be added to the claim.
     * @return An object containing the candidate for which the add operation has been applied.
     * @throws ConstraintViolationException If the add operator is not compliant with the given candidate.
     */
    @Nonnull protected Object applyAddOperator(@Nullable final Object candidate, @Nonnull final Object add)
            throws ConstraintViolationException {
        if (candidate != null) {
            if (candidate instanceof List) {
                final List<Object> list = new ArrayList<>((List<?>)candidate);
                if (add instanceof Collection) {
                    list.addAll((Collection<?>) add);
                } else {
                    list.add(add);
                }
                return list;
            } else {
                if (add instanceof Collection) {
                    final Collection<?> collection = (Collection<?>) add;
                    if (collection.size() > 1 || !collection.contains(candidate)) {
                        throw new ConstraintViolationException("The array-values in add (" + add +
                                ") not compliant with the single existing value " + candidate);
                    }
                }
                if (!add.equals(candidate)) {
                    throw new ConstraintViolationException("The single-value in add (" + add + 
                            ") does not match with the single existing value " + candidate);
                }
                return candidate;
            }
        }
        if (add instanceof Collection) {
            return List.copyOf((Collection<?>) add);
        }
        return add;
    }
   
 // Checkstyle: MethodLength|CyclomaticComplexity OFF
    /**
     * Runs the value check operators for the candidate.
     * 
     * @param candidate The candidate to be verified.
     * @param policy The metadata policy whose value check operators are used.
     * @return true if the candidate is compliant with the metadata policy, false otherwise.
     */
    protected boolean doValueChecks(@Nullable final Object candidate, @Nonnull final MetadataPolicy policy) {
        boolean validation = true;
        
        if (candidate != null) {
            final List<Object> oneOfValues = policy.getOneOfValues();
            if (oneOfValues != null) {
                if (candidate instanceof List) {
                    final List<?> list = (List<?>) candidate;
                    if (list.size() > 1) {
                        log.warn("The candidate {} contains multiple values, not compatible with one_of", candidate);
                        validation = false;
                    } else {
                        if (oneOfValues != null && !MetadataPolicyHelper.isSubsetOfValues(candidate, oneOfValues)) {
                            log.warn("The candidate {} is not compatible with one_of {}", candidate, oneOfValues);
                            validation = false;
                        }
                    }
                } else {
                    if (!oneOfValues.contains(candidate)) {
                        log.warn("The candidate {} does not contain a value required by one_of {}", candidate,
                                oneOfValues);
                        validation = false;
                    }
                }
            }
            final List<Object> subsetOfValues = policy.getSubsetOfValues();
            if (subsetOfValues != null && !MetadataPolicyHelper.isSubsetOfValues(candidate, subsetOfValues)) {
                log.warn("The candidate {} is not a subset as required by subset_of {}", candidate, subsetOfValues);
                validation = false;
            }
            final List<Object> supersetOfValues = policy.getSupersetOfValues();
            if (supersetOfValues != null && !MetadataPolicyHelper.isSupersetOfValues(candidate, supersetOfValues)) {
                log.warn("The candidate {} is not a superset as required by superset_of {}", candidate,
                        supersetOfValues);
                validation = false;
            }
            if (!verifyRegexp(candidate, policy.getRegexp())) {
                validation = false;
            }
        } else {
            if (policy.isEssential()) {
                log.warn("No value even though essential is set to true");
                validation = false;
            }
        }
        return validation;
    }
 // Checkstyle: MethodLength|CyclomaticComplexity ON
    /**
     * Verifies that the given candidate meets the regular expression.
     * 
     * @param candidate The candidate to be verified.
     * @param regexp The regular expression.
     * @return true if the candidate is compliant with regex, false otherwise.
     */
    protected boolean verifyRegexp(@Nonnull final Object candidate, @Nullable final String regexp) {
        if (regexp != null) {
            if (candidate instanceof List) {
                for (final Object item : (List<?>) candidate) {
                    if (!Pattern.matches(regexp, item.toString())) {
                        log.warn("One of candidate values {} does not match the regex {}", item, regexp);
                        return false;
                    }
                }
            } else {
                if (!Pattern.matches(regexp, candidate.toString())) {
                    log.warn("The candidate value {} does not match the regex {}", candidate, regexp);
                    return false;
                }
            }
        }
        return true;
    }
}
