/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.keyinfo.ext.impl.provider;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.security.SecurityException;
import org.opensaml.security.credential.Credential;
import org.opensaml.security.credential.CredentialContext;
import org.opensaml.xmlsec.keyinfo.KeyInfoCredentialResolver;
import org.opensaml.xmlsec.keyinfo.impl.KeyInfoResolutionContext;
import org.opensaml.xmlsec.keyinfo.impl.provider.AbstractKeyInfoProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.jwk.AsymmetricJWK;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.jwk.KeyType;

import net.shibboleth.oidc.saml.xmlobject.JwksData;
import net.shibboleth.oidc.security.credential.BasicJWKCredential;
import net.shibboleth.oidc.security.credential.JWKCredential;
import net.shibboleth.oidc.security.impl.CredentialConversionUtil;
import net.shibboleth.utilities.java.support.codec.Base64Support;
import net.shibboleth.utilities.java.support.codec.DecodingException;
import net.shibboleth.utilities.java.support.collection.LazySet;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * Implementation of {@link org.opensaml.xmlsec.keyinfo.impl.KeyInfoProvider} which provides basic support for
 * extracting a {@link JWKCredential} child of KeyInfo. The value is expected to be either a Base64-encoded value of
 * either a single JWK credential or a JSON array of JWK credentials.
 */
public class InlineJwksProvider extends AbstractKeyInfoProvider {

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(InlineJwksProvider.class);

    /** {@inheritDoc} */
    @Override @Nullable
    public Collection<Credential> process(@Nonnull final KeyInfoCredentialResolver resolver,
            @Nonnull final XMLObject keyInfoChild, @Nullable final CriteriaSet criteriaSet,
            @Nonnull final KeyInfoResolutionContext kiContext) throws SecurityException {
        final JwksData jwksData = getJwksData(keyInfoChild);
        if (jwksData == null || StringSupport.trimOrNull(jwksData.getValue()) == null) {
            return null;
        }
        final String data;
        try {
            data = new String(Base64Support.decode(jwksData.getValue()));
        } catch (final DecodingException e) {
            log.error("Could not decode the JWKS data", e);
            return null;
        }
        final List<JWK> jwks = parseJWKs(data);
        if (jwks == null) {
            return null;
        }
        final LazySet<Credential> credentials = new LazySet<>();
        for (final JWK jwk : jwks) {
            credentials.add(buildCredential(jwk, kiContext));
        }
        return credentials;
    }
    
    @Nonnull protected BasicJWKCredential buildCredential(@Nonnull final JWK jwk,
            @Nonnull final KeyInfoResolutionContext kiContext) {
        final BasicJWKCredential credential = new BasicJWKCredential();
        if (jwk.getKeyType() == KeyType.EC || jwk.getKeyType() == KeyType.RSA) {
            try {
                credential.setPublicKey(((AsymmetricJWK) jwk).toPublicKey());
            } catch (final JOSEException e) {
                log.warn("Could not parse public key from JWK", e);
            }
        } else {
            log.warn("Unsupported key type {} found from JWK", jwk.getKeyType());
        }
        if (jwk.getKeyID() != null) {
            credential.getKeyNames().add(jwk.getKeyID());
            credential.setKid(jwk.getKeyID());
        } else {
            credential.getKeyNames().addAll(kiContext.getKeyNames());
        }

        if (jwk.getKeyUse() != null) {
            credential.setUsageType(CredentialConversionUtil.getUsageType(jwk));
        } else {
            final CredentialContext credContext = buildCredentialContext(kiContext);
            if (credContext != null) {
                credential.getCredentialContextSet().add(credContext);
            }
        }
        return credential;
    }
    
    @Nullable protected List<JWK> parseJWKs(@Nullable final String input) {
        final JWKSet jwkSet = parseJWKSet(input);
        if (jwkSet != null) {
            return jwkSet.getKeys();
        }
        final JWK jwk = parseJWK(input);
        if (jwk != null) {
            return Arrays.asList(jwk);
        }
        log.warn("Could not parse neither JWKSet nor JWK from the given input {}", input);
        return null;
    }
    
    @Nullable protected JWKSet parseJWKSet(@Nullable final String input) {
        try {
            return JWKSet.parse(input);
        } catch (final ParseException e) {
            if (log.isTraceEnabled()) {
                log.trace("Could not parse JWKSet from the given input", e);
            } else {
                log.debug("Could not parse JWKSet from the given input: {}", e.getMessage());
            }
        }
        return null;
    }
    
    @Nullable protected JWK parseJWK(@Nullable final String input) {
        try {
            return JWK.parse(input);
        } catch (final ParseException e) {
            if (log.isTraceEnabled()) {
                log.trace("Could not parse JWK from the given input", e);
            } else {
                log.debug("Could not parse JWK from the given input: {}", e.getMessage());
            }
        }
        return null;
    }

    @Override
    public boolean handles(@Nonnull final XMLObject keyInfoChild) {
        return getJwksData(keyInfoChild) != null;
    }

    /**
     * Get the JwksData from the passed XML object.
     * 
     * @param xmlObject an XML object, presumably a {@link JwksData}
     * @return the JwksData which was found, or null if none
     */
    @Nullable protected JwksData getJwksData(@Nonnull final XMLObject xmlObject) {

        if (xmlObject instanceof JwksData) {
            return (JwksData) xmlObject;
        } else {
            return null;
        }
    }
}
