/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.keyinfo.ext.impl.provider;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.security.credential.Credential;
import org.opensaml.xmlsec.keyinfo.KeyInfoCredentialResolver;
import org.opensaml.xmlsec.keyinfo.impl.KeyInfoResolutionContext;
import org.opensaml.xmlsec.keyinfo.impl.provider.AbstractKeyInfoProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.auth.Secret;

import net.shibboleth.oidc.metadata.ClientSecretValueResolver;
import net.shibboleth.oidc.metadata.criterion.ClientSecretReferenceCriterion;
import net.shibboleth.oidc.saml.xmlobject.ClientSecretKeyReference;
import net.shibboleth.oidc.security.credential.BasicNimbusSecretCredential;
import net.shibboleth.oidc.security.credential.NimbusSecretCredential;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.collection.LazySet;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * Implementation of {@link org.opensaml.xmlsec.keyinfo.impl.KeyInfoProvider} which provides basic support for
 * extracting a {@link NimbusSecretCredential} child of KeyInfo. The values are resolved using the given
 * collection of {@link ClientSecretValueResolver}s.
 */
public class ClientSecretReferenceProvider  extends AbstractKeyInfoProvider {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ClientSecretReferenceProvider.class);
    
    /** The list of resolvers for resolving the values for client secrets. */
    @Nonnull @NotEmpty private Collection<ClientSecretValueResolver> clientSecretValueResolvers;

    /**
     * Constructor.
     *
     * @param valueResolvers The list of resolvers for resolving the values for client secrets.
     */
    public ClientSecretReferenceProvider(
            @Nullable @NotEmpty final Collection<ClientSecretValueResolver> valueResolvers) {
        if (valueResolvers != null) {
            clientSecretValueResolvers = List.copyOf(valueResolvers);
        } else {
            clientSecretValueResolvers = Collections.emptyList();
        }
    }
    
    /** {@inheritDoc} */
    @Override @Nullable
    public Collection<Credential> process(@Nonnull final KeyInfoCredentialResolver resolver,
            @Nonnull final XMLObject keyInfoChild, @Nullable final CriteriaSet criteriaSet,
            @Nonnull final KeyInfoResolutionContext kiContext) throws SecurityException {
        final ClientSecretKeyReference clientSecretReference = getClientSecret(keyInfoChild);
        if (clientSecretReference == null || StringSupport.trimOrNull(clientSecretReference.getValue()) == null) {
            log.debug("No value found for the ClientSecretReferenceKey element");
            return null;
        }
        final String secretReference = clientSecretReference.getValue();
        for (final ClientSecretValueResolver valueResolver : clientSecretValueResolvers) {
            log.trace("Attempting to resolve value for {} with {}", secretReference, valueResolver.getId());
            final CriteriaSet criteria =  new CriteriaSet(new ClientSecretReferenceCriterion(secretReference));
            try {
                final String value = valueResolver.resolveSingle(criteria);
                if (value != null) {
                    log.debug("Found value for {} with {}", secretReference, valueResolver.getId());
                    final LazySet<Credential> credentials = new LazySet<>();
                    credentials.add(new BasicNimbusSecretCredential(new Secret(value)));
                    return credentials;
                }
            } catch (final ResolverException e) {
                log.warn("Client secret value resolution failed", e);
            }
        }
        log.warn("No values could be resolved for the client secret reference {}", clientSecretReference.getValue());
        return null;
    }

    /** {@inheritDoc} */
    @Override
    public boolean handles(final @Nonnull XMLObject keyInfoChild) {
        return getClientSecret(keyInfoChild) != null;
    }

    /**
     * Get the ClientSecretReferenceKey from the passed XML object.
     * 
     * @param xmlObject an XML object, presumably a {@link ClientSecretKeyReference}
     * @return the ClientSecret which was found, or null if none
     */
    @Nullable protected ClientSecretKeyReference getClientSecret(@Nonnull final XMLObject xmlObject) {

        if (xmlObject instanceof ClientSecretKeyReference) {
            return (ClientSecretKeyReference) xmlObject;
        } else {
            return null;
        }
    }
}