/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.storage.StorageRecord;
import org.opensaml.storage.StorageService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.util.JSONObjectUtils;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;

import net.shibboleth.oidc.metadata.ClientInformationResolver;
import net.shibboleth.oidc.metadata.criterion.ClientIDCriterion;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * A {@link ClientInformationResolver} exploiting {@link StorageService} for fetching the stored data.
 */
public class StorageServiceClientInformationResolver extends BaseStorageServiceClientInformationComponent
        implements ClientInformationResolver {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(StorageServiceClientInformationResolver.class);

    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public Iterable<OIDCClientInformation> resolve(@Nullable final CriteriaSet criteria)
            throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        final ClientIDCriterion clientIdCriterion = criteria.get(ClientIDCriterion.class);
        if (clientIdCriterion == null || clientIdCriterion.getClientID() == null) {
            log.info("No client ID criteria found, returning empty set.");
            return Collections.emptyList();
        }
        // TODO: support other criterion
        final String clientId = clientIdCriterion.getClientID().getValue();
        final List<OIDCClientInformation> result = new ArrayList<>();
        try {
            final StorageRecord<?> record = getStorageService().read(CONTEXT_NAME, clientId);
            if (record == null) {
                log.debug("Could not find any records with clientId {}", clientId);
            } else {
                final OIDCClientInformation clientInformation =
                        OIDCClientInformation.parse(JSONObjectUtils.parse(record.getValue()));
                log.debug("Found a record with clientId {}", clientId);
                result.add(clientInformation);
            }
        } catch (final IOException | ParseException e) {
            log.error("Could not read the storage data", e);
        }
        return result;
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public OIDCClientInformation resolveSingle(@Nullable final CriteriaSet criteria)
            throws ResolverException {
        final Iterable<OIDCClientInformation> iterable = resolve(criteria);
        if (iterable != null) {
            final Iterator<OIDCClientInformation> iterator = iterable.iterator();
            if (iterator != null && iterator.hasNext()) {
                return iterator.next();
            }
        }
        log.warn("Could not find any clients with the given criteria");
        return null;
    }
}
