/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.criterion.EntityIdCriterion;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.IdPAttributeValue;
import net.shibboleth.idp.attribute.StringAttributeValue;
import net.shibboleth.idp.attribute.resolver.AttributeResolver;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolutionContext;
import net.shibboleth.oidc.metadata.criterion.ClientSecretReferenceCriterion;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.collection.LazySet;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;
import net.shibboleth.utilities.java.support.service.ReloadableService;

/**
 * A client secret value resolver that fetches the values from the given {@link AttributeResolver} service.
 * 
 * This class builds a new {@link AttributeResolutionContext} and sets the client secret reference key value to
 * {@link AttributeResolutionContext#setPrincipal(String)} and its related entity ID to
 * {@link AttributeResolutionContext#setAttributeRecipientID(String)}. The resolution context does not have any
 * parent contexts.
 */
public class ResolverServiceClientSecretValueResolver extends AbstractClientSecretValueResolver {
    
    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(ResolverServiceClientSecretValueResolver.class);
   
    /** The attribute resolver service used for the client secret value resolution. */
    @NonnullAfterInit private ReloadableService<AttributeResolver> service;
    
    /** The list of attribute IDs that may contain the resolved client secret values. */
    @Nonnull private List<String> attributeIds;
    
    /**
     * Constructor.
     */
    public ResolverServiceClientSecretValueResolver() {
        attributeIds = Collections.emptyList();
    }
    
    /**
     * Set the attribute resolver service used for the client secret value resolution.
     * 
     * @param resolver The attribute resolver service used for the client secret value resolution.
     */
    public void setAttributeResolver(@Nonnull final ReloadableService<AttributeResolver> resolver) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        service = Constraint.isNotNull(resolver, "Attribute resolver service can not be null");
    }
    
    /**
     * Get the attribute resolver service used for the client secret value resolution.
     * 
     * @return The attribute resolver service used for the client secret value resolution.
     */
    public @NonnullAfterInit ReloadableService<AttributeResolver> getAttributeResolver() {
        return service;
    }
    
    /**
     * Set the list of attribute IDs that may contain the resolved client secret values.
     * 
     * @param ids The list of attribute IDs that may contain the resolved client secret values.
     */
    public void setAttributeIds(@Nonnull final List<String> ids) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        attributeIds = (List<String>) Constraint.isNotEmpty(ids, "The list of attribute ids cannot be empty");
    }
    
    /**
     * Get the list of attribute IDs that may contain the resolved client secret values.
     * 
     * @return The list of attribute IDs that may contain the resolved client secret values.
     */
    public @Nonnull List<String> getAttributeIds() {
        return attributeIds;
    }

    /** {@inheritDoc} */
    @Override
    public @Nonnull Iterable<String> resolve(@Nonnull final CriteriaSet criteria) throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        final ClientSecretReferenceCriterion referenceCriterion = criteria.get(ClientSecretReferenceCriterion.class);
        Constraint.isNotNull(referenceCriterion,
                "The client secret reference criterion must be included in the criteria.");

        final ProfileRequestContext profileRequestContext = new ProfileRequestContext();
        final AttributeResolutionContext resolutionContext =
                profileRequestContext.getSubcontext(AttributeResolutionContext.class, true);
        resolutionContext.setPrincipal(referenceCriterion.getSecretReference());
        resolutionContext.setResolutionLabel(getClass().getSimpleName());
        if (criteria.contains(EntityIdCriterion.class)) {
            resolutionContext.setAttributeRecipientID(criteria.get(EntityIdCriterion.class).getEntityId());
        }
        resolutionContext.resolveAttributes(service);
        final Map<String, IdPAttribute> resolvedAttributes = resolutionContext.getResolvedIdPAttributes();
        final LazySet<String> result = new LazySet<>();
        for (final String attributeId : attributeIds) {
            if (resolvedAttributes.containsKey(attributeId)) {
                for (final IdPAttributeValue value : resolvedAttributes.get(attributeId).getValues()) {
                    if (value instanceof StringAttributeValue) {
                        log.debug("Found a value for reference '{}' via attribute ID {}",
                                referenceCriterion.getSecretReference(), attributeId);
                        result.add(((StringAttributeValue) value).getValue());
                    }
                }
            }
        }
        return result;
    }

    /** {@inheritDoc} */
    @Override
    public @Nullable String resolveSingle(@Nonnull final CriteriaSet criteria) throws ResolverException {
        final Iterator<String> iterator = resolve(criteria).iterator();
        return iterator.hasNext() ? iterator.next() : null;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doDestroy() {
        service = null;
        attributeIds = null;
        super.doDestroy();
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (getAttributeResolver() == null) {
            throw new ComponentInitializationException("Attribute resolver service can not be null");
        }
    }
}
