/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.io.IOException;
import java.util.Collections;
import java.util.Properties;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;

import net.shibboleth.oidc.metadata.criterion.ClientSecretReferenceCriterion;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.collection.LazySet;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * A client secret value resolver that fetches the values from the given Properties file.
 */
public class PropertiesClientSecretValueResolver extends AbstractClientSecretValueResolver {

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(PropertiesClientSecretValueResolver.class);

    /** The properties resource containing client secret values. */
    @NonnullAfterInit private Resource propertiesResource;
    
    /**
     * Set the properties resource containing client secret values.
     * 
     * @param resource The properties resource containing client secret values.
     */
    public void setResource(@Nonnull final Resource resource) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        propertiesResource = Constraint.isNotNull(resource, "Properties resource cannot be null");
        final Properties properties = new Properties();
        try {
            properties.load(propertiesResource.getInputStream());
        } catch (final IOException e) {
            log.error("Could not read the properties from the given resource", e);
            throw new ConstraintViolationException("The properties resource must be readable");
        }
    }
    
    /**
     * Get the properties resource containing client secret values.
     * 
     * @return The properties resource containing client secret values.
     */
    public @NonnullAfterInit Resource getResource() {
        return propertiesResource;
    }
    
    /** {@inheritDoc} */
    @Override
    public @Nonnull Iterable<String> resolve(@Nonnull final CriteriaSet criteria) throws ResolverException {
        final String value = resolveSingle(criteria);
        if (value != null) {
            final LazySet<String> result = new LazySet<>();
            result.add(value);
            return result;
        }
        return Collections.emptySet();
    }

    /** {@inheritDoc} */
    @Override
    public @Nullable String resolveSingle(@Nonnull final CriteriaSet criteria) throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        final ClientSecretReferenceCriterion criterion = criteria.get(ClientSecretReferenceCriterion.class);
        final Properties properties = new Properties();
        try {
            properties.load(propertiesResource.getInputStream());
        } catch (final IOException e) {
            log.error("Could not read the properties from the resource", e);
            throw new ResolverException(e);
        }
        return properties.getProperty(criterion.getSecretReference());
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (getResource() == null) {
            throw new ComponentInitializationException("Properties resource cannot be null");
        }
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doDestroy() {
        propertiesResource = null;
          
        super.doDestroy();
    }

}
