/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.oidc.metadata.impl;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import net.shibboleth.oidc.metadata.DynamicBackingStore;
import net.shibboleth.oidc.metadata.MetadataManagementData;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Default implementation of a {@link DynamicBackingStore}.
 *
 * @param <I> the metadata identifier type.
 * @param <T> the metadata type.
 */
@ThreadSafe
public class DefaultDynamicBackingStore<I,T> extends AbstractBackingStore<I,T> implements DynamicBackingStore<I, T>{
    
    
    /** Map holding management data for each entityID. */
    private final Map<I, MetadataManagementData<I>> mgmtDataMap;    
    
    /**
     * Constructor.
     */
    public DefaultDynamicBackingStore() {
        super();        
        mgmtDataMap = new ConcurrentHashMap<>();
    }
    
    @Override
    public MetadataManagementData<I> computeManagementDataIfAbsent(@Nonnull final I identifier,
            @Nonnull final Function<I, MetadataManagementData<I>> mappingFunction) {
        Constraint.isNotNull(identifier, "identifier may not be null");
        
        return mgmtDataMap.computeIfAbsent(identifier, mappingFunction);  
    }
    
    @Override
    public MetadataManagementData<I> getManagementData(@Nonnull final I identifier) {
        Constraint.isNotNull(identifier, "identifier may not be null");
        return mgmtDataMap.get(identifier);
    }

    
    @Override
    //TODO is concurrent hashmap threadsafe for remove and get - do we need the synchronized
    public synchronized void removeManagementData(@Nonnull final I identifier) {
        Constraint.isNotNull(identifier, "Identifier may not be null");
        mgmtDataMap.remove(identifier);
        
    }
   
    
    @Override
    @Nonnull @NonnullElements @Unmodifiable @NotLive
    public synchronized Set<I> getManagementDataIdentifiers() {
        return Set.copyOf(mgmtDataMap.keySet());        
    }

 
}
