/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Arrays;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.id.Identifier;

import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * Based on {@link org.opensaml.saml.metadata.resolver.impl.AbstractReloadingMetadataResolver}.
 * 
 * @param <Key> The identifier type in the backing store
 * @param <Value> The entity type in the backing store
 */
public abstract class AbstractReloadingOIDCEntityResolver<Key extends Identifier, Value> 
    extends AbstractOIDCEntityResolver<Key, Value> {

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(AbstractReloadingOIDCEntityResolver.class);
    
    /** Timer used to schedule background metadata update tasks. */
    private final Timer taskTimer;
    
    /** Whether we created our own task timer during object construction. */
    private final boolean createdOwnTaskTimer;
        
    /** Current task to refresh metadata. */
    private RefreshMetadataTask refreshMetadataTask;
    
    /**
     * Refresh interval used when metadata does not contain any validUntil or cacheDuration information. Default value:
     * 4 hours
     */
    @Nonnull @Positive private Duration maxRefreshDelay = Duration.ofHours(4);

    /** Floor, in milliseconds, for the refresh interval. Default value: 5 minutes */
    @Nonnull @Positive private Duration minRefreshDelay = Duration.ofMinutes(5);

    /** Last time the metadata was updated. */
    @Nullable private Instant lastUpdate;

    /** Last time a refresh cycle occurred. */
    @Nullable private Instant lastRefresh;

    /** Next time a refresh cycle will occur. */
    @Nullable private Instant nextRefresh;

    /** Constructor. */
    protected AbstractReloadingOIDCEntityResolver() {
        this(null);
    }

    /**
     * Constructor.
     * 
     * @param backgroundTaskTimer time used to schedule background refresh tasks
     */
    protected AbstractReloadingOIDCEntityResolver(@Nullable final Timer backgroundTaskTimer) {
        if (backgroundTaskTimer == null) {
            taskTimer = new Timer(true);
            createdOwnTaskTimer = true;
        } else {
            taskTimer = backgroundTaskTimer;
            createdOwnTaskTimer = false;
        }
    }

    protected void initOIDCResolver() throws ComponentInitializationException {
        super.initOIDCResolver();
        try {
            refresh();
        } catch (final ResolverException e) {
            log.error("Could not refresh the entity information", e);
            throw new ComponentInitializationException("Could not refresh the entity information", e);
        }
    }

    /**
     * Get last update of resolver.
     * 
     * @return last update
     */
    @Nullable public Instant getLastUpdate() {
        return lastUpdate;
    }

    /**
     * Get last refresh of resolver.
     * 
     * @return last refresh
     */
    @Nullable public Instant getLastRefresh() {
        return lastRefresh;
    }
    
    /**
     * Sets the minimum amount of time between refreshes.
     * 
     * @param delay minimum amount of time between refreshes
     */
    public void setMinRefreshDelay(@Positive @Nonnull final Duration delay) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        Constraint.isFalse(delay == null || delay.isNegative(), "Minimum refresh delay must be greater than 0");
        minRefreshDelay = delay;
    }
    
    /**
     * Sets the maximum amount of time between refresh intervals.
     * 
     * @param delay maximum amount of time, in milliseconds, between refresh intervals
     */
    public void setMaxRefreshDelay(@Positive @Nonnull final Duration delay) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        Constraint.isFalse(delay == null || delay.isNegative(), "Maximum refresh delay must be greater than 0");
        maxRefreshDelay = delay;
    }
    
    /**
     * Refreshes the metadata from its source.
     * 
     * @throws ResolverException thrown is there is a problem retrieving and processing the metadata
     */
    public synchronized void refresh() throws ResolverException {
        final Instant now = Instant.now();
        final String mdId = getMetadataIdentifier();

        Duration refreshDelay = null;
        
        log.debug("Beginning refresh of metadata from '{}'", mdId);
        try {
            final byte[] mdBytes = fetchMetadata();
            if (mdBytes == null) {
                log.debug("Metadata from '{}' has not changed since last refresh", mdId);
            } else {
                log.debug("Processing new metadata from '{}'", mdId);
                final List<Value> resolvedInformation = parse(mdBytes);
                final JsonBackingStore newBackingStore = new JsonBackingStore();
                for (final Value information : resolvedInformation) {
                    final Key id = getKey(information);
                    log.info("Parsed entity information for {}", id);
                    newBackingStore.getIndexedInformation().put(id, Arrays.asList(information));
                    newBackingStore.getOrderedInformation().add(information);                    
                }
                setBackingStore(newBackingStore);
                lastUpdate = now;
            }
        } catch (final Throwable t) {
            log.error("Error occurred while attempting to refresh metadata from '" + mdId + "'", t);
            refreshDelay = minRefreshDelay;
            if (t instanceof Exception) {
                throw new ResolverException((Exception) t);
            } else {
                throw new ResolverException(String.format("Saw an error of type '%s' with message '%s'", 
                        t.getClass().getName(), t.getMessage()));
            }
        } finally {
            scheduleNextRefresh(refreshDelay);
            lastRefresh = now;
        }
    }
    
    /**
     * Schedules the next refresh. If the given delay is 0 or null, then {@link #maxRefreshDelay} is used.
     * @param delay The delay before the next refresh.
     */
    protected void scheduleNextRefresh(@Nullable final Duration delay) {
        refreshMetadataTask = new RefreshMetadataTask();
        Duration refreshDelay = delay;
        if (delay == null || delay.isZero()) {
            refreshDelay = maxRefreshDelay;
        }
        nextRefresh = Instant.now().plus(refreshDelay);
        final long nextRefreshDelay = nextRefresh.toEpochMilli() - System.currentTimeMillis();

        taskTimer.schedule(refreshMetadataTask, nextRefreshDelay);
        log.info("Next refresh cycle for metadata provider '{}' will occur on '{}' ('{}' local time)",
                new Object[] {getMetadataIdentifier(), nextRefresh, nextRefresh.atZone(ZoneId.systemDefault()),});
    }
    
    /**
     * Parses an entity from the byte array.
     * 
     * @param bytes The encoded entity
     * @return The parsed entity
     * 
     * @throws ParseException if parse fails
     */
    protected abstract List<Value> parse(final byte[] bytes) throws ParseException;
    
    /**
     * Gets the identifier for the given entity.
     * 
     * @param value The entity whose identifier will be returned.
     * @return The identifier for the given entity.
     */
    protected abstract Key getKey(final Value value);
    
    /**
     * Gets an identifier which may be used to distinguish this metadata in logging statements.
     * 
     * @return identifier which may be used to distinguish this metadata in logging statements
     */
    protected abstract String getMetadataIdentifier();

    /**
     * Fetches metadata from a source.
     * 
     * @return the fetched metadata, or null if the metadata is known not to have changed since the last retrieval
     * 
     * @throws ResolverException thrown if there is a problem fetching the metadata
     */
    protected abstract byte[] fetchMetadata() throws ResolverException;
    
    /** Background task that refreshes metadata. */
    private class RefreshMetadataTask extends TimerTask {

        /** {@inheritDoc} */
        @Override
        public void run() {
            try {
                if (!isInitialized()) {
                    // just in case the metadata provider was destroyed before this task runs
                    return;
                }
                
                refresh();
            } catch (final ResolverException e) {
                // nothing to do, error message already logged by refreshMetadata()
                return;
            }
        }
    }
}