/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.criterion.SatisfyAnyCriterion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Iterables;

import net.shibboleth.oidc.metadata.AbstractEvaluableMetadataCriterion;
import net.shibboleth.oidc.metadata.OIDCMetadataResolver;
import net.shibboleth.oidc.metadata.cache.MetadataCache;
import net.shibboleth.oidc.metadata.cache.MetadataCacheException;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.component.DestructableComponent;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.CriterionPredicateRegistry;
import net.shibboleth.utilities.java.support.resolver.ResolverException;
import net.shibboleth.utilities.java.support.resolver.ResolverSupport;

/**
 * An abstract metadata resolver which supports any type of metadata. 
 *
 * @param <MetadataIdentifier> the type used to identify the metadata
 * @param <MetadataType> the metadata type
 */
// TODO Would need to remove OIDC from the name as is generic - but confusing clash with opensaml.
public abstract class AbstractOIDCMetadataResolver<MetadataIdentifier, MetadataType> 
        extends AbstractIdentifiableInitializableComponent implements OIDCMetadataResolver<MetadataType> {
    
    
    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(AbstractOIDCMetadataResolver.class);
    
    /** Logging prefix. */
    private String logPrefix;
    
    /** The metadata cache.*/
    @Nonnull private final MetadataCache<MetadataType> cache;   
    
    
    /** Flag which determines whether predicates used in filtering are connected by 
     * a logical 'OR' (true) or by logical 'AND' (false). Defaults to false. */
    private boolean satisfyAnyPredicates;
    
    /** Registry used in resolving predicates from criteria. */
    @NonnullAfterInit private CriterionPredicateRegistry<MetadataType> criterionPredicateRegistry;
    
    /**
     * Whether problems during initialization should cause the provider to fail or go on without metadata. The
     * assumption being that in most cases a provider will recover at some point in the future. Default: true.
     */
    private boolean failFastInitialization;
    
    /**
     * Constructor.
     *
     * @param metadataCache the metadata cache to use.
     */
    protected AbstractOIDCMetadataResolver(@Nonnull final MetadataCache<MetadataType> metadataCache) {
        failFastInitialization = true;
        cache = Constraint.isNotNull(metadataCache, "Metadata cache can not be null");
    }
    
    
    /**
     * Return a prefix for logging messages for this component.
     * 
     * @return a string for insertion at the beginning of any log messages
     */
    @Nonnull @NotEmpty protected String getLogPrefix() {
        if (logPrefix == null) {
            logPrefix = String.format("Metadata Resolver %s %s:", getClass().getSimpleName(), getId());
        }
        return logPrefix;
    }
    
    /** {@inheritDoc} */
    @Override @Nullable public MetadataType resolveSingle(final CriteriaSet criteria) throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        final Iterable<MetadataType> iterable = resolve(criteria);
        if (iterable != null) {
            final Iterator<MetadataType> iterator = iterable.iterator();
            if (iterator != null && iterator.hasNext()) {
                return iterator.next();
            }
        }
        return null;
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull public Iterable<MetadataType> resolve(@Nonnull final CriteriaSet criteria) throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
               
        try {            
            final List<MetadataType> metadata = getCache().get(criteria);            
            return predicateFilterCandidates(metadata, criteria, false);
            
        } catch (final MetadataCacheException e) {
            throw new ResolverException(e);
        } 
    }     
    
    /**
     * Filter the supplied candidates by resolving predicates from the supplied criteria and applying
     * the predicates to return a filtered {@link Iterable}.
     * 
     * @param candidates the candidates to evaluate
     * @param criteria the criteria set to evaluate
     * @param onEmptyPredicatesReturnEmpty if true and no predicates are supplied, then return an empty iterable;
     *          otherwise return the original input candidates
     * 
     * @return an iterable of the candidates filtered by the resolved predicates
     * 
     * @throws ResolverException if there is a fatal error during resolution
     */
    protected Iterable<MetadataType> predicateFilterCandidates(@Nonnull final Iterable<MetadataType> candidates,
            @Nonnull final CriteriaSet criteria, final boolean onEmptyPredicatesReturnEmpty)
                    throws ResolverException {
        
        if (!candidates.iterator().hasNext()) {
            log.debug("{} Candidates iteration was empty, nothing to filter via predicates", getLogPrefix());
            return Collections.emptySet();
        }
        
        log.debug("{} Attempting to filter candidate metadata via resolved Predicates", getLogPrefix());
        
        // The criterion has to be a subtype of AbstractEvaluableMetadataCriterion to avoid errors.
        @SuppressWarnings("unchecked") final Set<Predicate<MetadataType>> predicates = 
                ResolverSupport.getPredicates(criteria, AbstractEvaluableMetadataCriterion.class, 
                        getCriterionPredicateRegistry());
        
        log.trace("{} Resolved {} Predicates: {}", getLogPrefix(), predicates.size(), predicates);
        
        final boolean satisfyAny;
        final SatisfyAnyCriterion satisfyAnyCriterion = criteria.get(SatisfyAnyCriterion.class);
        if (satisfyAnyCriterion  != null) {
            log.trace("{} CriteriaSet contained SatisfyAnyCriterion", getLogPrefix());
            satisfyAny = satisfyAnyCriterion.isSatisfyAny();
        } else {
            log.trace("{} CriteriaSet did NOT contain SatisfyAnyCriterion", getLogPrefix());
            satisfyAny = isSatisfyAnyPredicates();
        }
        
        log.trace("{} Effective satisyAny value: {}", getLogPrefix(), satisfyAny);
        
        final Iterable<MetadataType> result = 
                ResolverSupport.getFilteredIterable(candidates, predicates, satisfyAny, onEmptyPredicatesReturnEmpty);
        if (log.isDebugEnabled()) {
            log.debug("{} After predicate filtering {} Metadata entities remain", 
                    getLogPrefix(), Iterables.size(result));
        }
        return result;
    }
   
    
    /**
     * Get the flag indicating whether resolved credentials may satisfy any predicates 
     * (i.e. connected by logical 'OR') or all predicates (connected by logical 'AND').
     * 
     * <p>Defaults to false.</p>
     * 
     * @return true if must satisfy all, false otherwise
     */
    public boolean isSatisfyAnyPredicates() {
        return satisfyAnyPredicates;
    }
    
    /**
     * Set the flag indicating whether resolved credentials may satisfy any predicates 
     * (i.e. connected by logical 'OR') or all predicates (connected by logical 'AND').
     * 
     * <p>Defaults to false.</p>
     * 
     * @param flag true if must satisfy all, false otherwise
     */
    public void setSatisfyAnyPredicates(final boolean flag) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        satisfyAnyPredicates = flag;
    }
    
    /**
     * Get the registry used in resolving predicates from criteria.
     * 
     * @return the effective registry instance used
     */
    @NonnullAfterInit public CriterionPredicateRegistry<MetadataType> getCriterionPredicateRegistry() {
        return criterionPredicateRegistry;
    }

    /**
     * Set the registry used in resolving predicates from criteria.
     * 
     * @param registry the registry instance to use
     */
    public void setCriterionPredicateRegistry(@Nullable final CriterionPredicateRegistry<MetadataType> registry) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        criterionPredicateRegistry = registry;
    }

    
    /**
     * Get the metadata cache currently in use by the metadata resolver.
     * 
     * @return the current effective entity backing store
     */
    @Nonnull protected MetadataCache<MetadataType> getCache() {
        return cache;
    }    
    
    
    /**
     * Gets whether problems during initialization should cause the provider to fail or go on without metadata. The
     * assumption being that in most cases a provider will recover at some point in the future.
     * 
     * @return whether problems during initialization should cause the provider to fail
     */
    public boolean isFailFastInitialization() {
        return failFastInitialization;
    }

    /**
     * Sets whether problems during initialization should cause the provider to fail or go on without metadata. The
     * assumption being that in most cases a provider will recover at some point in the future.
     * 
     * @param failFast whether problems during initialization should cause the provider to fail
     */
    public void setFailFastInitialization(final boolean failFast) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        failFastInitialization = failFast;
    }
    
    
    /** {@inheritDoc} */
    @Override protected final void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        try {
            initMetadataResolver();
        } catch (final ComponentInitializationException e) {
            if (isFailFastInitialization()) {
                log.error("{} Metadata provider failed to properly initialize, fail-fast=true, halting", 
                        getLogPrefix());
                throw e;
            }
            log.error("{} Metadata provider failed to properly initialize, fail-fast=false, "
                    + "continuing on in a degraded state", getLogPrefix(), e);
        }
    }  
    
    @Override protected void doDestroy() {
        if (cache instanceof DestructableComponent) {
            ((DestructableComponent)cache).destroy();
        }
    }
    
    /** Initialise this metadata provider. Subclasses will need to override this method.
     * @throws ComponentInitializationException if initialization fails.
     */
    protected abstract void initMetadataResolver() throws ComponentInitializationException;
    

    
}
